unit uLoaderArt;

interface

uses Classes, SysUtils, uUtilities, Dialogs, uLoaderHues, uTexCache, dglOpenGL,
        uCursors, uLoaderTiledata, uLog, uLoaderVerdata, uRenderer, SyncObjs,
        uMulCrypt, uLoaderProgress;

type    TArt = Class;

        TTempTexture = Class
                public
                        Data : PByteArray;
                        Id : LongWord;
                        Hue : Word;
                        Breite, Hoehe : Word;
                        HasAlpha : Boolean;
                        constructor Create( Id : LongWord; Hue : Word );
                        destructor Free;
        end;

        TArtThread = Class( TThread )
                private
                        Art : TArt;
                protected
                        procedure Execute; override;
                        procedure Init( Art : TArt );
                public
                        constructor Create( CreateSuspended : Boolean );
                        destructor Free;
        end;

        TArt = Class
                private
                        idxBaseStream, mulBaseStream, idxShardStream, mulShardStream : TFileStream;
                        Cache : TTexCache;
                        ArtThread : TArtThread;
                        LoadList, LoadedList : TList;
                        CriticalSection, CriticalMulSection : TCriticalSection;
                        function LoadArt( Id, Hue : LongWord; Immediately : Boolean = False ) : TTexObject;
                public
                        constructor Create( baseDir, shardDir : String );
                        destructor Free;
                        procedure Poll;
                        function GetStaticArt( Id, Hue : LongWord; Immediately : Boolean = False ) : TTexObject;
                        function LoadCursor( Id, Hue : LongWord ) : TCustomCursor;
                        function GetMapArt( Id, Hue : LongWord; Immediately : Boolean = False ) : TTexObject;
                        function LoadData( Texture : TTexObject ) : TTempTexture;
                        procedure IncMapArt( Id, Hue : LongWord );
                        procedure DecMapArt( Id, Hue : LongWord );
                        procedure IncStaticArt( Id, Hue : LongWord );
                        procedure DecStaticArt( Id, Hue : LongWord );
                        function GetStaticArtWidth( Id : LongWord ) : Word;
                        function GetStaticArtHeight( Id : LongWord ) : Word;
        end;

implementation

uses uPalanthir;

constructor TTempTexture.Create( Id : LongWord; Hue : Word );
begin
        Self.Id := Id;
        Self.Hue := Hue;
        HasAlpha := False;
        Data := nil;
end;

destructor TTempTexture.Free;
begin
        if Data <> nil then
                FreeMem( Data );
end;

procedure TArtThread.Execute;
var     Count : LongWord;
        Texture : TTexObject;
        TempTexture : TTempTexture;
begin
        while not Terminated do begin
                Art.CriticalSection.Enter;
                Count := Art.LoadList.Count;
                Art.CriticalSection.Leave;

                while (Count > 0) and (not Terminated) do begin
                        Art.CriticalSection.Enter;
                        Texture := TTexObject( Art.LoadList.Items[ 0 ] );
                        Art.LoadList.Delete( 0 );
                        Art.CriticalSection.Leave;

                        if Texture <> nil then begin
                                TempTexture := Art.LoadData( Texture );
                        end
                        else begin
                                TempTexture := nil;
                        end;

                        Art.CriticalSection.Enter;
                        if TempTexture <> nil then
                                Art.LoadedList.Add( TempTexture );

                        Count := Art.LoadList.Count;
                        Art.CriticalSection.Leave;
                end;

                Sleep( 10 );
        end;
end;

procedure TArtThread.Init( Art : TArt );
begin
        Self.Art := Art;
end;

constructor TArtThread.Create( CreateSuspended : Boolean );
begin
        inherited Create( CreateSuspended );
end;

destructor TArtThread.Free;
begin
end;

constructor TArt.Create( baseDir, shardDir : String );
begin
        idxBaseStream := TFileStream.Create( baseDir + 'Artidx.mul', fmOpenRead + fmShareDenyNone );
        mulBaseStream := TFileStream.Create( baseDir + 'Art.mul', fmOpenRead + fmShareDenyNone );
        if FileExists( shardDir + 'Artidx.mul' ) then
                idxShardStream := TFileStream.Create( shardDir + 'Artidx.mul', fmOpenRead + fmShareDenyNone )
        else
                idxShardStream := nil;
        if FileExists( shardDir + 'Art.mul' ) then
                mulShardStream := TFileStream.Create( shardDir + 'Art.mul', fmOpenRead + fmShareDenyNone )
        else
                mulShardStream := nil;
        Cache := TTexCache.Create;
        Cache.AutoDelete := False;

        LoadList := TList.Create;
        LoadedList := TList.Create;

        CriticalSection := TCriticalSection.Create;
        CriticalMulSection := TCriticalSection.Create;

        ArtThread := TArtThread.Create( True );
        ArtThread.Init( Self );
        ArtThread.Resume;
end;

destructor TArt.Free;
var     I : Integer;
begin
        ArtThread.Terminate;
        Sleep( 100 );
        ArtThread.Free;

        idxBaseStream.Free;
        mulBaseStream.Free;
        if idxShardStream <> nil then
                idxShardStream.Free;
        if mulShardStream <> nil then
                mulShardStream.Free;
        Cache.Free;

        LoadList.Free;
        for I := 0 to LoadedList.Count-1 do
                TTempTexture( LoadedList.Items[ I ] ).Free;
        LoadedList.Free;

        CriticalSection.Free;
        CriticalMulSection.Free;
end;

procedure TArt.Poll;
var     TempTexture : TTempTexture;
        Count : Integer;
        Texture : TTexObject;
begin
        CriticalSection.Enter;
        Count := LoadedList.Count;
        CriticalSection.Leave;

        while (Count > 0) do begin
                CriticalSection.Enter;
                TempTexture := TTempTexture( LoadedList.Items[ 0 ] );
                LoadedList.Delete( 0 );
                CriticalSection.Leave;

                if TempTexture.Id < $4000 then
                        Texture := GetMapArt( TempTexture.Id, TempTexture.Hue )
                else
                        Texture := GetStaticArt( TempTexture.Id - $4000, TempTexture.Hue );

                Renderer.InitTexture32( Texture, TempTexture.Breite, TempTexture.Hoehe, TempTexture.Data );
                Texture.HasAlpha := TempTexture.HasAlpha;
                TempTexture.Free;

                Texture.Loading := False;

                CriticalSection.Enter;
                Count := LoadedList.Count;
                CriticalSection.Leave;
        end;

        Cache.GarbageCollection;
end;

function TArt.LoadData( Texture : TTexObject ) : TTempTexture;
var     IndexRecord : TIndexRecord;
        Y, X, I, Start, Count : Integer;
        Color, Red, Green, Blue : Word;
        PY, PX : Integer;
        PartialHue : Boolean;
        LookupList : PWordArray;
        Hoehe, Breite : Word;
        Art : PByteArray;
        XOffset, RunLength : Word;
        Unknown : Integer;
        mulStream : TStream;
        Position : Integer;
        RBreite, RHoehe : Word;
        VerdataEntry : TVerdataEntry;
        Gefunden : Boolean;
        Id : LongWord;
        Hue : Word;
        ProgressImage : TProgressImage;
begin
        Id := Texture.ID;
        Hue := Texture.Hue;

        if Id >= $4000 then begin
                ProgressImage := Palanthir.Data.Progress.GetGump( Id-$4000, Hue );
                if ProgressImage <> nil then begin
                        Result := TTempTexture.Create( Id, Hue );
                        Count := GetNextBit( ProgressImage.Width )*GetNextBit( ProgressImage.Height )*4;
                        GetMem( Result.Data, Count );
                        Move( ProgressImage.Data^[0], Result.Data^[0], Count );
                        Result.Breite := ProgressImage.Width;
                        Result.Hoehe := ProgressImage.Height;
                        Result.HasAlpha := True;
                        Exit;
                end;
        end;

        CriticalMulSection.Enter;

        VerdataEntry := Palanthir.Data.Verdata.GetArt( Id );
        if VerdataEntry <> nil then begin
                IndexRecord := VerdataEntry.IndexRecord;
                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        CriticalMulSection.Leave;
                        Result := nil;
                        exit;
                end;
                mulStream := VerdataEntry.mulStream;
        end
        else begin
                Gefunden := False;
                if (idxShardStream <> nil) and (mulShardStream <> nil) and (Id*12 < idxShardStream.Size) then begin
                        idxShardStream.Seek( Id*12, soFromBeginning );
                        idxShardStream.Read( IndexRecord, 12 );

                        if Palanthir.MulEncryption then begin
                                if Palanthir.Data.MulCrypt.DecryptIndexRecordArt( Id, IndexRecord ) then begin
                                        if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                                Gefunden := True;
                                                mulStream := mulShardStream;
                                        end;
                                end
                                else begin
                                        Log.Write( 'Not able to decrypt Art.' );
                                end;
                        end
                        else begin
                                if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                        Gefunden := True;
                                        mulStream := mulShardStream;
                                end;
                        end;
                end;

                if (not Gefunden) and (idxBaseStream <> nil) and (mulBaseStream <> nil) then begin
                        if Id*12 >= idxBaseStream.Size then begin
                                CriticalMulSection.Leave;
                                Result := nil;
                                exit;
                        end;

                        idxBaseStream.Seek( Id*12, soFromBeginning );
                        idxBaseStream.Read( IndexRecord, 12 );

                        if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                                CriticalMulSection.Leave;
                                Result := nil;
                                exit;
                        end;

                        mulStream := mulBaseStream;
                end;
        end;

        mulStream.Seek( IndexRecord.Offset, soFromBeginning );

        if Id < $4000 then begin
                Breite := 44;
                Hoehe := 44;

                RBreite := GetNextBit( Breite );
                RHoehe := GetNextBit( Hoehe );

                GetMem( Art, RBreite*RHoehe*4 );
                for I := 0 to RBreite*RHoehe*4-1 do
                        Art^[I] := 0;

                for Y := 0 to 21 do begin
                        Position := ( Y*RBreite + 21-Y )*4;
                        for X := 0 to (Y+1)*2-1 do begin
                                mulStream.Read( Color, 2 );
                                if Hue > 0 then begin
                                        Red := Color15toRed( Color ) div 8;
                                        Art^[ Position ] := Color15toRed( Hues.GetColor( Hue, Red ) );
                                        Art^[ Position+1 ] := Color15toGreen( Hues.GetColor( Hue, Red ) );
                                        Art^[ Position+2 ] := Color15toBlue( Hues.GetColor( Hue, Red ) );
                                end
                                else begin
                                        Art^[ Position ] := Color15toRed( Color );
                                        Art^[ Position+1 ] := Color15toGreen( Color );
                                        Art^[ Position+2 ] := Color15toBlue( Color );
                                end;
                                Art^[ Position+3 ] := 255;
                                Inc( Position, 4 );
                        end;
                end;
                for Y := 0 to 21 do begin
                        Position := ( (Y+22)*RBreite + Y )*4;
                        for X := 0 to 43-Y*2 do begin
                                mulStream.Read( Color, 2 );
                                if Hue > 0 then begin
                                        Red := Color15toRed( Color ) div 8;
                                        Art^[ Position ] := Color15toRed( Hues.GetColor( Hue, Red ) );
                                        Art^[ Position+1 ] := Color15toGreen( Hues.GetColor( Hue, Red ) );
                                        Art^[ Position+2 ] := Color15toBlue( Hues.GetColor( Hue, Red ) );
                                end
                                else begin
                                        Art^[ Position ] := Color15toRed( Color );
                                        Art^[ Position+1 ] := Color15toGreen( Color );
                                        Art^[ Position+2 ] := Color15toBlue( Color );
                                end;
                                Art^[ Position+3 ] := 255;
                                Inc( Position, 4 );
                        end;
                end;
        end
        else begin
                PartialHue := Palanthir.Data.Tiledata.GetStaticFlag( Id-$4000, dPARTIALHUE );
                mulStream.Read( Unknown, 4 );
                mulStream.Read( Breite, 2 );
                mulStream.Read( Hoehe, 2 );

                RBreite := GetNextBit( Breite );
                RHoehe := GetNextBit( Hoehe );

                GetMem( Art, RBreite*RHoehe*4 );
                for I := 0 to RBreite*RHoehe*4-1 do
                        Art^[I] := 0;

                GetMem( LookupList, Hoehe*2 );
                mulStream.Read( LookupList^, Hoehe*2 );
                Start := mulStream.Position;

                for Y := 0 to Hoehe-1 do begin
                        mulStream.Seek( Start + LookupList^[Y]*2, soFromBeginning );
                        X := 0;

                        mulStream.Read( XOffset, 2 );
                        mulStream.Read( RunLength, 2 );

                        while ( RunLength <> 0 ) do begin
                                X := X + XOffset;

                                Position := ( Y*RBreite + X )*4;
                                for I := 0 to RunLength-1 do begin
                                        mulStream.Read( Color, 2 );

                                        if Position+3 < RBreite*RHoehe*4 then begin
                                                if Hue > 0 then begin
                                                        Red := Color15toRed( Color );
                                                        Green := Color15toGreen( Color );
                                                        Blue := Color15toBlue( Color );

                                                        if ( PartialHue and (Red = Blue) and (Red = Green) ) or ( not PartialHue ) then begin
                                                                Red := Red div 8;
                                                                Art^[ Position ] := Color15toRed( Hues.GetColor( Hue, Red ) );
                                                                Art^[ Position+1 ] := Color15toGreen( Hues.GetColor( Hue, Red ) );
                                                                Art^[ Position+2 ] := Color15toBlue( Hues.GetColor( Hue, Red ) );
                                                        end
                                                        else begin
                                                                Art^[ Position ] := Color15toRed( Color );
                                                                Art^[ Position+1 ] := Color15toGreen( Color );
                                                                Art^[ Position+2 ] := Color15toBlue( Color );
                                                        end;
                                                end
                                                else begin
                                                        Art^[ Position ] := Color15toRed( Color );
                                                        Art^[ Position+1 ] := Color15toGreen( Color );
                                                        Art^[ Position+2 ] := Color15toBlue( Color );
                                                end;
                                                Art^[ Position+3 ] := 255;
                                        end;
                                        Inc( Position, 4 );
                                end;

                                X := X + RunLength;
                                mulStream.Read( XOffset, 2 );
                                mulStream.Read( RunLength, 2 );
                        end;
                end;
        end;

        CriticalMulSection.Leave;

        Result := TTempTexture.Create( Texture.ID, Texture.Hue );
        Result.Data := Art;
        Result.Breite := Breite;
        Result.Hoehe := Hoehe;
end;

function TArt.LoadArt( Id, Hue : LongWord; Immediately : Boolean = False ) : TTexObject;
var     Typ : Byte;
        TempTexture : TTempTexture;
begin
        if Id < $4000 then begin
                Typ := Tex_ArtMap;
        end
        else begin
                Typ := Tex_ArtStatic;
        end;

        //maybe height or widht = 0 causes problems anywhere
        Result := Renderer.CreateTexture32( Typ, 1, 1, nil, Id, Hue );
        Result.Loading := True;

        Cache.AddObject( Result );

        if Immediately then begin
                TempTexture := LoadData( Result );
                Renderer.InitTexture32( Result, TempTexture.Breite, TempTexture.Hoehe, TempTexture.Data );
                TempTexture.Free;
        end
        else begin
                CriticalSection.Enter;
                LoadList.Add( Result );
                CriticalSection.Leave;
        end;
end;

function TArt.LoadCursor( Id, Hue : LongWord ) : TCustomCursor;
var     IndexRecord : TIndexRecord;
        Y, X, I, Start : Integer;
        Color, Red, Green, Blue : Word;
        PY, PX : Integer;
        PartialHue : Boolean;
        LookupList : PWordArray;
        Hoehe, Breite : Word;
        Art, CursorArt : PByteArray;
        XOffset, RunLength : Word;
        Unknown : Integer;
        Texture : TTexObject;
        mulStream : TFileStream;
        Gefunden : Boolean;
        RBreite, RHoehe : Word;
        TBreite, THoehe : Word;
        Position : Integer;
begin
        Gefunden := False;

        CriticalMulSection.Enter;
        if (idxShardStream <> nil) and (mulShardStream <> nil) and ((Id+$4000)*12 < idxShardStream.Size) then begin
                idxShardStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxShardStream.Read( IndexRecord, 12 );

                if Palanthir.MulEncryption then begin
                        if Palanthir.Data.MulCrypt.DecryptIndexRecordArt( Id+$4000, IndexRecord ) then begin
                                if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                        Gefunden := True;
                                        mulStream := mulShardStream;
                                end;
                        end
                        else begin
                                Log.Write( 'Not able to decrypt Art.' );
                        end;
                end
                else begin
                        if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                Gefunden := True;
                                mulStream := mulShardStream;
                        end;
                end;
        end;

        if not Gefunden then begin
                if (Id+$4000)*12 >= idxBaseStream.Size then begin
                        CriticalMulSection.Leave;
                        Result := nil;
                        exit;
                end;

                idxBaseStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxBaseStream.Read( IndexRecord, 12 );

                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        CriticalMulSection.Leave;
                        Result := nil;
                        exit;
                end;

                mulStream := mulBaseStream;
        end;
        mulStream.Seek( IndexRecord.Offset, soFromBeginning );

        PartialHue := Palanthir.Data.Tiledata.GetStaticFlag( Id, dPARTIALHUE );
        mulStream.Read( Unknown, 4 );
        mulStream.Read( Breite, 2 );
        mulStream.Read( Hoehe, 2 );

        if (Breite < 3) or (Hoehe < 3) then begin
                CriticalMulSection.Leave;
                Result := nil;
                exit;
        end;

        RBreite := GetNextBit( Breite );
        RHoehe := GetNextBit( Hoehe );

        GetMem( Art, RBreite*RHoehe*4 );
        for I := 0 to RBreite*RHoehe*4-1 do
                Art^[I] := 0;

        GetMem( LookupList, Hoehe*2 );
        mulStream.Read( LookupList^, Hoehe*2 );
        Start := mulStream.Position;

        for Y := 0 to Hoehe-1 do begin
                mulStream.Seek( Start + LookupList^[Y]*2, soFromBeginning );
                X := 0;

                mulStream.Read( XOffset, 2 );
                mulStream.Read( RunLength, 2 );

                while ( RunLength <> 0 ) do begin
                        X := X + XOffset;

                        Position := ( Y*RBreite + X )*4;
                        for I := 0 to RunLength-1 do begin
                                mulStream.Read( Color, 2 );
                                if Hue > 0 then begin
                                        Red := Color15toRed( Color );
                                        Green := Color15toGreen( Color );
                                        Blue := Color15toBlue( Color );

                                        if ( PartialHue and (Red = Blue) and (Red = Green) ) or ( not PartialHue ) then begin
                                                Red := Red div 8;
                                                Art^[ Position ] := Color15toRed( Hues.GetColor( Hue, Red ) );
                                                Art^[ Position+1 ] := Color15toGreen( Hues.GetColor( Hue, Red ) );
                                                Art^[ Position+2 ] := Color15toBlue( Hues.GetColor( Hue, Red ) );
                                        end
                                        else begin
                                                Art^[ Position ] := Color15toRed( Color );
                                                Art^[ Position+1 ] := Color15toGreen( Color );
                                                Art^[ Position+2 ] := Color15toBlue( Color );
                                        end;
                                end
                                else begin
                                        Art^[ Position ] := Color15toRed( Color );
                                        Art^[ Position+1 ] := Color15toGreen( Color );
                                        Art^[ Position+2 ] := Color15toBlue( Color );
                                end;
                                Art^[ Position+3 ] := 255;
                                Inc( Position, 4 );
                        end;

                        X := X + RunLength;
                        mulStream.Read( XOffset, 2 );
                        mulStream.Read( RunLength, 2 );
                end;
        end;

        CriticalMulSection.Leave;

        TBreite := GetNextBit( Breite-2 );
        THoehe := GetNextBit( Hoehe-2 );

        GetMem( CursorArt, TBreite*THoehe*4 );
        for I := 0 to TBreite*THoehe*4-1 do
                CursorArt^[ I ] := 0;
        
        for I := 0 to Hoehe-3 do
                Move( Art^[ (I+1)*RBreite*4+4 ], CursorArt^[ I*TBreite*4 ], TBreite*4 );

        Result := TCustomCursor.Create;
        for I := 1 to Breite-2 do
                if ( Art^[ I*4 ] <> 0 ) or ( Art^[ I*4+1 ] <> 0 ) or ( Art^[ I*4+2 ] <> 248 ) then begin
                        Result.SelX := I-1;
                        break;
                end;
        for I := 1 to Hoehe-2 do
                if ( Art^[ I*RBreite*4 ] <> 0 ) or ( Art^[ I*RBreite*4+1 ] <> 0 ) or ( Art^[ I*RBreite*4+2 ] <> 248 ) then begin
                        Result.SelY := I-1;
                        break;
                end;
        FreeMem( Art );

        Texture := Renderer.CreateTexture32( Tex_ArtStatic, Breite-2, Hoehe-2, CursorArt, Id, Hue );
        FreeMem( CursorArt );        
        Renderer.CurrentTexID := Texture.TexID;
        Result.Texture := Texture;
end;

function TArt.GetMapArt( Id, Hue : LongWord; Immediately : Boolean ) : TTexObject;
begin
        try
                Result := Cache.GetTexObject( Id, Hue );

                if Result = nil then
                        Result := LoadArt( Id, Hue, Immediately );
        except
                on E : Exception do begin
                        Log.Write( Format( 'uLoaderArt(Map): %s Id:%d Hue:%d' , [E.Message,Id,Hue] ) );
                        raise E;
                end;
        end;
end;

function TArt.GetStaticArt( Id, Hue : LongWord; Immediately : Boolean ) : TTexObject;
begin
        try
                Result := Cache.GetTexObject( Id+$4000, Hue );

                if Result = nil then
                        Result := LoadArt( Id+$4000, Hue, Immediately );
                
        except
                on E : Exception do begin
                        Log.Write( Format( 'uLoaderArt(Static): %s Id:%d Hue:%d' , [E.Message,Id,Hue] ) );
                        raise E;
                end;
        end;
end;

procedure TArt.IncMapArt( Id, Hue : LongWord );
begin
        Cache.IncCount( Id, Hue );
end;

procedure TArt.DecMapArt( Id, Hue : LongWord );
begin
        Cache.DecCount( Id, Hue );
end;

procedure TArt.IncStaticArt( Id, Hue : LongWord );
begin
        Cache.IncCount( Id+$4000, Hue );
end;

procedure TArt.DecStaticArt( Id, Hue : LongWord );
begin
        Cache.DecCount( Id+$4000, Hue );
end;

function TArt.GetStaticArtWidth( Id : LongWord ) : Word;
var     IndexRecord : TIndexRecord;
        Breite : Word;
        Unknown : Integer;
        mulStream : TFileStream;
        Gefunden : Boolean;
begin
        Gefunden := False;

        CriticalMulSection.Enter;
        if (idxShardStream <> nil) and (mulShardStream <> nil) and ((Id+$4000)*12 < idxShardStream.Size) then begin
                idxShardStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxShardStream.Read( IndexRecord, 12 );

                if Palanthir.MulEncryption then begin
                        if Palanthir.Data.MulCrypt.DecryptIndexRecordArt( Id+$4000, IndexRecord ) then begin
                                if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                        Gefunden := True;
                                        mulStream := mulShardStream;
                                end;
                        end
                        else begin
                                Log.Write( 'Not able to decrypt Art.' );
                        end;
                end
                else begin
                        if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                Gefunden := True;
                                mulStream := mulShardStream;
                        end;
                end;
        end;

        if not Gefunden then begin
                if (Id+$4000)*12 >= idxBaseStream.Size then begin
                        CriticalMulSection.Leave;
                        Result := 0;
                        exit;
                end;

                idxBaseStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxBaseStream.Read( IndexRecord, 12 );

                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        CriticalMulSection.Leave;
                        Result := 0;
                        exit;
                end;

                mulStream := mulBaseStream;
        end;
        mulStream.Seek( IndexRecord.Offset, soFromBeginning );

        mulStream.Read( Unknown, 4 );
        mulStream.Read( Breite, 2 );
        //mulStream.Read( Hoehe, 2 );

        CriticalMulSection.Leave;

        Result := Breite;
end;

function TArt.GetStaticArtHeight( Id : LongWord ) : Word;
var     IndexRecord : TIndexRecord;
        Hoehe, Breite : Word;
        Unknown : Integer;
        mulStream : TFileStream;
        Gefunden : Boolean;
begin
        Gefunden := False;

        CriticalMulSection.Enter;
        if (idxShardStream <> nil) and (mulShardStream <> nil) and ((Id+$4000)*12 < idxShardStream.Size) then begin
                idxShardStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxShardStream.Read( IndexRecord, 12 );

                if Palanthir.MulEncryption then begin
                        if Palanthir.Data.MulCrypt.DecryptIndexRecordArt( Id+$4000, IndexRecord ) then begin
                                if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                        Gefunden := True;
                                        mulStream := mulShardStream;
                                end;
                        end
                        else begin
                                Log.Write( 'Not able to decrypt Art.' );
                        end;
                end
                else begin
                        if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                Gefunden := True;
                                mulStream := mulShardStream;
                        end;
                end;
        end;

        if not Gefunden then begin
                if (Id+$4000)*12 >= idxBaseStream.Size then begin
                        CriticalMulSection.Leave;
                        Result := 0;
                        exit;
                end;

                idxBaseStream.Seek( (Id+$4000)*12, soFromBeginning );
                idxBaseStream.Read( IndexRecord, 12 );

                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        CriticalMulSection.Leave;
                        Result := 0;
                        exit;
                end;

                mulStream := mulBaseStream;
        end;
        mulStream.Seek( IndexRecord.Offset, soFromBeginning );

        mulStream.Read( Unknown, 4 );
        mulStream.Read( Breite, 2 );
        mulStream.Read( Hoehe, 2 );

        CriticalMulSection.Leave;

        Result := Hoehe;
end;

end.
