unit uLoaderLight;

interface

uses    Classes, SysUtils, uUtilities, Dialogs, uTexCache, dglOpenGL,
        uLoaderHues, Math, uLog, uRenderer;

type    TLights = Class
                private
                        idxBaseStream, mulBaseStream, idxShardStream, mulShardStream : TFileStream;
                        Cache : TTexCache;
                        function LoadLight( Id, Hue : Word ) : TTexObject;
                public
                        constructor Create( baseDir, shardDir : String );
                        destructor Free;
                        function GetLight( Id, Hue : Word ) : TTexObject;
                        procedure IncLight( Id, Hue : Word );
                        procedure DecLight( Id, Hue : Word );
        end;

implementation

uses    uPalanthir, uShard;

constructor TLights.Create( baseDir, shardDir : String );
begin
        idxBaseStream := TFileStream.Create( baseDir + 'Lightidx.mul', fmOpenRead + fmShareDenyNone );
        mulBaseStream := TFileStream.Create( baseDir + 'Light.mul', fmOpenRead + fmShareDenyNone );
        if FileExists( shardDir + 'Lightidx.mul' ) then
                idxShardStream := TFileStream.Create( shardDir + 'Lightidx.mul', fmOpenRead + fmShareDenyNone )
        else
                idxShardStream := nil;
        if FileExists( shardDir + 'Light.mul' ) then
                mulShardStream := TFileStream.Create( shardDir + 'Light.mul', fmOpenRead + fmShareDenyNone )
        else
                mulShardStream := nil;
        Cache := TTexCache.Create;
end;

destructor TLights.Free;
begin
        idxBaseStream.Free;
        mulBaseStream.Free;
        if idxShardStream <> nil then
                idxShardStream.Free;
        if mulShardStream <> nil then
                mulShardStream.Free;

        Cache.Free;
end;

function TLights.LoadLight( Id, Hue : Word ) : TTexObject;
var     IndexRecord : TIndexRecord;
        X, Y : Integer;
        Textur : PByteArray;
        mulStream : TFileStream;
        Color : Byte;
        ColorEntry : PLightTypEntry;
begin
        if (idxShardStream <> nil) and (mulShardStream <> nil) then begin
                if Id*12 >= idxShardStream.Size then begin
                        Result := nil;
                        exit;
                end;

                idxShardStream.Seek( Id*12, soFromBeginning );
                idxShardStream.Read( IndexRecord, 12 );
                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        Result := nil;
                        exit;
                end;

                mulStream := mulShardStream;
        end
        else begin
                if Id*12 >= idxBaseStream.Size then begin
                        Result := nil;
                        exit;
                end ;

                idxBaseStream.Seek( Id*12, soFromBeginning );
                idxBaseStream.Read( IndexRecord, 12 );
                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        Result := nil;
                        exit;
                end;

                mulStream := mulBaseStream;
        end;

        Result := TTexObject.Create( Tex_Light );
        Result.ID := Id;
        Result.Hue := Hue;
        Result.Breite := IndexRecord.Extra shr 16;
        Result.Hoehe := IndexRecord.Extra and $FFFF;
        Result.Count := 1;

        ColorEntry := Palanthir.Shard.GetLightColorEntry( Hue );

        mulStream.Seek( IndexRecord.Offset, soFromBeginning );

        GetMem( Textur, GetNextBit( Result.Breite )*GetNextBit( Result.Hoehe )*4 );

        for Y := 0 to Result.Hoehe-1 do
                for X := 0 to Result.Breite-1 do begin
                        mulStream.Read( Color, 1 );
                        if Color > 0 then begin
                                if ColorEntry = nil then begin
                                        Color := Color*8;
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4 ] := Color;
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+1 ] := Color;
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+2 ] := Color;
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+3 ] := 255;
                                end
                                else begin
                                        if Color > 31 then
                                                Color := 31;
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4 ] := ColorEntry.Red[ Color ];
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+1 ] := ColorEntry.Green[ Color ];
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+2 ] := ColorEntry.Blue[ Color ];
                                        Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+3 ] := 255;
                                end;
                        end
                        else begin
                                Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4 ] := 0;
                                Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+1 ] := 0;
                                Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+2 ] := 0;
                                Textur^[ (Y*GetNextBit( Result.Breite ) + X)*4+3 ] := 0;
                        end;

                end;

        glGenTextures( 1, @Result.TexID );
        glBindTexture( GL_TEXTURE_2D, Result.TexID );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, GetNextBit( Result.Breite ), GetNextBit( Result.Hoehe ), 0, GL_RGBA, GL_UNSIGNED_BYTE, Textur );
        FreeMem( Textur );
        Renderer.CurrentTexID := Result.TexID;        

        Cache.AddObject( Result );
end;

function TLights.GetLight( Id, Hue : Word ) : TTexObject;
begin
        try
                Result := Cache.GetTexObject( Id, Hue );

                if Result = nil then
                        Result := LoadLight( Id, Hue );
        except
                on E : Exception do begin
                        Log.Write( Format( 'uLoaderLight: %s' , [E.Message] ) );
                        raise E;
                end;
        end;
end;

procedure TLights.IncLight( Id, Hue : Word );
begin
        Cache.IncCount( Id, Hue );
end;

procedure TLights.DecLight( Id, Hue : Word );
begin
        Cache.DecCount( Id, Hue );
end;

end.
