unit uLoaderMap;

interface

uses    Classes, SysUtils, uUtilities, Dialogs, uUObject, uBasicTypedefs,
        uWorldCache, uPos, uShard, uLog, uVector, uLoaderTiledata, uTexCache,
        dglOpenGL, Md5, uLoaderVerdata, uRenderer, uTimer, SyncObjs;

const   DefaultMap = 0;

type    TMaps = Class;
        TMap = Class
                private
                        shardDir, baseDir : String;
                        mapStream : TFileStream;
                        idxStream, mulStream : TFileStream;
                        WorldRadar : TTexObject;
                        ShardDigest : TMD5Digest;
                        MapName, IdxName, MulName : String;
                        Maps : TMaps;
                        function InternalGetMapHeight( X, Y : Word ) : ShortInt;
                        procedure InternalLoadWorldRadar( shardDir : String );
                public
                        Breite, Hoehe, Index : Word;
                        constructor Create;
                        destructor Free;
                        function Init( baseDir, shardDir, MapName, StaIdxName, StaMulName : String; Width, Height : Word ) : Boolean;
                        procedure LoadBlock( Block : TLoadBlock );
                        function GetMapHeight( X, Y : Word ) : ShortInt;
                        function GetRadarTexture( BlockID : LongWord ) : TTexObject;
                        function GetWorldRadar : TTexObject;
                        function GetStream( FileName : String ) : TFileStream;
                        procedure FreeStream( FileStream : TFileStream );
                        function InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;
        end;

        TMaps = Class
                private
                        Cache : TList;
                        function GetMapFromIndex( Index : Word ) : TMap;
                public
                        MapSection, StaticSection : TCriticalSection;                
                        constructor Create( baseDir, shardDir : String );
                        destructor Free;
                        function GetWidth : Word;
                        function GetHeight : Word;
                        procedure LoadBlock( Block : TLoadBlock );
                        function GetMapHeight( X, Y : Word ) : ShortInt;
                        function GetRadarTexture( BlockID : LongWord ) : TTexObject;
                        function GetWorldRadar : TTexObject;
                        function GetStream( FileName : String ) : TFileStream;
                        procedure FreeStream( FileStream : TFileStream );
                        function InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;
        end;

implementation

uses    uPalanthir;

constructor TMaps.Create( baseDir, shardDir : String );
var     I : Integer;
        Entry : PMapEntry;
        Map : TMap;
begin
        Cache := TList.Create;

        MapSection := TCriticalSection.Create;
        StaticSection := TCriticalSection.Create;

        for I := 0 to Palanthir.Shard.MapListe.Count-1 do begin
                Entry := PMapEntry( Palanthir.Shard.MapListe.Items[ I ] );
                Map := TMap.Create;
                Map.Index := Entry.MapId;
                Map.Maps := Self;
                if Map.Init( baseDir, shardDir, Entry.MapName, Entry.StaIdxName, Entry.StaMulName, Entry.Breite, Entry.Hoehe ) then begin
                        Cache.Add( Map );
                        log.Write( Format( 'Map %d loaded.', [Map.Index] ) );
                end
                else
                        Map.Free;
        end;

        if Cache.Count = 0 then
                log.Write( 'No map loaded.' );
end;

destructor TMaps.Free;
var     I : Integer;
        Map : TMap;
begin
        for I := Cache.Count-1 downto 0 do begin
                Map := TMap( Cache.Items[ I ] );
                Cache.Delete( I );
                Map.Free;
        end;
        Cache.Free;

        MapSection.Free;
        StaticSection.Free;
end;

function TMaps.GetMapFromIndex( Index : Word ) : TMap;
var     I : Integer;
begin
        for I := 0 to Cache.Count-1 do begin
                if TMap( Cache.Items[ I ] ).Index = Index then begin
                        Result := TMap( Cache.Items[ I ] );
                        exit;
                end;
        end;
        Result := nil;
end;

procedure TMaps.LoadBlock( Block : TLoadBlock );
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Map.LoadBlock( Block );
end;

function TMaps.GetMapHeight( X, Y : Word ) : ShortInt;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.GetMapHeight( X, Y )
        else
                Result := 0;
end;

function TMaps.GetWidth : Word;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.Breite
        else
                Result := 0;
end;

function TMaps.GetHeight : Word;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.Hoehe
        else
                Result := 0;
end;

function TMaps.GetRadarTexture( BlockID : LongWord ) : TTexObject;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.GetRadarTexture( BlockID )
        else
                Result := nil;
end;

function TMaps.GetWorldRadar : TTexObject;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.GetWorldRadar
        else
                Result := nil;
end;

function TMaps.GetStream( FileName : String ) : TFileStream;
var     Map : TMap;
        I : Integer;
        Stream : TFileStream;
begin
        for I := 0 to Cache.Count-1 do begin
                Map := GetMapFromIndex( I );
                Stream := Map.GetStream( FileName );

                if Stream <> nil then begin
                        Result := Stream;
                        exit;
                end;
        end;

        Result := nil;
end;

procedure TMaps.FreeStream( FileStream : TFileStream );
var     Map : TMap;
        I : Integer;
begin
        for I := 0 to Cache.Count-1 do begin
                Map := GetMapFromIndex( I );
                Map.FreeStream( FileStream );
        end;

end;

function TMaps.InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;
var     Map : TMap;
begin
        Map := GetMapFromIndex( Palanthir.CurrentMap );
        if Map = nil then
                Map := GetMapFromIndex( DefaultMap );
        if Map <> nil then
                Result := Map.InitMap( BlockID, MapBlock )
        else
                Result := nil;
end;

constructor TMap.Create;
begin
        mapStream := nil;
        idxStream := nil;
        mulStream := nil;
        WorldRadar := nil;
end;

destructor TMap.Free;
begin
        if mapStream <> nil then begin
                Maps.FreeStream( mapStream );
                mapStream.Free;
        end;
        if idxStream <> nil then begin
                Maps.FreeStream( idxStream );
                idxStream.Free;
        end;
        if mulStream <> nil then begin
                Maps.FreeStream( mulStream );
                mulStream.Free;
        end;
        if WorldRadar <> nil then
                WorldRadar.Free;
end;

function TMap.GetStream( FileName : String ) : TFileStream;
begin
        if FileName = MapName then
                Result := mapStream
        else if FileName = IdxName then
                Result := idxStream
        else if FileName = MulName then
                Result := mulStream
        else
                Result := nil;
end;

procedure TMap.FreeStream( FileStream : TFileStream );
begin
        if mapStream = FileStream then
                mapStream := nil;
        if idxStream = FileStream then
                idxStream := nil;
        if mulStream = FileStream then
                mulStream := nil;
end;

function TMap.Init( baseDir, shardDir, MapName, StaIdxName, StaMulName : String; Width, Height : Word ) : Boolean;
begin
        Breite := Width;
        Hoehe := Height;
        Self.shardDir := shardDir;
        Self.baseDir := baseDir;

        if FileExists( shardDir + MapName ) then begin
                mapStream := Maps.GetStream( shardDir + MapName );
                if mapStream = nil then begin
                        try
                                mapStream := TFileStream.Create( shardDir + MapName, fmOpenRead or fmShareDenyNone );
                        except
                                Log.Write( 'Could not open map0.mul' );
                                Palanthir.State := State_Ending;
                                Result := False;
                                Exit;
                        end;
                end;
                mapStream.Seek( 0, soFromBeginning );
                ShardDigest := Md5Stream( mapStream );
                Self.MapName := shardDir + MapName;
        end
        else if FileExists( baseDir + MapName ) then begin
                mapStream := Maps.GetStream( baseDir + MapName );
                if mapStream = nil then begin
                        try
                                mapStream := TFileStream.Create( baseDir + MapName, fmOpenRead + fmShareDenyNone );
                        except
                                Log.Write( 'Could not open map0.mul' );
                                Palanthir.State := State_Ending;
                                Exit;
                        end;
                end;
                mapStream.Seek( 0, soFromBeginning );
                ShardDigest := Md5Stream( mapStream );
                Self.MapName := baseDir + MapName;
        end
        else begin
                Result := False;
                exit;
        end;

        if FileExists( shardDir + StaMulName ) and FileExists( shardDir + StaIdxName ) then begin
                idxStream := Maps.GetStream( shardDir + StaIdxName );
                if idxStream = nil then begin
                        try
                                idxStream := TFileStream.Create( shardDir + StaIdxName, fmOpenRead + fmShareDenyNone );
                        except
                                Log.Write( 'Could not open staidx0.mul' );
                                Palanthir.State := State_Ending;
                                Exit;
                        end;
                end;
                mulStream := Maps.GetStream( shardDir + StaMulName );
                if mulStream = nil then begin
                        try
                                mulStream := TFileStream.Create( shardDir + StaMulName, fmOpenRead + fmShareDenyNone );
                        except
                                Log.Write( 'Could not open statics0.mul' );
                                Palanthir.State := State_Ending;
                                Exit;
                        end;
                end;
                Self.IdxName := shardDir + StaIdxName;
                Self.MulName := shardDir + StaMulName;
        end
        else if FileExists( baseDir + StaMulName ) and FileExists( baseDir + StaIdxName ) then begin
                idxStream := Maps.GetStream( baseDir + StaIdxName );
                if idxStream = nil then begin
                        try
                                idxStream := TFileStream.Create( baseDir + StaIdxName, fmOpenRead + fmShareDenyNone );
                        except
                                Log.Write( 'Could not open staidx0.mul' );
                                Palanthir.State := State_Ending;
                                Exit;
                        end;
                end;
                mulStream := Maps.GetStream( baseDir + StaMulName );
                if mulStream = nil then begin
                        try
                                mulStream := TFileStream.Create( baseDir + StaMulName, fmOpenRead + fmShareDenyNone );
                        except
                                Log.Write( 'Could not open statics0.mul' );
                                Palanthir.State := State_Ending;
                                Exit;
                        end;
                end;
                Self.IdxName := baseDir + StaIdxName;
                Self.MulName := baseDir + StaMulName;
        end
        else begin
                mapStream.Free;
                mapStream := nil;
                Result := False;
                exit;
        end;

        WorldRadar := nil;
        Result := True;
end;

function TMap.InternalGetMapHeight( X, Y : Word ) : ShortInt;
var     BlockID : LongWord;
        MapBlock : TMapBlock;
begin
        BlockID := ( X div 8 ) * Hoehe + ( Y div 8 );

        Maps.MapSection.Enter;
        if BlockId*sizeof( TMapBlock ) > mapStream.Size then begin
                Result := 0;
        end
        else begin
                mapStream.Seek( BlockId*sizeof( TMapBlock ), soFromBeginning );
                mapStream.Read( MapBlock, sizeof( TMapBlock ) );

                Result := MapBlock.Cells[ (Y mod 8)*8+(X mod 8) ].Z;
        end;
        Maps.MapSection.Leave;
end;

procedure TMap.InternalLoadWorldRadar( shardDir : String );
var     Pixel : PByteArray;
        X, Y, BId, Count : Integer;
        Color, PixelCount : LongWord;
        MapBlock : TMapBlock;
        RBreite : Word;
        FStream : TFileStream;
        H1, H2, H3, H4 : LongWord;
        Load : Boolean;
begin
        PixelCount := GetNextBit( Breite div 2 )*GetNextBit( Hoehe div 2 )*3;
        GetMem( Pixel, PixelCount );

        if FileExists( Format( '%sMapCache%d.bin', [shardDir,Index] ) ) then begin
                FStream := TFileStream.Create( Format( '%sMapCache%d.bin', [shardDir,Index] ), fmOpenRead );
                FStream.Seek( 0, soFromBeginning );
                FStream.Read( H1, 4 );
                FStream.Read( H2, 4 );
                FStream.Read( H3, 4 );
                FStream.Read( H4, 4 );

                if (ShardDigest.A <> H1) or (ShardDigest.B <> H2) or (ShardDigest.C <> H3) or (ShardDigest.D <> H4) then begin
                        FStream.Free;
                        DeleteFile( Format( '%sMapCache%d.bin', [shardDir,Index] ) );
                        Load := True;
                end
                else begin
                        FStream.Read( Pixel^, PixelCount );
                        FStream.Free;
                        Load := False;
                end;
        end
        else begin
                Load := True;
        end;

        if Load then begin
                FStream := TFileStream.Create( Format( '%sMapCache%d.bin', [shardDir,Index] ), fmCreate );
                FStream.Seek( 0, soFromBeginning );
                FStream.Write( ShardDigest.A, 4 );
                FStream.Write( ShardDigest.B, 4 );
                FStream.Write( ShardDigest.C, 4 );
                FStream.Write( ShardDigest.D, 4 );

                RBreite := GetNextBit( Breite div 2 );

                for Y := 0 to (Hoehe div 2)-1 do begin
                        for X := 0 to (Breite div 2)-1 do begin
                                BId := (X*2)*Hoehe + (Y*2);

                                Maps.MapSection.Enter;
                                if mapStream.Size > BId*SizeOf( TMapBlock ) then begin
                                        mapStream.Seek( BId*SizeOf( TMapBlock ), soFromBeginning );
                                        mapStream.Read( MapBlock, SizeOf( TMapBlock ) );

                                        Count := ( Y*RBreite + X )*3;
                                        Color := Palanthir.Data.GetMapRadarColor( MapBlock.Cells[0].Id );

                                        Pixel^[ Count ] := Color15toRed( Color );
                                        Pixel^[ Count+1 ] := Color15toGreen( Color );
                                        Pixel^[ Count+2 ] := Color15toBlue( Color );
                                end;
                                Maps.MapSection.Leave;
                        end;
                end;

                FStream.Write( Pixel^, PixelCount );
                FStream.Free;
        end;


        WorldRadar := TTexObject.Create( Tex_Radar );
        WorldRadar.Breite := (Breite div 2);
        WorldRadar.Hoehe := (Hoehe div 2);

        glGenTextures( 1, @WorldRadar.TexID );
        glBindTexture( GL_TEXTURE_2D, WorldRadar.TexID );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, GetNextBit( WorldRadar.Breite ), GetNextBit( WorldRadar.Hoehe ), 0, GL_RGB, GL_UNSIGNED_BYTE, Pixel );
        FreeMem( Pixel );
        Renderer.CurrentTexID := WorldRadar.TexID;
end;

procedure TMap.LoadBlock( Block : TLoadBlock );
var     MapBlock : TMapBlock;
        StaticItem : TStaticItem;
        IndexRecord : TIndexRecord;
        MapItem : TMapItem;
        PX, PY : Word;
        X, Y : Byte;
        I : Integer;
        SID, SColor : Word;
        VerdataEntry : TVerdataEntry;
        TmpMulStream : TStream;
        TmpStream : TMemoryStream;
begin
        Maps.MapSection.Enter;
        if Block.BlockId*sizeof( TMapBlock ) >= mapStream.Size then begin
                GetMem( Block.MapListe, 64*sizeof( TMapItem ) );
                for Y := 0 to 63 do begin
                        MapItem := TMapItem.Create;
                        MapItem.Pos.Z := 0;
                        MapItem.Pos.Map := Palanthir.CurrentMap;
                        MapItem.Z2 := 0;
                        MapItem.Z3 := 0;
                        MapItem.Z4 := 0;
                        MapItem.Typ := Typ_map;
                        MapItem.ID := $2;
                        MapItem.Color := 0;
                        PX := ( Block.BlockID div Hoehe )*8 + X mod 8;
                        PY := ( Block.BlockID mod Hoehe )*8 + Y div 8;
                        MapItem.Pos.X := PX;
                        MapItem.Pos.Y := PY;
                        Block.MapListe^[ Y ] := MapItem;
                end;

                Maps.MapSection.Leave;
                Exit;
        end else begin
                Maps.MapSection.Leave;
        end;

        Maps.MapSection.Enter;
        mapStream.Seek( Block.BlockId*sizeof( TMapBlock ), soFromBeginning );
        mapStream.Read( MapBlock, sizeof( TMapBlock ) );
        Maps.MapSection.Leave;

        Block.MapListe := InitMap( Block.BlockID, MapBlock );
        
        VerdataEntry := Palanthir.Data.Verdata.GetStaticEntry( Block.BlockID );
        if VerdataEntry <> nil then begin
                IndexRecord := VerdataEntry.IndexRecord;
                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        Block.StaticCount := 0;
                        Timer.Pause( Timer_LoadMap );
                        exit;
                end;
                TmpMulStream := VerdataEntry.mulStream;
        end
        else begin
                if Block.BlockID*12 >= idxStream.Size then begin
                        Block.StaticCount := 0;
                        Timer.Pause( Timer_LoadMap );
                        exit;
                end;

                Maps.StaticSection.Enter;
                idxStream.Seek( Block.BlockID*12, soFromBeginning );
                idxStream.Read( IndexRecord, 12 );
                Maps.StaticSection.Leave;
                TmpMulStream := mulStream;
        end;

        if ( IndexRecord.Offset = -1 ) or ( IndexRecord.Length <= 0 ) then begin
                Block.StaticCount := 0;
                Timer.Pause( Timer_LoadMap );
                exit;
        end
        else
                Block.StaticCount := IndexRecord.Length div 7;

        if Block.StaticListe <> nil then
                FreeMem( Block.StaticListe );
        GetMem( Block.StaticListe, Block.StaticCount*sizeof( TStaticItem ) );
        I := 0;

        TmpStream := TMemoryStream.Create;
        TmpStream.Seek( 0, soFromBeginning );

        Maps.StaticSection.Enter;
        TmpMulStream.Seek( IndexRecord.Offset, soFromBeginning );
        TmpStream.CopyFrom( TmpMulStream, IndexRecord.Length );
        Maps.StaticSection.Leave;

        TmpStream.Seek( 0, soFromBeginning );

        while TmpStream.Position < IndexRecord.Length do begin
                StaticItem := TStaticItem.Create;

                TmpStream.Read( SID, 2 );
                if SID > $7FFF then begin
                        Log.Write( Format( 'Invalid Static Item Id: %d  BlockId: %d', [SID,Block.BlockID] ) );
                        StaticItem.ID := 0;
                end
                else begin
                        StaticItem.ID := SID;
                end;

                TmpStream.Read( X, 1 );
                StaticItem.Pos.X := ( Block.BlockID div Hoehe )*8 + X;
                TmpStream.Read( Y, 1 );
                StaticItem.Pos.Y := ( Block.BlockID mod Hoehe )*8 + Y;
                TmpStream.Read( StaticItem.Pos.Z, 1 );
                StaticItem.Pos.Map := Palanthir.CurrentMap;
                TmpStream.Read( SColor, 2 );
                StaticItem.Color := SColor;

                StaticItem.Index := I;
                Block.StaticListe^[ I ] := StaticItem;
                StaticItem.DrawID := StaticItem.ID;
                Inc( I );
        end;

        TmpStream.Free;
end;

function TMap.GetMapHeight( X, Y : Word ) : ShortInt;
var     Block : TWorldBlock;
begin
        Block := Palanthir.Data.WorldCache.GetObjectWithoutLoad( ( X div 8 ) * GetMapHoehe + ( Y div 8 ) );
        if (Block <> nil) and (Block.MapListe <> nil) then begin
                if Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].IsStretched then
                        Result := ( Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].Pos.Z + Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].Z2 + Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].Z3 + Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].Z4 ) div 4
                else
                        Result := Block.MapListe^[ (Y mod 8)*8+(X mod 8) ].Pos.Z;
        end
        else begin
                Result := ( InternalGetMapHeight( X, Y ) + InternalGetMapHeight( X+1, Y ) + InternalGetMapHeight( X, Y+1 ) + InternalGetMapHeight( X+1, Y+1 ) ) div 4;
        end;
end;

function TMap.InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;
var     MapItem : TMapItem;
        BaseNormals : Array[-1..8,-1..8,0..3] of TVector;
        V, W : TVector;
        Cell, Left, Right, Bottom : ShortInt;
        X, Y, I : Integer;
        PX, PY : Word;
        X0, Y0 : Integer;
begin
        GetMem( Result, 64*sizeof( TMapItem ) );

        for Y := 0 to 7 do
                for X := 0 to 7 do begin
                        PX := ( BlockID div Hoehe )*8 + X;
                        PY := ( BlockID mod Hoehe )*8 + Y;

                        MapItem := TMapItem.Create;
                        MapItem.Pos.X := PX;
                        MapItem.Pos.Y := PY;
                        MapItem.Pos.Map := Palanthir.CurrentMap;

                        MapItem.Pos.Z := MapBlock.Cells[ Y*8+X ].Z;
                        if X < 7 then
                                MapItem.Z2 := MapBlock.Cells[ Y*8+(X+1) ].Z
                        else
                                MapItem.Z2 := InternalGetMapHeight( PX+1, PY );
                        if Y < 7 then
                                MapItem.Z3 := MapBlock.Cells[ (Y+1)*8+X ].Z
                        else
                                MapItem.Z3 := InternalGetMapHeight( PX, PY+1 );
                        if (Y < 7) and (X < 7) then
                                MapItem.Z4 := MapBlock.Cells[ (Y+1)*8+(X+1) ].Z
                        else
                                MapItem.Z4 := InternalGetMapHeight( PX+1, PY+1 );

                        if (Palanthir.Data.Tiledata.GetLandTexId( MapBlock.Cells[ Y*8+X ].Id ) > 0) and (not Palanthir.Data.Tiledata.GetLandFlag( MapBlock.Cells[ Y*8+X ].Id, dLIQUID ))  then
                                MapItem.Typ := typ_map_stretched
                        else
                                MapItem.Typ := typ_map;

                        MapItem.Color := 0;
                        MapItem.ID := MapBlock.Cells[ Y*8+X ].Id;

                        Result^[ Y*8+X ] := MapItem;
                end;

        for Y0 := -1 to 8 do begin
                for X0 := -1 to 8 do begin
                        PX := ( BlockID div Hoehe )*8 + X0;
                        PY := ( BlockID mod Hoehe )*8 + Y0;

                        Cell := InternalGetMapHeight( PX, PY );
                        Left := InternalGetMapHeight( PX, PY+1 );
                        Right := InternalGetMapHeight( PX+1, PY );
                        Bottom := InternalGetMapHeight( PX+1, PY+1 );

                        if (Cell = Left) and (Cell = Right) and (Cell = Bottom) then begin
                                BaseNormals[ X0, Y0, 0 ] := TVector.Create( 0, 0, 1 );
                                BaseNormals[ X0, Y0, 1 ] := TVector.Create( 0, 0, 1 );
                                BaseNormals[ X0, Y0, 2 ] := TVector.Create( 0, 0, 1 );
                                BaseNormals[ X0, Y0, 3 ] := TVector.Create( 0, 0, 1 );
                        end
                        else begin
                                V := TVector.Create( -22, 22, (Cell-Right)*4 );
                                W := TVector.Create( -22, -22, (Left-Cell)*4 );
                                V.Kreutzproduct( W );
                                V.Normalize;
                                BaseNormals[ X0, Y0, 0 ] := V;

                                V := TVector.Create( 22, 22, (Right-Bottom)*4 );
                                W.SetTo( -22, 22, (Cell-Right)*4 );
                                V.Kreutzproduct( W );
                                V.Normalize;
                                BaseNormals[ X0, Y0, 1 ] := V;

                                V := TVector.Create( 22, -22, (Bottom-Left)*4 );
                                W.SetTo( 22, 22, (Right-Bottom)*4 );
                                V.Kreutzproduct( W );
                                V.Normalize;
                                BaseNormals[ X0, Y0, 2 ] := V;

                                V := TVector.Create( -22, -22, (Left-Cell)*4 );
                                W.SetTo( 22, -22, (Bottom-Left)*4 );
                                V.Kreutzproduct( W );
                                V.Normalize;
                                BaseNormals[ X0, Y0, 3 ] := V;

                                W.Free;
                        end;
                end;
        end;

        for Y := 0 to 7 do begin
                for X := 0 to 7 do begin
                        MapItem := Result^[ Y*8+X ];

                        if MapItem.IsStretched then begin
                                MapItem.Normals[ 0 ].SetTo( BaseNormals[X-1,Y-1,2] );
                                MapItem.Normals[ 0 ].Add( BaseNormals[X-1,Y,1] );
                                MapItem.Normals[ 0 ].Add( BaseNormals[X,Y-1,3] );
                                MapItem.Normals[ 0 ].Add( BaseNormals[X,Y,0] );
                                MapItem.Normals[ 0 ].Normalize;

                                MapItem.Normals[ 1 ].SetTo( BaseNormals[X,Y-1,2] );
                                MapItem.Normals[ 1 ].Add( BaseNormals[X,Y,1] );
                                MapItem.Normals[ 1 ].Add( BaseNormals[X+1,Y-1,3] );
                                MapItem.Normals[ 1 ].Add( BaseNormals[X+1,Y,0] );
                                MapItem.Normals[ 1 ].Normalize;

                                MapItem.Normals[ 2 ].SetTo( BaseNormals[X,Y,2] );
                                MapItem.Normals[ 2 ].Add( BaseNormals[X,Y+1,1] );
                                MapItem.Normals[ 2 ].Add( BaseNormals[X+1,Y,3] );
                                MapItem.Normals[ 2 ].Add( BaseNormals[X+1,Y+1,0] );
                                MapItem.Normals[ 2 ].Normalize;

                                MapItem.Normals[ 3 ].SetTo( BaseNormals[X-1,Y,2] );
                                MapItem.Normals[ 3 ].Add( BaseNormals[X-1,Y+1,1] );
                                MapItem.Normals[ 3 ].Add( BaseNormals[X,Y,3] );
                                MapItem.Normals[ 3 ].Add( BaseNormals[X,Y+1,0] );
                                MapItem.Normals[ 3 ].Normalize;
                        end;
                end;
        end;

        for Y0 := -1 to 8 do begin
                for X0 := -1 to 8 do begin
                        for I := 0 to 3 do
                                BaseNormals[ X0, Y0, I ].Free;
                end;
        end;
end;

function TMap.GetRadarTexture( BlockID : LongWord ) : TTexObject;
var     Pixel : PByteArray;
        PX, PY, X, Y, BId, X0, Y0, Count : Integer;
        Color : Word;
        MapBlock : TMapBlock;
        IndexRecord : TIndexRecord;
        HeightMap : Array[0..63,0..63] of ShortInt;
        SColor, SId : Word;
        SX, SY : Byte;
        SZ : ShortInt;
        TmpStream : TMemoryStream;
begin
        Result := TTexObject.Create( Tex_Radar );
        Result.Breite := 64;
        Result.Hoehe := 64;

        GetMem( Pixel, Result.Breite*Result.Hoehe*4 );

        PX := (BlockID div GetMapHoehe)*64;
        PY := (BlockID mod GetMapHoehe)*64;

        for Y := 0 to 7 do begin
                for X := 0 to 7 do begin
                        BId := ((PX div 8) + X)*GetMapHoehe + ((PY div 8) + Y);

                        Maps.MapSection.Enter;
                        if mapStream.Size > BId*SizeOf( TMapBlock ) then begin
                                mapStream.Seek( BId*SizeOf( TMapBlock ), soFromBeginning );
                                mapStream.Read( MapBlock, SizeOf( TMapBlock ) );

                                for Y0 := 0 to 7 do begin
                                        for X0 := 0 to 7 do begin
                                                Count := ( ((Y*8)+Y0)*64 + (X*8+X0) )*4;
                                                Color := Palanthir.Data.GetMapRadarColor( MapBlock.Cells[Y0*8+X0].Id );

                                                Pixel[ Count ] := Color15toRed( Color );
                                                Pixel[ Count+1 ] := Color15toGreen( Color );
                                                Pixel[ Count+2 ] := Color15toBlue( Color );
                                                Pixel[ Count+3 ] := 255;

                                                HeightMap[X*8+X0,Y*8+Y0] := MapBlock.Cells[Y0*8+X0].Z;
                                        end;
                                end;
                        end;
                        Maps.MapSection.Leave;

                        if idxStream.Size > BId*SizeOf( TIndexRecord ) then begin
                                Maps.StaticSection.Enter;
                                idxStream.Seek( BId*SizeOf( TIndexRecord ), soFromBeginning );
                                idxStream.Read( IndexRecord, SizeOf( TIndexRecord ) );
                                Maps.StaticSection.Leave;

                                if (IndexRecord.Offset <> -1) and (IndexRecord.Length > 0) then begin
                                        TmpStream := TMemoryStream.Create;
                                        TmpStream.Seek( 0, soFromBeginning );

                                        Maps.StaticSection.Enter;
                                        mulStream.Seek( IndexRecord.Offset, soFromBeginning );
                                        TmpStream.CopyFrom( mulStream, IndexRecord.Length );
                                        Maps.StaticSection.Leave;

                                        TmpStream.Seek( 0, soFromBeginning );
                                        while TmpStream.Position < IndexRecord.Length do begin
                                                TmpStream.Read( SId, 2 );
                                                TmpStream.Read( SX, 1 );
                                                TmpStream.Read( SY, 1 );
                                                TmpStream.Read( SZ, 1 );
                                                TmpStream.Read( SColor, 2 );

                                                if (X*8+SX >= 0) and (X*8+SX < 64) and (Y*8+SY >= 0) and (Y*8+SY < 64) then begin
                                                        if SZ >= HeightMap[X*8+SX,Y*8+SY] then begin
                                                                HeightMap[X*8+SX,Y*8+SY] := SZ;
                                                                Color := Palanthir.Data.GetStaticRadarColor( SId );

                                                                Count := ( ((Y*8)+SY)*64 + (X*8+SX) )*4;
                                                                Pixel^[ Count ] := Color15toRed( Color );
                                                                Pixel^[ Count+1 ] := Color15toGreen( Color );
                                                                Pixel^[ Count+2 ] := Color15toBlue( Color );
                                                                Pixel^[ Count+3 ] := 255;
                                                        end;
                                                end;
                                        end;

                                        TmpStream.Free;
                                end;
                        end;
                end;
        end;

        glGenTextures( 1, @Result.TexID );
        glBindTexture( GL_TEXTURE_2D, Result.TexID );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, Result.Breite, Result.Hoehe, 0, GL_RGBA, GL_UNSIGNED_BYTE, Pixel );
        FreeMem( Pixel );
        Renderer.CurrentTexID := Result.TexID;
end;

function TMap.GetWorldRadar : TTexObject;
begin
        if WorldRadar = nil then
                InternalLoadWorldRadar( shardDir );                

        Result := WorldRadar;
end;

end.
