unit uLoaderMultis;

interface

uses    uUtilities, Classes, SysUtils, uLoaderVerdata, uPixelBuffer, uTexCache,
        uRenderer, Math, dglOpenGL, uBitMask, uLog;

type    TMultis = Class
                private
                        idxBaseStream, mulBaseStream, idxShardStream, mulShardStream : TFileStream;
                public
                        constructor Create( baseDir, shardDir : String );
                        destructor Free;
                        function GetItems( Id : Word ) : TList;
                        function GetMultiAsPixelBuffer( Id : Word; Color : Word = 0 ) : TPixelBuffer;
        end;

        TBasicMultiItem = Class
                public
                        ID : Word;
                        X, Y, Z : SmallInt;
                        Flag : Integer;
                        constructor Create;
                        destructor Free;
        end;


implementation

uses    uPalanthir;

constructor TMultis.Create( baseDir, shardDir : String );
begin
        try
                idxBaseStream := TFileStream.Create( baseDir + 'Multi.idx', fmOpenRead + fmShareDenyNone );
                mulBaseStream := TFileStream.Create( baseDir + 'Multi.mul', fmOpenRead + fmShareDenyNone );
        except
                Log.Write( 'Could not load multi.idx or multi.mul' );
                Palanthir.State := State_Ending;
                Exit;
        end;
        
        if FileExists( shardDir + 'Multi.idx' ) then
                idxShardStream := TFileStream.Create( shardDir + 'Multi.idx', fmOpenRead + fmShareDenyNone )
        else
                idxShardStream := nil;
        if FileExists( shardDir + 'Multi.mul' ) then
                mulShardStream := TFileStream.Create( shardDir + 'Multi.mul', fmOpenRead + fmShareDenyNone )
        else
                mulShardStream := nil;
end;

destructor TMultis.Free;
begin
        idxBaseStream.Free;
        mulBaseStream.Free;
        if idxShardStream <> nil then
                idxShardStream.Free;
        if mulShardStream <> nil then
                mulShardStream.Free;
end;

function TMultis.GetItems( Id : Word ) : TList;
var     IndexRecord : TIndexRecord;
        muLStream : TStream;
        Item : TBasicMultiItem;
        VerdataEntry : TVerdataEntry;
        Gefunden : Boolean;
begin
        Result := TList.Create;

        VerdataEntry := Palanthir.Data.Verdata.GetMulti( Id );
        if VerdataEntry <> nil then begin
                IndexRecord := VerdataEntry.IndexRecord;
                if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                        exit;
                end;
                mulStream := VerdataEntry.mulStream;
        end
        else begin
                Gefunden := False;
                if (idxShardStream <> nil) and (mulShardStream <> nil) and (Id*12 < idxShardStream.Size) then begin
                        idxShardStream.Seek( Id*12, soFromBeginning );
                        idxShardStream.Read( IndexRecord, 12 );
                        if (IndexRecord.Offset <> -1) and (IndexRecord.Length <> -1) then begin
                                Gefunden := True;
                                mulStream := mulShardStream;
                        end;
                end;

                if (not Gefunden) and (idxBaseStream <> nil) and (mulBaseStream <> nil) then begin
                        if Id*12 >= idxBaseStream.Size then begin
                                exit;
                        end;

                        idxBaseStream.Seek( Id*12, soFromBeginning );
                        idxBaseStream.Read( IndexRecord, 12 );

                        if (IndexRecord.Offset = -1) or (IndexRecord.Length = -1) then begin
                                exit;
                        end;

                        mulStream := mulBaseStream;
                end;
        end;

        mulStream.Seek( IndexRecord.Offset, soFromBeginning );
        while mulStream.Position < IndexRecord.Offset + IndexRecord.Length do begin
                Item := TBasicMultiItem.Create;
                mulStream.Read( Item.Id, 2 );
                mulStream.Read( Item.X, 2 );
                mulStream.Read( Item.Y, 2 );
                mulStream.Read( Item.Z, 2 );
                mulStream.Read( Item.Flag, 4 );
                Result.Add( Item );
        end;
end;

function TMultis.GetMultiAsPixelBuffer( Id : Word; Color : Word = 0 ) : TPixelBuffer;
var     Liste : TList;
        I : Integer;
        MinX, MaxX, MinY, MaxY : Integer;
        PX, PY : Integer;
        Item : TBasicMultiItem;
        Texture : TTexObject;
        Breite, Hoehe : Word;
        PosXMin, PosXMax, PosYMin, PosYMax : Integer;
        XMin, XMax, YMin, YMax : Integer;
        MX, MY : Integer;
begin
        Result := nil;

        if Id = 0 then
                exit;

        Liste := Palanthir.Data.GetMultiItems( Id );

        MinX := 0;
        MinY := 0;
        MaxX := 0;
        MaxY := 0;

        for I := 0 to Liste.Count-1 do begin
                Item := TBasicMultiItem( Liste.Items[ I ] );
                Texture := Palanthir.Data.GetStaticArt( Item.ID, Color, True );

                if Texture = nil then
                        continue;
                
                PX := (Item.X - Item.Y) * 22 - ( Texture.Breite div 2 );
                PY := (Item.X + Item.Y) * 22 - Item.Z*4 - Texture.Hoehe +44;
                if PX < MinX then
                        MinX := PX;
                if PX + Texture.Breite > MaxX then
                        MaxX := PX + Texture.Breite;
                if PY < MinY then
                        MinY := PY;
                if PY + Texture.Hoehe > MaxY then
                        MaxY := PY + Texture.Hoehe;
        end;

        MinX := Max( MinX, -Palanthir.GameWindow.Breite );
        MinY := Max( MinY, -Palanthir.GameWindow.Hoehe );
        MaxX := Min( MaxX, Palanthir.GameWindow.Breite );
        MaxY := Min( MaxY, Palanthir.GameWindow.Hoehe );        

        Breite := GetNextBit( MaxX - MinX );
        Hoehe := GetNextBit( MaxY - MinY );

        MX := MinX;
        MY := MinY;

        Result := TPixelBuffer.Create( Breite, Hoehe, Renderer.DC, Renderer.RC, nil );
        if Result.Init then begin
                wglShareLists( Renderer.RC, Result.RC );
        end
        else begin
                Result.Destroy;
                Result := nil;
                exit;
        end;

        Result.RelX := MX;
        Result.RelY := MY;

        Result.Enable;

        glEnable( GL_TEXTURE_2D );

        glMatrixMode( GL_PROJECTION );
        glLoadIdentity;
        glViewPort( 0, 0, Result.Width, Result.Height );

        glMatrixMode( GL_MODELVIEW );
        glLoadIdentity;
        glOrtho( 0, Result.Width, 0, Result.Height, -1, 1 );

        glClearColor( 0, 0, 0, 0 );
        glClear( GL_COLOR_BUFFER_BIT );

        glEnable( GL_ALPHA_TEST );
        glAlphaFunc( GL_GREATER, 0.1 );

        glColor3f( 1, 1, 1 );

        Palanthir.SortMultiItemListe( Liste );

        Result.BitMask := TBitMask.Create;
        Result.BitMask.InitEmpty( Result.Width, Result.Height );

        for I := Liste.Count-1 downto 0 do begin
                Item := TBasicMultiItem( Liste.Items[ I ] );
                Texture := Palanthir.Data.GetStaticArt( Item.ID, Color, True );
                if Texture = nil then
                        continue;

                if Renderer.CurrentTexID <> Texture.TexID then begin
                        glBindTexture( GL_TEXTURE_2D, Texture.TexID );
                        Renderer.CurrentTexID := Texture.TexID;
                end;

                PX := -MX + (Item.X - Item.Y) * 22 - ( Texture.Breite div 2 );
                PY := -MY + (Item.X + Item.Y) * 22 - Item.Z*4 - Texture.Hoehe +44;

                if ( PX >= Result.Width ) or ( PX + Texture.Breite < 0 ) or ( PY >= Result.Height ) or ( PY + Texture.Hoehe < 0 ) then
                        continue;

                XMin := Max( PX, 0 );
                PosXMin := XMin - PX;

                YMin := Max( PY, 0 );
                PosYMin := YMin - PY;

                XMax := Min( PX + Texture.Breite, Result.Width );
                PosXMax := XMax - PX;

                YMax := Min( PY + Texture.Hoehe, Result.Height );
                PosYMax := YMax - PY;

                glBegin( gl_Quads );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMin );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMin );
                glEnd;

                Item.Free;

                Result.BitMask.AddBitMask( XMin, YMin, Texture.BitMask );
        end;
        Liste.Free;
        
        Result.Disable;
end;

constructor TBasicMultiItem.Create;
begin
end;

destructor TBasicMultiItem.Free;
begin
end;

end.
 