unit uLoaderProgress;

interface

uses    Classes, SyncObjs, uUtilities, uLoaderHues, PngImage, SysUtils,
        Dialogs;

const   ProgressTypeStaticArt = 0;
        ProgressTypeGumps = 1;

type    TProgressRecord = packed Record
                Offset : Integer;
                Length : Integer;
                Id : Integer;
                TypeId : Integer;
        end;

        TProgressImage = Class;
        TProgress = Class
                private
                        idxStream, mulStream : TFileStream;
                        Loaded : Boolean;
                        StaticArt : Array[0..$7FFF] of Integer;
                        Gumps : Array[0..$FFFF] of Integer;
                        CriticalSection : TCriticalSection;
                        procedure DecryptProgressRecord( var ProgressRecord : TProgressRecord );
                        function GetImage( TypeId : Integer; Id : Integer; Hue : Word ) : TProgressImage;
                public
                        constructor Create( shardDir : String );
                        destructor Free;
                        function GetGump( Id : Word; Hue : Word = 0 ) : TProgressImage;
                        function GetStaticArt( Id : Word; Hue : Word = 0 ) : TProgressImage;                        
        end;

        TProgressImage = Class
                public
                        Width, Height : Word;
                        Data : PByteArray;
                        constructor Create( NWidth, NHeight : Word );
                        destructor Free;
        end;

implementation

uses    uPalanthir;

procedure TProgress.DecryptProgressRecord( var ProgressRecord : TProgressRecord );
var     IndexRecord : TIndexRecord;
begin
        IndexRecord.Offset := ProgressRecord.Offset;
        IndexRecord.Length := ProgressRecord.Length;
        IndexRecord.Extra := ProgressRecord.TypeId;

        if Palanthir.MulEncryption then begin
                if not Palanthir.Data.MulCrypt.DecryptIndexRecordTex( ProgressRecord.Id, IndexRecord ) then begin
                        Exit;
                end;
        end else begin
                Exit;
        end;

        ProgressRecord.Offset := IndexRecord.Offset;
        ProgressRecord.Length := IndexRecord.Length;
        ProgressRecord.TypeId := IndexRecord.Extra;
end;

function TProgress.GetImage( TypeId : Integer; Id : Integer; Hue : Word ) : TProgressImage;
var     PNG : TPNGObject;
        Stream : TMemoryStream;
        Pos : Integer;
        ProgressRecord : TProgressRecord;
        X, Y : Integer;
        Alpha : PByte;
        Color : PByteArray;
        RealHue : Word;
        PartialHue : Boolean;
        Red, Green, Blue : Byte;
begin
        if not Loaded then begin
                Result := nil;
                Exit;
        end;

        Case TypeId of
                ProgressTypeStaticArt: Pos := StaticArt[Id];
                ProgressTypeGumps: Pos := Gumps[Id];
        else
                Pos := -1;
        end;

        if Pos = -1 then begin
                Result := nil;
                Exit;
        end;

        CriticalSection.Enter;

        idxStream.Seek( Pos, soFromBeginning );
        idxStream.Read( ProgressRecord, sizeof( TProgressRecord ) );
        DecryptProgressRecord( ProgressRecord );

        mulStream.Seek( ProgressRecord.Offset, soFromBeginning );
        Stream := TMemoryStream.Create;
        Stream.CopyFrom( mulStream, ProgressRecord.Length );

        CriticalSection.Leave;

        Stream.Seek( 0, soFromBeginning );
        PNG := TPNGObject.Create;
        PNG.LoadFromStream( Stream );
        Stream.Free;

        Result := TProgressImage.Create( PNG.Width, PNG.Height );
        if Hue = 0 then begin
                for Y := 0 to PNG.Height-1 do begin
                        Pos := Y*GetNextBit( PNG.Width )*4;
                        Color := PByteArray( PNG.ScanLine[Y] );
                        Alpha := PByte( PNG.AlphaScanline[Y] );
                        for X := 0 to PNG.Width-1 do begin
                                Result.Data^[Pos] := Color^[X*3+2];
                                Inc( Pos );
                                Result.Data^[Pos] := Color^[X*3+1];
                                Inc( Pos );
                                Result.Data^[Pos] := Color^[X*3];
                                Inc( Pos );

                                Result.Data^[Pos] := Alpha^;
                                Inc( Alpha );
                                Inc( Pos );
                        end;
                end;
        end else begin
                RealHue := Hue mod $4000;
                PartialHue := ( ( Hue and $8000 ) = $8000 );
                if PartialHue then begin
                        for Y := 0 to PNG.Height-1 do begin
                                Pos := Y*GetNextBit( PNG.Width )*4;
                                Color := PByteArray( PNG.ScanLine[Y] );
                                Alpha := PByte( PNG.AlphaScanline[Y] );
                                for X := 0 to PNG.Width-1 do begin
                                        Red := Color^[X*3+2] div 8;
                                        Green := Color^[X*3+1] div 8;
                                        Blue := Color^[X*3] div 8;

                                        if (Red = Green) and (Green = Blue) then begin
                                                Result.Data^[Pos] := Color15toRed( Hues.GetColor( RealHue, Red ) );
                                                Inc( Pos );
                                                Result.Data^[Pos] := Color15toGreen( Hues.GetColor( RealHue, Red ) );
                                                Inc( Pos );
                                                Result.Data^[Pos] := Color15toBlue( Hues.GetColor( RealHue, Red ) );
                                                Inc( Pos );
                                        end else begin
                                                Result.Data^[Pos] := Red;
                                                Inc( Pos );
                                                Result.Data^[Pos] := Green;
                                                Inc( Pos );
                                                Result.Data^[Pos] := Blue;
                                                Inc( Pos );
                                        end;

                                        Result.Data^[Pos] := Alpha^;
                                        Inc( Alpha );
                                        Inc( Pos );
                                end;
                        end;
                end else begin
                        for Y := 0 to PNG.Height-1 do begin
                                Pos := Y*GetNextBit( PNG.Width )*4;
                                Color := PByteArray( PNG.ScanLine[Y] );
                                Alpha := PByte( PNG.AlphaScanline[Y] );
                                for X := 0 to PNG.Width-1 do begin
                                        Red := Color^[X*3+2] div 8;
                                                                                
                                        Result.Data^[Pos] := Color15toRed( Hues.GetColor( RealHue, Red ) );
                                        Inc( Pos );
                                        Result.Data^[Pos] := Color15toGreen( Hues.GetColor( RealHue, Red ) );
                                        Inc( Pos );
                                        Result.Data^[Pos] := Color15toBlue( Hues.GetColor( RealHue, Red ) );
                                        Inc( Pos );

                                        Result.Data^[Pos] := Alpha^;
                                        Inc( Alpha );
                                        Inc( Pos );
                                end;
                        end;
                end;
        end;
        PNG.Free;
end;

constructor TProgress.Create( shardDir : String );
var     ProgressRecord : TProgressRecord;
        I : Integer;
begin
        if not ( FileExists( shardDir + 'Progress.idx' ) and FileExists( shardDir + 'Progress.mul' ) ) then begin
                Loaded := False;
                Exit;
        end;

        for I := 0 to $7FFF do begin
                StaticArt[I] := -1;
        end;
        for I := 0 to $FFFF do begin
                Gumps[I] := -1;
        end;

        idxStream := TFileStream.Create( shardDir + 'Progress.idx', fmOpenRead );
        mulStream := TFileStream.Create( shardDir + 'Progress.mul', fmOpenRead );

        idxStream.Seek( 0, soFromBeginning );
        while idxStream.Position < idxStream.Size do begin
                idxStream.Read( ProgressRecord, sizeof( TProgressRecord ) );
                DecryptProgressRecord( ProgressRecord );
                case ProgressRecord.TypeId of
                        ProgressTypeStaticArt: StaticArt[ProgressRecord.Id] := idxStream.Position - sizeof( TProgressRecord );
                        ProgressTypeGumps: Gumps[ProgressRecord.Id] := idxStream.Position - sizeof( TProgressRecord );
                end;
        end;

        CriticalSection := TCriticalSection.Create;
        Loaded := True;
end;

destructor TProgress.Free;
begin
        if Loaded then begin
                Loaded := False;

                idxStream.Free;
                mulStream.Free;
                CriticalSection.Free;
        end;
end;

function TProgress.GetGump( Id : Word; Hue : Word = 0 ) : TProgressImage;
begin
        Result := GetImage( ProgressTypeGumps, Id, Hue );
end;

function TProgress.GetStaticArt( Id : Word; Hue : Word = 0 ) : TProgressImage;
begin
        Result := GetImage( ProgressTypeStaticArt, Id, Hue );
end;

constructor TProgressImage.Create( NWidth, NHeight : Word );
begin
        Width := NWidth;
        Height := NHeight;
        GetMem( Data, GetNextBit( NWidth )* GetNextBit( NHeight )*4 );
end;

destructor TProgressImage.Free;
begin
        FreeMem( Data );
end;

end.
