unit uLoaderUnicodeFonts;

interface

uses    SysUtils, Classes, uLoaderHues, uUtilities, Dialogs, uTexCache, dglOpenGL,
        StrLib, uRenderer, Math, uByteCache;

const   Whitespace = 3;

type    TUnicodeBuchstabenImage = Class
                public
                        xOffset, yOffset : ShortInt;
                        Hoehe, Breite : Integer;
                        Image : TByteCache;
                        constructor Create;
                        destructor Free;
        end;

        TUnicodeBuchstabe = Class
                public
                        xOffset, yOffset : ShortInt;
                        Hoehe, Breite : Byte;
                        Exists : Boolean;
                        Image, NonBorderImage : TByteCache;
                        constructor Create;
                        destructor Free;
        end;

        TLine = Record
                Text : String;
                Breite, Hoehe : Word;
        end;

        TUnicodeFont = Class
                public
                        mulStream : TFileStream;
                        Buchstabe : array[0..223] of TUnicodeBuchstabe;
                        constructor Create;
                        destructor Free;
        end;

        TUnicode = Class
                private
                        FontList : TList;
                        procedure LoadFont( FontNr : Integer );
                        function GetHoehe( FontNr : Byte; Buchstabe : Word; Border : Boolean = True ) : Byte;
                public
                        constructor Create( baseDir, shardDir : String );
                        destructor Free;
                        function GetBreite( FontNr : Byte; Buchstabe : Word; Border : Boolean = True ) : Byte;
                        function GetBuchstabe( FontNr : Byte; Buchstabe : Word; Hue : Word; RealColor : Integer = -1; Border : Boolean = True; System : Boolean = False ) : TUnicodeBuchstabenImage;
                        function GetUnicodeTexture( FontNr : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word; SelPos : SmallInt; Border, System : Boolean; DivHeight : Integer ) : TTexObject;
                        function GetUnicodeLines( FontNr : Byte; Text : String; MaxBreite : Word; Border : Boolean ) : TStringList;
                        function GetUnicodeHtmlTexture( FontNr : Byte; Text : String; MaxBreite : Word; Color : Word = 0 ) : TTexObject;                        
        end;

implementation

uses    uLog;

constructor TUnicodeBuchstabenImage.Create;
begin
        Image := nil;
end;

destructor TUnicodeBuchstabenImage.Free;
begin
        if Image <> nil then
                Image.Free;
end;

constructor TUnicodeBuchstabe.Create;
begin
        Image := nil;
        NonBorderImage := nil;
end;

destructor TUnicodeBuchstabe.Free;
begin
        if Image <> nil then
                Image.Free;
        if NonBorderImage <> nil then
                NonBorderImage.Free;
end;

constructor TUnicodeFont.Create;
var     I : Integer;
begin
        for I := 0 to 223 do begin
                Buchstabe[I] := TUnicodeBuchstabe.Create;
        end;
        mulStream := nil;
end;

destructor TUnicodeFont.Free;
var     I : Integer;
begin
        for I := 0 to 223 do begin
                Buchstabe[I].Free;
        end;
        if mulStream <> nil then
                mulStream.Free;
end;

constructor TUnicode.Create( baseDir, shardDir : String );
var     I : Integer;
        mulStream : TFileStream;
        Font : TUnicodeFont;
begin
        FontList := TList.Create;
        if FileExists( shardDir + 'Unifont.mul' ) then
                mulStream := TFileStream.Create( shardDir + 'Unifont.mul', fmOpenRead + fmShareDenyNone )
        else
                mulStream := TFileStream.Create( baseDir + 'Unifont.mul', fmOpenRead + fmShareDenyNone );

        Font := TUnicodeFont.Create;
        Font.mulStream := mulStream;
        FontList.Add( Font );

        I := 1;
        while FileExists( Format( '%sUnifont%d.mul', [baseDir,I] ) ) or FileExists( Format( '%sUnifont%d.mul', [shardDir,I] ) ) do begin
                if FileExists( Format( '%sUnifont%d.mul', [shardDir,I] ) ) then
                        mulStream := TFileStream.Create( Format( '%sUnifont%d.mul', [shardDir,I] ), fmOpenRead + fmShareDenyNone )
                else
                        mulStream := TFileStream.Create( Format( '%sUnifont%d.mul', [baseDir,I] ), fmOpenRead + fmShareDenyNone );

                Font := TUnicodeFont.Create;
                Font.mulStream := mulStream;
                FontList.Add( Font );
                
                Inc( I );
        end;

        for I := 0 to FontList.Count-1 do
                LoadFont( I );
end;

destructor TUnicode.Free;
var     I : Integer;
        Font : TUnicodeFont;
begin
        for I := 0 to FontList.Count-1 do begin
                Font := TUnicodeFont( FontList.Items[ I ] );
                Font.Free;
        end;
        FontList.Free;
end;

function TUnicode.GetHoehe( FontNr : Byte; Buchstabe : Word; Border : Boolean ) : Byte;
var     Font : TUnicodeFont;
        Lookup : LongWord;
        B : ShortInt;
begin
        if FontNr > FontList.Count-1 then
                FontNr := 0;
        Font := TUnicodeFont( FontList.Items[ FontNr ] );

        if (Buchstabe >= 32) and (Buchstabe < 256) then begin
                Buchstabe := Buchstabe - 32;
                Result := Font.Buchstabe[ Buchstabe ].Hoehe + Font.Buchstabe[ Buchstabe ].yOffset;
        end
        else begin
                Font.mulStream.Seek( Buchstabe*4, soFromBeginning );
                Font.mulStream.Read( Lookup, 4 );
                Font.mulStream.Seek( Lookup, soFromBeginning );
                Font.mulStream.Read( B, 1 );
                Result := B;
                Font.mulStream.Read( B, 1 );
                Font.mulStream.Read( B, 1 );
                if Border then
                        Result := Result + B + 2
                else
                        Result := Result + B;
        end;
end;

function TUnicode.GetBreite( FontNr : Byte; Buchstabe : Word; Border : Boolean ) : Byte;
var     Font : TUnicodeFont;
        Lookup : LongWord;
        B : ShortInt;
begin
        if FontNr > FontList.Count-1 then
                FontNr := 0;
        Font := TUnicodeFont( FontList.Items[ FontNr ] );

        if (Buchstabe >= 32) and (Buchstabe < 256) then begin
                Buchstabe := Buchstabe - 32;
                Result := Font.Buchstabe[ Buchstabe ].Breite + Font.Buchstabe[ Buchstabe ].xOffset;
        end
        else begin
                Font.mulStream.Seek( Buchstabe*4, soFromBeginning );
                Font.mulStream.Read( Lookup, 4 );
                Font.mulStream.Seek( Lookup+1, soFromBeginning );
                Font.mulStream.Read( B, 1 );
                Result := B;
                Font.mulStream.Read( B, 1 );
                Font.mulStream.Read( B, 1 );
                if Border then
                        Result := Result + B + 2
                else
                        Result := Result + B + 2;
        end;
end;

procedure TUnicode.LoadFont( FontNr : Integer );
var     Font : TUnicodeFont;
        I, X, Y, J : Integer;
        Lookup : LongWord;
        P : Byte;
begin
        Font := TUnicodeFont( FontList.Items[ FontNr ] );

        for I := 0 to 223 do begin
                Font.mulStream.Seek( (I+32)*4, soFromBeginning );
                Font.mulStream.Read( Lookup, 4 );
                Font.mulStream.Seek( Lookup, soFromBeginning );

                Font.mulStream.Read( Font.Buchstabe[ I ].xOffset, 1 );
                Font.mulStream.Read( Font.Buchstabe[ I ].yOffset, 1 );
                Font.mulStream.Read( Font.Buchstabe[ I ].Breite, 1 );
                Font.mulStream.Read( Font.Buchstabe[ I ].Hoehe, 1 );

                if (Font.Buchstabe[ I ].Breite = 0) or (Font.Buchstabe[ I ].Hoehe = 0) then begin
                        Font.Buchstabe[ I ].Exists := False;
                        continue;
                end
                else begin
                        Font.Buchstabe[ I ].Exists := True;
                end;

                Font.Buchstabe[ I ].Image := TByteCache.Create( Font.Buchstabe[ I ].Breite+2, Font.Buchstabe[ I ].Hoehe+2 );

                for Y := Font.Buchstabe[ I ].Hoehe-1 downto 0 do begin
                        for X := 0 to ((Font.Buchstabe[ I ].Breite-1) div 8) do begin
                                Font.mulStream.Read( P, 1 );
                                for J := 0 to 7 do begin
                                        if X*8+J < Font.Buchstabe[ I ].Breite then begin
                                                if ( P and (1 shl (7-J))) = (1 shl (7-J)) then begin
                                                        Font.Buchstabe[ I ].Image.Pixels[X*8+J+1,Y+1].Alpha := 255;
                                                end;
                                        end;
                                end;
                        end;
                end;

                Font.Buchstabe[ I ].Breite := Font.Buchstabe[ I ].Breite+2;
                Font.Buchstabe[ I ].Hoehe := Font.Buchstabe[ I ].Hoehe+2;

                Font.Buchstabe[ I ].NonBorderImage := TByteCache.Create( Font.Buchstabe[ I ].Breite, Font.Buchstabe[ I ].Hoehe );
                Font.Buchstabe[ I ].NonBorderImage.CopyFrom( Font.Buchstabe[ I ].Image, 0, 0 );

                for Y := 0 to Font.Buchstabe[ I ].Hoehe-1 do begin
                        for X := 0 to Font.Buchstabe[ I ].Breite-1 do begin
                                if Font.Buchstabe[ I ].Image.Pixels[ X, Y ].Alpha = 0 then begin
                                        if Font.Buchstabe[ I ].Image.Pixels[X-1,Y-1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X,Y-1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X+1,Y-1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X+1,Y].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X+1,Y+1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X,Y+1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X-1,Y+1].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                        if Font.Buchstabe[ I ].Image.Pixels[X-1,Y].Alpha = 255 then begin
                                                Font.Buchstabe[ I ].Image.Pixels[X,Y].Alpha := 254;
                                                continue;
                                        end;
                                end;
                        end;
                end;
        end;
end;

function TUnicode.GetBuchstabe( FontNr : Byte; Buchstabe : Word; Hue : Word; RealColor : Integer; Border, System : Boolean ) : TUnicodeBuchstabenImage;
var     mulStream : TFileStream;
        Lookup, Count : LongWord;
        P : Byte;
        Y, X, J : Integer;
        Font : TUnicodeFont;
        Red, Blue, Green : Byte;
begin
        Result := TUnicodeBuchstabenImage.Create;

        if FontNr > FontList.Count-1 then
                FontNr := 0;
        Font := TUnicodeFont( FontList.Items[ FontNr ] );

        if RealColor >= 0 then begin
                Red := ( RealColor shr 16 ) and $FF;
                Green := ( RealColor shr 8 ) and $FF;
                Blue := RealColor and $FF;
        end
        else begin
                if Hue = 0 then
                        Hue := $3CE;
                if not System then begin
                        Red := Color15toRed( Hues.GetColor( Hue, 28 ) );
                        Green := Color15toGreen( Hues.GetColor( Hue, 28 ) );
                        Blue := Color15toBlue( Hues.GetColor( Hue, 28 ) );
                end
                else begin
                        Red := Color15toRed( Hues.GetColor( Hue, 14 ) );
                        Green := Color15toGreen( Hues.GetColor( Hue, 14 ) );
                        Blue := Color15toBlue( Hues.GetColor( Hue, 14 ) );
                end;
        end;

        if (Buchstabe >= 32) and (Buchstabe < 256) then begin
                Buchstabe := Buchstabe - 32;
                Result.xOffset := Font.Buchstabe[ Buchstabe ].xOffset;
                Result.yOffset := Font.Buchstabe[ Buchstabe ].yOffset;
                Result.Hoehe := Font.Buchstabe[ Buchstabe ].Hoehe;
                Result.Breite := Font.Buchstabe[ Buchstabe ].Breite;
                if Font.Buchstabe[ Buchstabe ].Exists then begin
                        Result.Image := TByteCache.Create( Result.Breite, Result.Hoehe );
                        if Border then begin
                                for Y := 0 to Result.Hoehe-1 do begin
                                        for X := 0 to Result.Breite-1 do begin
                                                if Font.Buchstabe[ Buchstabe ].Image.Pixels[X,Y].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Red := Red;
                                                        Result.Image.Pixels[X,Y].Green := Green;
                                                        Result.Image.Pixels[X,Y].Blue := Blue;
                                                        Result.Image.Pixels[X,Y].Alpha := 255;
                                                end
                                                else if Font.Buchstabe[ Buchstabe ].Image.Pixels[X,Y].Alpha = 254 then begin
                                                        Result.Image.Pixels[X,Y].Red := 0;
                                                        Result.Image.Pixels[X,Y].Green := 0;
                                                        Result.Image.Pixels[X,Y].Blue := 0;
                                                        Result.Image.Pixels[X,Y].Alpha := 255;
                                                end
                                                else begin
                                                        //everything is already zero
                                                end;
                                        end;
                                end;
                        end else begin
                                for Y := 0 to Result.Hoehe-1 do begin
                                        for X := 0 to Result.Breite-1 do begin
                                                if Font.Buchstabe[ Buchstabe ].NonBorderImage.Pixels[X,Y].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Red := Red;
                                                        Result.Image.Pixels[X,Y].Green := Green;
                                                        Result.Image.Pixels[X,Y].Blue := Blue;
                                                        Result.Image.Pixels[X,Y].Alpha := 255;
                                                end
                                                else begin
                                                        //everything is already zero
                                                end;
                                        end;
                                end;
                        end;
                        Exit;
                end
                else begin
                        Exit;
                end;
        end
        else begin
                mulStream := Font.mulStream;
                mulStream.Seek( Buchstabe*4, soFromBeginning );
                mulStream.Read( Lookup, 4 );
                mulStream.Seek( Lookup, soFromBeginning );

                mulStream.Read( Result.xOffset, 1 );
                mulStream.Read( Result.yOffset, 1 );
                mulStream.Read( Result.Breite, 1 );
                mulStream.Read( Result.Hoehe, 1 );

                if (Result.Breite = 0) or (Result.Hoehe = 0) then begin
                        Result.Image := TByteCache.Create( 0, 0 );
                        exit;
                end;

                Result.Image := TByteCache.Create( Result.Breite+2, Result.Hoehe+2 );

                for Y := Result.Hoehe-1 downto 0 do begin
                        for X := 0 to ((Result.Breite-1) div 8 ) do begin
                                Font.mulStream.Read( P, 1 );
                                for J := 0 to 7 do begin
                                        if X*8+J < Result.Breite then begin
                                                if ( P and (1 shl (7-J))) = (1 shl (7-J)) then begin
                                                        Result.Image.Pixels[X*8+J+1,Y+1].Red := Red;
                                                        Result.Image.Pixels[X*8+J+1,Y+1].Green := Green;
                                                        Result.Image.Pixels[X*8+J+1,Y+1].Blue := Blue;
                                                        Result.Image.Pixels[X*8+J+1,Y+1].Alpha := 255;
                                                end;
                                        end;
                                end;
                        end;
                end;

                Result.Breite := Result.Breite+2;
                Result.Hoehe := Result.Hoehe+2;

                if Border then begin
                        for Y := 0 to Result.Hoehe-1 do begin
                                for X := 0 to Result.Breite-1 do begin
                                        if Result.Image.Pixels[ X, Y ].Alpha = 0 then begin
                                                if Result.Image.Pixels[X-1,Y-1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X,Y-1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X+1,Y-1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X+1,Y].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X+1,Y+1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X,Y+1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X-1,Y+1].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                                if Result.Image.Pixels[X-1,Y].Alpha = 255 then begin
                                                        Result.Image.Pixels[X,Y].Alpha := 254;
                                                        continue;
                                                end;
                                        end;
                                end;
                        end;
                end;
        end;
end;

function TUnicode.GetUnicodeTexture( FontNr : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word; SelPos : SmallInt; Border : Boolean; System : Boolean; DivHeight : Integer ) : TTexObject;
var     CurLine, Breite, Hoehe : Word;
        TmpBreite, TmpHoehe : Word;
        TmpText : String;
        I, J, K, PX, PY, X, Y : Integer;
        Count, CopyCount, Source, Target : Integer;
        Pixel : TByteCache;
        BImage : TUnicodeBuchstabenImage;
        Lines : Array of TLine;
        RBreite, RHoehe : Word;
        UsedLines : Word;
        LineCount : Word;
        MinHoehe : Word;
begin
        //Besser mal beschrnken
        MaxLines := Min( MaxLines, 100 );

        if (Text = '') and (SelPos = -1) then begin
                Result := nil;
                exit;
        end;

        if (MaxLines = 0) or (MaxBreite = 0) then begin
                Result := nil;
                exit;
        end;

        SetLength( Lines, MaxLines );
        Breite := 0;
        Hoehe := 0;

        TmpBreite := 0;
        TmpHoehe := 0;
        TmpText := '';
        CurLine := 0;

        if Border then
                MinHoehe := GetHoehe( FontNr, Ord( 'A' ), Border ) + 2
        else
                MinHoehe := GetHoehe( FontNr, Ord( 'A' ), Border ) + DivHeight;

        Lines[ CurLine ].Text := '';
        Lines[ CurLine ].Breite := 2;
        Lines[ CurLine ].Hoehe := MinHoehe;

        I := 1;
        while I <= Length( Text ) do begin
                if Text[ I ] = #10 then begin
                        if CurLine < MaxLines-1 then begin
                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                                Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite;
                                if Lines[ CurLine ].Hoehe < TmpHoehe then
                                        Lines[ CurLine ].Hoehe := TmpHoehe;

                                TmpText := '';
                                TmpBreite := 0;
                                TmpHoehe := MinHoehe;

                                if Lines[ CurLine ].Breite > Breite then
                                        Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                Inc( CurLine );
                                Lines[ CurLine ].Text := '';
                                Lines[ CurLine ].Breite := 2;
                                Lines[ CurLine ].Hoehe := MinHoehe;
                        end;
                end
                else if Text[ I ] = ' ' then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                Lines[ CurLine ].Hoehe := TmpHoehe;

                        TmpText := '';
                        TmpBreite := 0;
                        TmpHoehe := MinHoehe;

                        if Lines[ CurLine ].Breite + WhiteSpace <= MaxBreite then begin
                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + ' ';
                        end
                        else begin
                                if CurLine < MaxLines-1 then begin
                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                        Inc( CurLine );
                                        Lines[ CurLine ].Text := '';
                                        Lines[ CurLine ].Breite := 2;
                                        Lines[ CurLine ].Hoehe := MinHoehe;
                                end
                                else begin
                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                                        Break;
                                end;
                        end;
                end
                else begin
                        if Lines[ CurLine ].Breite + TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ), Border )-1 > MaxBreite then begin
                                if Lines[ CurLine ].Text = '' then begin
                                        Lines[ CurLine ].Text := TmpText;
                                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                Lines[ CurLine ].Hoehe := TmpHoehe;

                                        TmpText := '';
                                        TmpBreite := 0;
                                        TmpHoehe := MinHoehe;
                                end;

                                if CurLine < MaxLines-1 then begin
                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                        Inc( CurLine );
                                        Lines[ CurLine ].Text := '';
                                        Lines[ CurLine ].Breite := 2;
                                        Lines[ CurLine ].Hoehe := MinHoehe;
                                end
                                else begin
                                        if TmpText <> '' then begin
                                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                                                Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                                                if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                        Lines[ CurLine ].Hoehe := TmpHoehe;
                                        end;

                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                                        Break;
                                end;
                        end;
                        TmpText := TmpText + Text[ I ];
                        TmpBreite := TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ), Border )-1;
                        if TmpHoehe < GetHoehe( FontNr, Ord( Text[ I ] ), Border ) then
                                TmpHoehe := GetHoehe( FontNr, Ord( Text[ I ] ), Border );
                end;

                if I = Length( Text ) then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                Lines[ CurLine ].Hoehe := TmpHoehe;

                        if Lines[ CurLine ].Breite > Breite then
                                Breite := Lines[ CurLine ].Breite;
                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                end;

                Inc( I );
        end;

        UsedLines := CurLine;

        if Hoehe = 0 then
                Hoehe := GetHoehe( FontNr, Ord( 'A' ), Border );        //Fix for only _
        if (SelPos > 0) and (Length( Text ) > 0) and (Text[Length(Text)] = #10) then begin
                Hoehe := Hoehe + GetHoehe( FontNr, Ord( 'A' ), Border );
        end
        else if SelPos > Pos( #10, Text ) then begin
                Hoehe := Hoehe + 6;
        end;
        if SelPos > Length( Text ) then
                Breite := Breite + GetBreite( FontNr, Ord( '_' ), Border ) +1;

        if Breite = 0 then begin
                Result := nil;
                exit;
        end;

        RBreite := GetNextBit( Breite );
        RHoehe := GetNextBit( Hoehe );

        Pixel := TByteCache.Create( Breite, Hoehe );

        Y := 0;
        K := 0;
        X := 1;

        LineCount := 0;
        if Text <> '' then for J := 0 to UsedLines do begin
                if Lines[ J ].Text = '' then begin
                        Inc( LineCount );
                        continue;
                end
                else if (J > 0) then begin
                        Y := Y + LineCount*MinHoehe;
                        LineCount := 0;
                        Y := Y + Lines[ J-1 ].Hoehe;

                        if not Border then
                                Y := Y + DivHeight;
                end;
                
                X := 1;                        
                for I := 1 to Length( Lines[ J ].Text ) do begin
                        Inc( K );

                        if K = SelPos then begin
                                BImage := GetBuchstabe( FontNr, Ord( '_' ), Hue, -1, Border, System );

                                if BImage.Image = nil then begin
                                        BImage.Free;
                                        continue;
                                end;

                                for PY := 0 to BImage.Hoehe-1 do begin
                                        if Pixel.Pixels[X+BImage.xOffset,Y+4+BImage.yOffset+PY].Alpha = 0 then begin
                                                Pixel.CopyFrom( BImage.Image, X+BImage.xOffset, Y+4+BImage.yOffset+PY, 0, BImage.Hoehe-PY-1, BImage.Breite, 1 );
                                        end else begin
                                                Pixel.CopyFrom( BImage.Image, X+BImage.xOffset+1, Y+4+BImage.yOffset+PY, 1, BImage.Hoehe-PY-1, BImage.Breite-1, 1 );
                                        end;
                                end;
                                BImage.Free;
                        end;

                        if Lines[ J ].Text[ I ] = ' ' then begin
                                X := X + 3;
                                continue;
                        end;

                        BImage := GetBuchstabe( FontNr, Ord( Lines[ J ].Text[ I ] ), Hue, -1, Border, System );

                        if BImage.Image = nil then begin
                                BImage.Free;
                                continue;
                        end;

                        for PY := 0 to BImage.Hoehe-1 do begin
                                if Pixel.Pixels[X+BImage.xOffset,Y+BImage.yOffset+PY].Alpha = 0 then begin
                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset, Y+BImage.yOffset+PY, 0, BImage.Hoehe-PY-1, BImage.Breite, 1 );
                                end else begin
                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset+1, Y+BImage.yOffset+PY, 1, BImage.Hoehe-PY-1, BImage.Breite-1, 1 );
                                end;
                        end;
                        X := X + BImage.Breite + BImage.xOffset -1;
                        BImage.Free;
                end;
        end;

        if SelPos > Length( Text ) then begin
                Y := Y + LineCount*MinHoehe;
                if Border then
                        Y := Y + LineCount * DivHeight;
                LineCount := 0;
                
                BImage := GetBuchstabe( FontNr, Ord( '_' ), Hue, -1, Border, System );

                if BImage.Image <> nil then begin
                        for PY := 0 to BImage.Hoehe-1 do begin
                                if Pixel.Pixels[X+BImage.xOffset,Y+4+BImage.yOffset+PY].Alpha = 0 then begin
                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset, Y+4+BImage.yOffset+PY, 0, BImage.Hoehe-PY-1, BImage.Breite, 1 );
                                end else begin
                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset+1, Y+4+BImage.yOffset+PY, 1, BImage.Hoehe-PY-1, BImage.Breite-1, 1 );
                                end;
                        end;
                end;
                BImage.Free;
        end;

        Result := Renderer.CreateTexture32( Tex_Unicode, Breite, Hoehe, Pixel.GetData );
        Renderer.CurrentTexID := Result.TexID;
        Pixel.Free;
end;

function TUnicode.GetUnicodeLines( FontNr : Byte; Text : String; MaxBreite : Word; Border : Boolean ) : TStringList;
var     CurLine, Breite : Word;
        TmpBreite : Word;
        TmpText : String;
        I : Integer;
        Lines : Array of Record Text : String; Breite : Word; end;
begin
        Result := TStringList.Create;

        if Text = '' then begin
                exit;
        end;

        if MaxBreite = 0 then begin
                exit;
        end;

        SetLength( Lines, $FFFF );
        Breite := 0;

        TmpBreite := 0;
        TmpText := '';
        CurLine := 0;

        Lines[ CurLine ].Text := '';
        Lines[ CurLine ].Breite := 2;

        I := 1;
        while I <= Length( Text ) do begin
                if Text[ I ] = #10 then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite;

                        TmpText := '';
                        TmpBreite := 0;

                        if Lines[ CurLine ].Breite > Breite then
                                Breite := Lines[ CurLine ].Breite;

                        Inc( CurLine );
                        Lines[ CurLine ].Text := '';
                        Lines[ CurLine ].Breite := 2;
                end
                else if Text[ I ] = ' ' then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;

                        TmpText := '';
                        TmpBreite := 0;

                        if Lines[ CurLine ].Breite + WhiteSpace <= MaxBreite then begin
                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + ' ';
                        end
                        else begin
                                if Lines[ CurLine ].Breite > Breite then
                                        Breite := Lines[ CurLine ].Breite;

                                Inc( CurLine );
                                Lines[ CurLine ].Text := '';
                                Lines[ CurLine ].Breite := 2;
                        end;
                end
                else begin
                        if Lines[ CurLine ].Breite + TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ), Border )-1 > MaxBreite then begin
                                if Lines[ CurLine ].Text = '' then begin
                                        Lines[ CurLine ].Text := TmpText;
                                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;

                                        TmpText := '';
                                        TmpBreite := 0;
                                end;

                                if Lines[ CurLine ].Breite > Breite then
                                        Breite := Lines[ CurLine ].Breite;

                                Inc( CurLine );
                                Lines[ CurLine ].Text := '';
                                Lines[ CurLine ].Breite := 2;
                        end;
                        TmpText := TmpText + Text[ I ];
                        TmpBreite := TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ), Border )-1;
                end;

                if I = Length( Text ) then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;

                        if Lines[ CurLine ].Breite > Breite then
                                Breite := Lines[ CurLine ].Breite;
                end;

                Inc( I );
        end;

        while Lines[ CurLine ].Text = '' do begin
                Dec( CurLine );
        end;

        for I := 0 to CurLine do begin
                Result.Add( Lines[ I ].Text );
        end;
end;

function TUnicode.GetUnicodeHtmlTexture( FontNr : Byte; Text : String; MaxBreite : Word; Color : Word ) : TTexObject;
var     CurLine, Breite, Hoehe : Word;
        TmpBreite, TmpHoehe, MaxLines : Word;
        UnderLine : Boolean;
        TmpText : String;
        I, J, PX, PY, X, Y, P : Integer;
        Count, CopyCount, Source, Target : Integer;
        Pixel : TByteCache;
        BImage : TUnicodeBuchstabenImage;
        RealColor : Integer;
        Lines : Array of TLine;
        Border : Boolean;
        RBreite, RHoehe : Word;
begin
        if Text = '' then begin
                Result := nil;
                exit;
        end;

        if MaxBreite = 0 then begin
                Result := nil;
                exit;
        end;

        Border := True;

        MaxLines := $FFFF;

        SetLength( Lines, MaxLines );
        Breite := 0;
        Hoehe := 0;

        TmpBreite := 0;
        TmpHoehe := 0;
        TmpText := '';
        CurLine := 0;

        Lines[ CurLine ].Text := '';
        Lines[ CurLine ].Breite := 2;
        Lines[ CurLine ].Hoehe := 15;

        I := 1;
        while I <= Length( Text ) do begin
                if Text[ I ] = '<' then begin
                        P := StrLib.PosAfter( '>', Text, I );
                        if P > 0 then begin
                                if (UpperCase(Copy( Text, I, 9 )) = '<BASEFONT') or
                                   (UpperCase(Copy( Text, I, 10 )) = '</BASEFONT') or
                                   (UpperCase(Copy( Text, I, 3 )) = '<U>') or
                                   (UpperCase(Copy( Text, I, 4 )) = '</U>') or
                                   (UpperCase(Copy( Text, I, 7 )) = '<CENTER') or
                                   (UpperCase(Copy( Text, I, 8 )) = '</CENTER') then begin
                                        TmpText := TmpText + Copy( Text, I, P+1-I );
                                        I := P+1;

                                        if I > Length( Text ) then begin
                                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                                                Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite;
                                                if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                        Lines[ CurLine ].Hoehe := TmpHoehe;

                                                if Lines[ CurLine ].Breite > Breite then
                                                        Breite := Lines[ CurLine ].Breite;
                                                Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                                        end;
                                        continue;
                                end
                                else if (UpperCase(Copy( Text, I, 4 )) = '<BR>') then begin
                                        if CurLine < MaxLines-1 then begin
                                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                                                Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite;
                                                if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                        Lines[ CurLine ].Hoehe := TmpHoehe;

                                                TmpText := '';
                                                TmpBreite := 0;
                                                TmpHoehe := 15;

                                                if Lines[ CurLine ].Breite > Breite then
                                                        Breite := Lines[ CurLine ].Breite;
                                                Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                                Inc( CurLine );
                                                Lines[ CurLine ].Text := '';
                                                Lines[ CurLine ].Breite := 2;
                                                Lines[ CurLine ].Hoehe := 15;
                                        end;
                                        I := P+1;
                                        continue;
                                end;
                        end;
                end;
                if Text[ I ] = ' ' then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                Lines[ CurLine ].Hoehe := TmpHoehe;

                        TmpText := '';
                        TmpBreite := 0;
                        TmpHoehe := 0;

                        if Lines[ CurLine ].Breite + WhiteSpace <= MaxBreite then begin
                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + ' ';
                        end
                        else begin
                                if CurLine < MaxLines-1 then begin
                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                        Inc( CurLine );
                                        Lines[ CurLine ].Text := '';
                                        Lines[ CurLine ].Breite := 2;
                                        Lines[ CurLine ].Hoehe := 15;
                                end
                                else begin
                                        Break;
                                end;
                        end;
                end
                else begin
                        if Lines[ CurLine ].Breite + TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ) )-1 > MaxBreite then begin
                                if Lines[ CurLine ].Text = '' then begin
                                        Lines[ CurLine ].Text := TmpText;
                                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                Lines[ CurLine ].Hoehe := TmpHoehe;

                                        TmpText := '';
                                        TmpBreite := 0;
                                        TmpHoehe := 15;
                                end;

                                if CurLine < MaxLines-1 then begin
                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;

                                        Inc( CurLine );
                                        Lines[ CurLine ].Text := '';
                                        Lines[ CurLine ].Breite := 2;
                                        Lines[ CurLine ].Hoehe := 15;
                                end
                                else begin
                                        if TmpText <> '' then begin
                                                Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                                                Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                                                if Lines[ CurLine ].Hoehe < TmpHoehe then
                                                        Lines[ CurLine ].Hoehe := TmpHoehe;
                                        end;

                                        if Lines[ CurLine ].Breite > Breite then
                                                Breite := Lines[ CurLine ].Breite;
                                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                                        Break;
                                end;
                        end;
                        TmpText := TmpText + Text[ I ];
                        TmpBreite := TmpBreite + GetBreite( FontNr, Ord( Text[ I ] ) )-1;
                        if TmpHoehe < GetHoehe( FontNr, Ord( Text[ I ] ) ) then
                                TmpHoehe := GetHoehe( FontNr, Ord( Text[ I ] ) );
                end;

                if I = Length( Text ) then begin
                        Lines[ CurLine ].Text := Lines[ CurLine ].Text + TmpText;
                        Lines[ CurLine ].Breite := Lines[ CurLine ].Breite + TmpBreite + WhiteSpace;
                        if Lines[ CurLine ].Hoehe < TmpHoehe then
                                Lines[ CurLine ].Hoehe := TmpHoehe;
                        
                        if Lines[ CurLine ].Breite > Breite then
                                Breite := Lines[ CurLine ].Breite;
                        Hoehe := Hoehe + Lines[ CurLine ].Hoehe;
                end;

                Inc( I );
        end;

        Hoehe := Hoehe + 3;

        RBreite := GetNextBit( Breite );
        RHoehe := GetNextBit( Hoehe );

        Pixel := TByteCache.Create( Breite, Hoehe );

        Y := 0;
        if Color > 0 then begin
                RealColor := ( Color15toRed( Color ) shl 16 ) + ( Color15toGreen( Color ) shl 8 ) + Color15toBlue( Color );
        end
        else begin
                RealColor := $00FFFFFF;
        end;
        UnderLine := False;        

        for J := 0 to Length( Lines )-1 do begin
                X := 0;
                I := 1;
                if Lines[ J ].Text = '' then begin
                        Y := Y + Lines[ J ].Hoehe;
                        continue;
                end;
                while I <= Length( Lines[ J ].Text ) do begin
                        while Lines[ J ].Text[ I ] = '<' do begin
                                P := StrLib.PosAfter( '>', Lines[ J ].Text, I );
                                if P > 0 then begin
                                        if (UpperCase(Copy( Lines[ J ].Text, I, 17 )) = '<BASEFONT COLOR=#') then begin
                                                RealColor := StrToInt( '0x' + Copy( Lines[ J ].Text, I+17, 6 ) );
                                                I := P+1;
                                        end
                                        else if (UpperCase(Copy( Lines[ J ].Text, I, 10 )) = '</BASEFONT') then begin
                                                RealColor := -1;
                                                I := P+1;
                                        end
                                        else if (UpperCase(Copy( Lines[ J ].Text, I, 3 )) = '<U>') then begin
                                                UnderLine := True;
                                                I := P+1;
                                        end
                                        else if (UpperCase(Copy( Lines[ J ].Text, I, 4 )) = '</U>') then begin
                                                UnderLine := False;
                                                I := P+1;
                                        end
                                        else if (UpperCase(Copy( Lines[ J ].Text, I, 7 )) = '<CENTER') then begin
                                                I := P+1;
                                        end
                                        else if (UpperCase(Copy( Lines[ J ].Text, I, 8 )) = '</CENTER') then begin
                                                I := P+1;
                                        end
                                        else
                                                break;
                                end
                                else
                                        break;
                        end;

                        if Lines[ J ].Text[ I ] = ' ' then begin
                               if UnderLine then begin
                                        for P := X to X+3 do begin
                                                if Color > 0 then begin
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Red := Color15toRed( Color );
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Green := Color15toGreen( Color );
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Blue := Color15toBlue( Color );
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Alpha := 255;
                                                end
                                                else if RealColor >= 0 then begin
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Red := (RealColor shr 16) and $FF;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Green := (RealColor shr 8) and $FF;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Blue := RealColor and $FF;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Alpha := 254;
                                                end
                                                else begin
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Red := 0;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Green := 0;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Blue := 0;
                                                        Pixel.Pixels[P,Y+Lines[J].Hoehe-1].Alpha := 254;
                                                end;
                                        end;
                                end;

                                X := X + 3;
                        end
                        else begin
                                BImage := GetBuchstabe( FontNr, Ord( Lines[ J ].Text[ I ] ), 0, RealColor );

                                PX := X;

                                if BImage.Image <> nil then begin
                                        for PY := 0 to BImage.Hoehe-1 do begin
                                                if Pixel.Pixels[X+BImage.xOffset,Y+BImage.yOffset+PY].Alpha = 0 then begin
                                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset, Y+BImage.yOffset+PY, 0, BImage.Hoehe-PY-1, BImage.Breite, 1 );
                                                end else begin
                                                        Pixel.CopyFrom( BImage.Image, X+BImage.xOffset+1, Y+BImage.yOffset+PY, 1, BImage.Hoehe-PY-1, BImage.Breite-1, 1 );
                                                end;
                                        end;
                                        X := X + BImage.Breite + BImage.xOffset -1;
                                end;
                                BImage.Free;

                                if UnderLine then begin
                                        for PY := Y to Y+1 do begin
                                                for P := PX to X+1 do begin
                                                        if Color > 0 then begin
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Red := Color15toRed( Color );
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Green := Color15toGreen( Color );
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Blue := Color15toBlue( Color );
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Alpha := 255;
                                                        end
                                                        else if RealColor >= 0 then begin
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Red := (RealColor shr 16) and $FF;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Green := (RealColor shr 8) and $FF;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Blue := RealColor and $FF;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Alpha := 254;
                                                        end
                                                        else begin
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Red := 0;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Green := 0;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Blue := 0;
                                                                Pixel.Pixels[P,PY+Lines[J].Hoehe-1].Alpha := 254;
                                                        end;
                                                end;
                                        end;
                                end;
                        end;
                        Inc( I );
                end;
                if J < Length( Lines )-1 then
                        Y := Y + Lines[ J ].Hoehe;
        end;
        
        Result := Renderer.CreateTexture32( Tex_Unicode, Breite, Hoehe, Pixel.GetData );
        Renderer.CurrentTexID := Result.TexID;
        Pixel.Free;
end;

end.
