unit uPacket;

interface

uses    SysUtils, Dialogs, uLog, Classes, Math, uCounter;

const   PacketLengths: array[ 0..255 ] of Word =
              ( 104, 5, 7, 0, 2, 5, 5, 7, 14, 5, 0, 0, 0, 0, 0, 0,      //0
                0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 37, 0, 5, 0, 0,       //1
                19, 8, 3, 26, 7, 20, 5, 2, 5, 1, 0, 0, 2, 17, 15, 10,   //2
                0, 0, 0, 2, 10, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0,        //3
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 2,         //4
                0, 0, 0, 2, 12, 1, 11, 0, 0, 0, 0, 4, 0, 73, 0, 0,      //5
                0, 0, 0, 0, 0, 4, 0, 0, 0, 5, 0, 0, 19, 3, 14, 0,       //6
                28, 0, 5, 2, 0, 35, 16, 17, 0, 0, 0, 0, 0, 13, 0, 0,    //7
                62, 0, 2, 39, 0, 2, 304, 0, 66, 0, 0, 0, 11, 0, 0, 0,   //8
                19, 65, 0, 99, 0, 9, 0, 2, 0, 26, 16, 258, 0, 0, 0, 0,  //9
                3, 9, 9, 9, 149, 0, 0, 4, 0, 0, 5, 0, 0, 0, 0, 13,      //A
                0, 0, 0, 0, 0, 64, 9, 0, 0, 3, 6, 9, 3, 0, 0, 0,        //B
                36, 0, 0, 0, 6, 0, 0, 49, 2, 6, 6, 7, 0, 0, 0, 0,       //C
                0, 2, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0,        //D
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,         //E
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 );       //F

type    UOPacket = class
                private
                        PacketLength : Word;
                public
                        RawPacket : PByteArray;                 
                        ForceUnencrypted : Boolean;
                        EncryptedPacket : PByteArray;
                        constructor Create( Data : PByteArray = nil ); overload;
                        constructor Create( PacketID : Byte; Length : Word ); overload;
                        destructor Free;

                        function GetByte( Pos : Word ) : Byte;
                        procedure SetByte( Pos : Word; Value : Byte );

                        function getPacketType : Byte;

                        function GetWord( StartPos : Word ) : Word;
                        function GetLongWord( StartPos : Word ) : LongWord;
                        function GetAsciiString( StartPos, Length : Word ) : String;
                        function GetUnicodeString( StartPos, Length : Word ) : String;

                        procedure SetWord( StartPos, Value : Word );
                        procedure SetLongWord( StartPos : Word; Value : Longword );
                        procedure SetAsciiString( Text : String; StartPos, Laenge : Word );
                        procedure SetUnicodeString( Text : String; StartPos, Laenge : Word );

                        function getLength : Word;
                        procedure SetLength( Length : Word );

                        function Dump : TStringList; virtual;
                        function ToDump : Boolean; virtual;
                        function DumpEncrypted : TStringList;
                        procedure CopyRawToEncrypted;
                        function GetInternalName : String; virtual;
        end;

        P_UOPacket = ^UOPacket;

implementation

constructor UOPacket.Create( Data : PByteArray );
begin
        RawPacket := Data;
        if PacketLengths[ RawPacket^[ 0 ] ] <> 0 then
                PacketLength := PacketLengths[ RawPacket^[ 0 ] ]
        else
                PacketLength := RawPacket^[ 2 ] + ( RawPacket^[ 1 ] shl 8 );
        ForceUnencrypted := False;

        Counter.IncCount( Count_Packet );
end;

constructor UOPacket.Create( PacketID : Byte; Length : Word );
begin
        GetMem( RawPacket, Length );
        PacketLength := Length;
        SetByte( 0, PacketID );

        Counter.IncCount( Count_Packet );
end;

destructor UOPacket.Free;
begin
        Counter.DecCount( Count_Packet );

        FreeMem( RawPacket );
end;

function UOPacket.GetByte( Pos : Word ) : Byte;
begin
        if Pos >= PacketLength then begin
                Log.Write( Format( 'Buggy read Packet: Bytepos: %d', [Pos] ) );
                Result := 0;
        end
        else begin
                Result := RawPacket^[ Pos ];
        end;
end;

procedure UOPacket.SetByte( Pos : Word; Value : Byte );
begin
        if Pos >= PacketLength then begin
                Log.Write( Format( 'Buggy write Packet: Bytepos: %d', [Pos] ) );
        end
        else begin
                RawPacket^[ Pos ] := Value;
        end;
end;

function UOPacket.getPacketType : Byte;
begin
        Result := GetByte( 0 );
end;

function UOPacket.GetWord( StartPos : Word ) : Word;
begin
        if StartPos+1 >= PacketLength then begin
                Log.Write( Format( 'Buggy Packet: Word Startpos: %d', [StartPos] ) );
                Result := 0;
        end
        else begin
                Result := GetByte( StartPos+1 );
                Result := Result + ( GetByte( StartPos ) shl 8 );
        end;
end;

function UOPacket.GetLongWord( StartPos : Word ) : LongWord;
begin
        if StartPos+3 >= PacketLength then begin
                Log.Write( Format( 'Buggy Packet: LongWord Startpos: %d', [StartPos] ) );
                Result := 0;
        end
        else begin
                Result := GetByte( StartPos+3 );
                Result := Result + ( GetByte( StartPos+2 ) shl 8 );
                Result := Result + ( GetByte( StartPos+1 ) shl 16 );
                Result := Result + ( GetByte( StartPos ) shl 24 );
        end;
end;

function UOPacket.GetAsciiString( StartPos, Length : Word ) : String;
var     I : Integer;
begin
        if StartPos+Length-1 >= PacketLength then begin
                Log.Write( Format( 'Buggy Packet: AsciiString Startpos: %d  Lenght: %d', [StartPos,Length] ) );
                Result := '';
        end
        else begin
                Result := '';
                for I := StartPos to StartPos+Length-1 do
                        if GetByte( I ) = 0 then
                                break
                        else
                                Result := Result + Chr( GetByte( I ) );
                Result := Utf8Decode( Result );
        end;
end;

function UOPacket.GetUnicodeString( StartPos, Length : Word ) : String;
var     I : Integer;
        Value : Word;
begin
        Result := '';
        for I := 0 to (( Length-2 ) div 2) do begin
                Value := GetWord( StartPos + 2*I );
                if Value = 0 then
                        break
                else
                        Result := Result + WideChar( Value );
        end;
end;

procedure UOPacket.SetWord( StartPos, Value : Word );
begin
        if StartPos+1 >= PacketLength then begin
                Log.Write( Format( 'Buggy SetPacket: Word Startpos: %d', [StartPos] ) );
        end
        else begin
                SetByte( StartPos, ( Value and $FF00 ) shr 8 );
                SetByte(  StartPos+1, ( Value and $00FF ) );
        end;
end;

procedure UOPacket.SetLongWord( StartPos : Word; Value : LongWord );
begin
        if StartPos+3 >= PacketLength then begin
                Log.Write( Format( 'Buggy SetPacket: LongWord Startpos: %d', [StartPos] ) );
        end
        else begin
                SetByte( StartPos, ( Value and $FF000000 ) shr 24 );
                SetByte( StartPos+1, ( Value and $00FF0000 ) shr 16 );
                SetByte( StartPos+2, ( Value and $0000FF00 ) shr 8 );
                SetByte( StartPos+3, ( Value and $000000FF ) );
        end;
end;

procedure UOPacket.SetAsciiString( Text : String; StartPos, Laenge : Word );
var     I : Integer;
begin
        if StartPos+Laenge-1 >= PacketLength then begin
                Log.Write( Format( 'Buggy Packet: AsciiString Startpos: %d  Lenght: %d', [StartPos,Laenge] ) );
        end
        else begin
                Text := Utf8Encode( Text );
                for I := StartPos to StartPos+Laenge-1 do
                        if Length( Text ) < (I-StartPos+1) then
                                SetByte( I, 0 )
                        else
                                SetByte( I, Ord( Text[ I-StartPos+1 ] ) );
        end;
end;

procedure UOPacket.SetUnicodeString( Text : String; StartPos, Laenge : Word );
var     I : Integer;
        Value : Byte;
begin
        for I := 1 to Laenge do begin
                if Length( Text ) < I then
                        Value := 0
                else
                        Value := Ord( Text[ I ] );

                SetByte( StartPos + 2*(I-1), 0 );
                SetByte( StartPos + 2*(I-1)+1, Value );
        end;
end;

function UOPacket.getLength : Word;
begin
        Result := PacketLength;
end;

procedure UOPacket.SetLength( Length : Word );
var     Temp : PByteArray;
begin
        GetMem( Temp, Length );
        Move( RawPacket^[0], Temp^[0], Min( PacketLength, Length ) );
        FreeMem( RawPacket );
        GetMem( RawPacket, Length );
        Move( Temp^[0], RawPacket^[0], Length );
        FreeMem( Temp );
        PacketLength := Length;
end;

function UOPacket.Dump : TStringList;
var     Zeile, TextZeile : String;
        I : Integer;
begin
        Result := TStringList.Create;
        I := 0;
        while I < GetLength do begin
                if ( I mod 16 ) = 0 then begin
                        if Zeile <> '' then
                                Result.Add( Zeile + '     ' + TextZeile );

                        if GetByte( I ) < $10 then
                                Zeile := Format( '0%x', [GetByte( I )] )
                        else
                                Zeile := Format( '%x', [GetByte( I )] );
                        if (GetByte( I ) > 31) and (GetByte( I ) < 128) and (I > 0) then
                                TextZeile := Chr( GetByte( I ) )
                        else
                                TextZeile := '.';
                end
                else begin
                        if GetByte( I ) < $10 then
                                Zeile := Format( '%s 0%x', [Zeile, GetByte( I )] )
                        else
                                Zeile := Format( '%s %x', [Zeile, GetByte( I )] );
                        if (GetByte( I ) > 31) and (GetByte( I ) < 128) and (I > 0) then
                                TextZeile := TextZeile + Chr( GetByte( I ) )
                        else
                                TextZeile := TextZeile + '.';
                end;
                Inc( I );
        end;
        if Zeile <> '' then begin
                while ( I mod 16 ) <> 0 do begin
                        Zeile := Zeile + '   ';
                        Inc( I );
                end;
                Result.Add( Zeile + '     ' + TextZeile );
        end;
end;

function UOPacket.ToDump : Boolean;
begin
        Result := True;
end;

function UOPacket.DumpEncrypted : TStringList;
var     Zeile, TextZeile : String;
        I : Integer;
begin
        Result := TStringList.Create;
        I := 0;
        while I < GetLength do begin
                if ( I mod 16 ) = 0 then begin
                        if Zeile <> '' then
                                Result.Add( Zeile + '     ' + TextZeile );

                        if EncryptedPacket^[I] < $10 then
                                Zeile := Format( '0%x', [EncryptedPacket^[I]] )
                        else
                                Zeile := Format( '%x', [EncryptedPacket^[I]] );
                        if (EncryptedPacket^[I] > 31) and (EncryptedPacket^[I] < 128) and (I > 0) then
                                TextZeile := Chr( EncryptedPacket^[I] )
                        else
                                TextZeile := '.';
                end
                else begin
                        if EncryptedPacket^[I] < $10 then
                                Zeile := Format( '%s 0%x', [Zeile, EncryptedPacket^[I]] )
                        else
                                Zeile := Format( '%s %x', [Zeile, EncryptedPacket^[I]] );
                        if (EncryptedPacket^[I] > 31) and (EncryptedPacket^[I] < 128) and (I > 0) then
                                TextZeile := TextZeile + Chr( EncryptedPacket^[I] )
                        else
                                TextZeile := TextZeile + '.';
                end;
                Inc( I );
        end;
        if Zeile <> '' then begin
                while ( I mod 16 ) <> 0 do begin
                        Zeile := Zeile + '   ';
                        Inc( I );
                end;
                Result.Add( Zeile + '     ' + TextZeile );
        end;
end;

procedure UOPacket.CopyRawToEncrypted;
begin
        GetMem( EncryptedPacket, PacketLength );
        Move( RawPacket^[0], EncryptedPacket^[0], PacketLength );
end;

function UOPacket.GetInternalName : String;
begin
        Result := Format( 'Packettype: 0x%x  Length: %d', [GetPacketType, GetLength] );
end;

end.
