unit uByteCache;

interface

uses    uUtilities, SysUtils, Math;

type    TRGBAColor = Record
                Red, Green, Blue, Alpha : Byte;
        end;

        PRGBAColor = ^TRGBAColor;

        TByteCache = Class
                private
                        Data : PByteArray;
                        _DummyPixel : TRGBAColor;
                        _Width, _Height : Integer;
                        _RealWidth, _RealHeight : Integer;
                        function GetPixel( X, Y : Integer ) : PRGBAColor;
                public
                        property Width : Integer read _Width;
                        property Height : Integer read _Height;
                        property RealWidth : Integer read _RealWidth;
                        property RealHeight : Integer read _RealHeight;
                        property Pixels[ X, Y : Integer ] : PRGBAColor read GetPixel;
                        constructor Create( NewWidth, NewHeight : Integer );
                        destructor Free;
                        procedure Resize( NewWidth, NewHeight : Integer );
                        function CopyFrom( Src : TByteCache; TrgX, TrgY : Integer; SrcX : Integer = 0; SrcY : Integer = 0; Width : Integer = $7FFFFFFF; Height : Integer = $7FFFFFFF ) : Boolean;
                        function GetData : PByteArray;
        end;

implementation

function TByteCache.GetPixel( X, Y : Integer ) : PRGBAColor;
begin
        if (X >= 0) and (X < _Width) and (Y >= 0) and (Y < _Height) then begin
                Result := PRGBAColor(Pointer(Cardinal(Data) + (Y*_RealWidth + X)*4));
        end else begin
                _DummyPixel.Red := 0;
                _DummyPixel.Green := 0;
                _DummyPixel.Blue := 0;
                _DummyPixel.Alpha := 0;
                Result := @_DummyPixel;
        end;
end;

constructor TByteCache.Create( NewWidth, NewHeight : Integer );
var     I : Integer;
begin
        _Width := NewWidth;
        _Height := NewHeight;
        _RealWidth := GetNextBit( _Width );
        _RealHeight := GetNextBit( _Height );

        GetMem( Data, _RealWidth*_RealHeight*4 );
        for I := 0 to _RealWidth*_RealHeight*4-1 do begin
                Data^[I] := 0;
        end;
end;

destructor TByteCache.Free;
begin
        FreeMem( Data );
end;

procedure TByteCache.Resize( NewWidth, NewHeight : Integer );
var     NewData : PByteArray;
        NewRealWidth, NewRealHeight, CopyWidth, CopyHeight : Integer;
        I, Y : Integer;
begin
        NewRealWidth := GetNextBit( NewWidth );
        NewRealHeight := GetNextBit( NewHeight );

        GetMem( NewData, NewRealWidth*NewRealHeight*4 );
        for I := 0 to NewRealWidth*NewRealHeight*4-1 do begin
                NewData^[I] := 0;
        end;

        CopyWidth := Min( _Width, NewWidth );
        CopyHeight := Min( _Height, NewHeight );
        for Y := 0 to CopyHeight-1 do begin
                Move( Data^[_RealWidth*Y*4], NewData^[NewRealWidth*Y*4], CopyWidth*4 );
        end;

        _Width := NewWidth;
        _Height := NewHeight;
        _RealWidth := NewRealWidth;
        _RealHeight := NewRealHeight;

        FreeMem( Data );
        Data := NewData;
end;

function TByteCache.CopyFrom( Src : TByteCache; TrgX, TrgY : Integer; SrcX, SrcY, Width, Height : Integer ) : Boolean;
var     Y : Integer;
        CopyWidth, CopyHeight : Integer;
begin
        if (TrgX < 0) or (TrgX >= _Width) or (TrgY < 0) or (TrgY >= _Height) then begin
                Result := False;
                Exit;
        end;

        if (SrcX < 0) or (SrcX >= Src.Width) or (SrcY < 0) or (SrcY >= Src.Height) then begin
                Result := False;
                Exit;
        end;

        CopyWidth := Min( Min( Src.Width-SrcX, _Width-TrgX ), Width );
        CopyHeight := Min( Min( Src.Height-SrcY, _Height-TrgY ), Height );

        for Y := 0 to CopyHeight-1 do begin
                Move( Src.GetData^[(Src.RealWidth*(SrcY+Y)+SrcX)*4], Data^[(_RealWidth*(TrgY+Y)+TrgX)*4], CopyWidth*4 );
        end;

        Result := True;
end;

function TByteCache.GetData : PByteArray;
begin
        Result := Data;
end;

end.
