unit uData;

interface

uses    uLoaderGump, SysUtils, Dialogs, uLoaderHues, uUtilities, uLoaderFonts,
        uLoaderMap, uWorldCache, uTexCache, uLoaderTexturen, uLoaderArt,
        uLoaderTiledata, uObjectListe, uLoaderAnimdata, uLoaderAnimations,
        uCursors, uLoaderUnicodeFonts, uLoaderCliloc, uLoaderSounds, SDL_Mixer,
        uLoaderMP3, uLoaderLight, uLoaderRadarcol, uLoaderSpeech, Classes,
        uLoaderMultis, uLoaderVerdata, uLog, uMulCrypt, uPixelBuffer,
        uBasicTypedefs, uLoaderIntloc, uLoaderSkills, uLoaderProgress;

type    TData = Class
                private
                        Art : TArt;
                        Gumps : TGumps;
                        Fonts : TFonts;
                        Maps : TMaps;
                        Texturen : TTexturen;
                        Animations : TAnimations;
                        Unicode : TUnicode;
                        Cliloc : TCliloc;
                        Intloc : TIntloc;
                        Sounds : TSounds;
                        Music : TMusic;
                        Light : TLights;
                        Radarcol : TRadarcol;
                        SpeechLoader : TLoaderSpeech;
                        Multis : TMultis;
                        Skills : TSkills;
                public
                        WorldCache : TWorldCache;
                        GlobalObjectList : TObjectCache;
                        MulCrypt : TMulCrypt;
                        Tiledata : TTiledata;
                        Animdata : TAnimdata;
                        Verdata : TVerdata;
                        Progress : TProgress;                        
                        constructor Create;
                        destructor Free;
                        procedure Load( baseDir, shardDir : String );
                        procedure CheckBaseDir( basedir : String );
                        procedure CheckFile( FileName : String );

                        //Global
                        procedure Clear;
                        procedure Poll;

                        //Animations
                        function GetAnimTexture( BaseId : LongWord; Anim : Word; Hue : Word = 0; Frame : Word = 0; Grayify : Boolean = False ) : TAnimObject;
                        function GetAnimSitDownFrameObject( BaseId : LongWord; Hue : Word = 0; Grayify : Boolean = False ) : TAnimObject;
                        function GetCorpseID( Id, Nr : Word ) : Word;
                        function GetCorpseTexture( BaseId : LongWord; Anim : Word; Hue : Word = 0; Frame : Word = 0; Grayify : Boolean = False ) : TAnimObject;
                        procedure IncAnimCache( BaseId : LongWord; Anim : Word; Hue : Word = 0; Frame : Word = 0; Grayify : Boolean = False );
                        procedure DecAnimCache( BaseId : LongWord; Anim : Word; Hue : Word = 0; Frame : Word = 0; Grayify : Boolean = False );

                        //Texturen
                        function GetTextureTexture( Id : Word; Hue : Word = 0 ) : TTexObject;
                        procedure IncTextureCache( Id : Word; Hue : Word = 0 );
                        procedure DecTextureCache( Id : Word; Hue : Word = 0 );

                        //Gump
                        function GetGumpTexture( Id : Word; Hue : Word = 0 ) : TTexObject;
                        function GetRadarcolTexture( Id : Word ) : TTexObject;
                        procedure IncGumpCache( Id : Word; Hue : Word = 0 );
                        procedure DecGumpCache( Id : Word; Hue : Word = 0 );

                        //Art
                        function GetStaticArt( Id : Word; Hue : Word = 0; Immediately : Boolean = False ) : TTexObject;
                        function LoadCursor( Id : Word; Hue : Word = 0 ) : TCustomCursor;
                        function GetMapArt( Id : Word; Hue : Word = 0; Immediately : Boolean = False ) : TTexObject;
                        procedure IncMapArt( Id : Word; Hue : Word = 0 );
                        procedure DecMapArt( Id : Word; Hue : Word = 0 );
                        procedure IncStaticArt( Id : Word; Hue : Word = 0 );
                        procedure DecStaticArt( Id : Word; Hue : Word = 0 );
                        function GetStaticArtWidth( Id : LongWord ) : Word;
                        function GetStaticArtHeight( Id : LongWord ) : Word;

                        //Fonts
                        function GetAsciiTexture( Id : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word = $FFFF; DivHeight : Integer = 0 ) : TTexObject;
                        function GetFontMaxHoehe( Id : Byte ) : Byte;

                        //UnicodeFonts
                        function GetBuchstabenBreite( FontNr : Byte; Buchstabe : Word ) : Byte;
                        function GetUnicodeTexture( Id : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word = $FFFF; SelPos : SmallInt = -1; Border : Boolean = True; System : Boolean = False; DivHeight : Integer = 0 ) : TTexObject;
                        function GetUnicodeLines( FontNr : Byte; Text : String; MaxBreite : Word; Border : Boolean ) : TStringList;
                        function GetUnicodeHtmlTexture( Id : Byte; Text : String; MaxBreite : Word; Color : Word = 0 ) : TTexObject;                        

                        //Map
                        procedure LoadMapBlock( Block : TLoadBlock );
                        function GetMapHeight( X, Y : Word ) : ShortInt;
                        function GetMapHoehe : Word;
                        function GetMapBreite : Word;
                        function GetRadarTexture( BlockID : LongWord ) : TTexObject;
                        function GetWorldRadar : TTexObject;
                        function InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;

                        //Light
                        function GetLight( Id : Word; Hue : Word = 0 ) : TTexObject;
                        procedure IncLight( Id : Word; Hue : Word = 0 );
                        procedure DecLight( Id : Word; Hue : Word = 0 );

                        //Cliloc
                        function GetClilocMessage( Id : LongWord ) : String;

                        //Intloc
                        function GetIntlocMessage( FileNr : Byte; Id : Word ) : String;

                        //Sound
                        function GetSound( Id : Word ) : PMix_Chunk;

                        //Music
                        function GetMusic( Id : Word ) : PMusicRecord;

                        //Radarcol
                        function GetMapRadarColor( Id : Word ) : Word;
                        function GetStaticRadarColor( Id : Word ) : Word;

                        //Speech
                        function GetKeyWords( Zeile : String ) : TList;

                        //Multis
                        function GetMultiItems( Id : Word ) : TList;
                        function GetMultiAsPixelBuffer( Id : Word; Color : Word = 0 ) : TPixelBuffer;

                        //Skills
                        function GetSkillName( Id : Word ) : String;
                        function SkillHasButton( Id : Word ) : Boolean;
                        function GetSkillGroup( Id : Word ) : Byte;
                        function GetSkillGroupName( Id : Byte ) : String;
                        procedure SetSkillGroup( Id : Word; Group : Byte );
                        procedure SetSkillGroupName( Id : Byte; Name : String );
                        function AddSkillGroup( Name : String ) : Byte;
                        function GetSkillGroupCount : Byte;
                        function GetSkillCount : Byte;                        
        end;

implementation

uses    uPalanthir;

constructor TData.Create;
begin
        WorldCache := TWorldCache.Create;
        GlobalObjectList := TObjectCache.Create;
        MulCrypt := TMulCrypt.Create( Palanthir.Shard.ShardConfig.GetString( 'APLogin' ), DecryptPassword( Palanthir.Shard.ShardConfig.GetString( 'APPassword' ) ), 1 );

        if Palanthir.EnabledME and (not Palanthir.MulEncryption) then begin
                ShowMessage( 'Starte die Client.exe neu. Wenn diese Nachricht ein zweites Mal erscheint, stimmt etwas mit deiner Client-Konfiguration nicht.' );
                Halt;
        end;

        if Palanthir.MulEncryption and (not MulCrypt.Test( Random( 3 ) )) then begin
                ShowMessage( 'Falsches Login oder Passwort oder noch kein freigeschalteter Char vorhanden.' );
                Halt;
        end;
end;

destructor TData.Free;
begin
        GlobalObjectList.Free;
        WorldCache.Free;
        if Gumps <> nil then
                Gumps.Free;
        if Fonts <> nil then
                Fonts.Free;
        if Unicode <> nil then
                Unicode.Free;
        if Art <> nil then
                Art.Free;
        if Texturen <> nil then
                Texturen.Free;
        if Maps <> nil then
                Maps.Free;
        if Tiledata <> nil then
                Tiledata.Free;
        if Animdata <> nil then
                Animdata.Free;
        if Animations <> nil then
                Animations.Free;
        if Cliloc <> nil then
                Cliloc.Free;
        if Intloc <> nil then
                Intloc.Free;
        if Sounds <> nil then
                Sounds.Free;
        if Music <> nil then
                Music.Free;
        if Light <> nil then
                Light.Free;
        if Radarcol <> nil then
                Radarcol.Free;
        if SpeechLoader <> nil then
                SpeechLoader.Free;
        if Multis <> nil then
                Multis.Free;
        if Skills <> nil then
                Skills.Free;
        if Verdata <> nil then
                Verdata.Free;
        if Progress <> nil then
                Progress.Free;
        uLoaderHues.Hues.Free;
        if MulCrypt <> nil then
                MulCrypt.Free;        
end;

procedure TData.Clear;
begin
        GlobalObjectList.Free;
        WorldCache.Free;

        WorldCache := TWorldCache.Create;
        GlobalObjectList := TObjectCache.Create;
end;

procedure TData.Poll;
begin
        WorldCache.Poll;
        Art.Poll;
        Animations.Poll;
end;

procedure TData.Load( baseDir, shardDir : String );
begin
        if baseDir = shardDir then begin
                Log.Write( 'Ultima Online directory and Data directory of this shard have to be different.', True );
                Halt;        
        end;

        CheckBaseDir( baseDir );

        if not DirectoryExists( shardDir ) then
                ForceDirectories( shardDir );

        Log.Write( 'Loading Verdata' );
        Verdata := TVerdata.Create( baseDir, shardDir );
        Log.Write( 'Loading Animations' );
        Animations := TAnimations.Create( baseDir, shardDir );
        Log.Write( 'Loading Gumps' );
        Gumps := TGumps.Create( baseDir, shardDir );
        Log.Write( 'Loading Textures' );
        Texturen := TTexturen.Create( baseDir, shardDir );
        Log.Write( 'Loading Art' );
        Art := TArt.Create( baseDir, shardDir );
        Log.Write( 'Loading Hues' );
        uLoaderHues.Hues := THues.Create( baseDir, shardDir );
        Log.Write( 'Loading Fonts' );
        Fonts := TFonts.Create( baseDir, shardDir );
        Log.Write( 'Loading Unicodefonts' );
        Unicode := TUnicode.Create( baseDir, shardDir );
        Log.Write( 'Loading Animdata' );
        Animdata := TAnimdata.Create( baseDir, shardDir );
        Log.Write( 'Loading Tiledata' );
        Tiledata := TTiledata.Create( baseDir, shardDir );
        Log.Write( 'Loading Cliloc' );
        Cliloc := TCliloc.Create( baseDir, shardDir );
        Log.Write( 'Loading Intloc' );
        Intloc := TIntloc.Create( baseDir, shardDir );
        Log.Write( 'Loading Sound' );
        Sounds := TSounds.Create( baseDir, shardDir );
        Log.Write( 'Loading Music' );
        Music := TMusic.Create( baseDir, shardDir );
        Log.Write( 'Loading Light' );
        Light := TLights.Create( baseDir, shardDir );
        Log.Write( 'Loading Radarcol' );
        Radarcol := TRadarcol.Create( baseDir, shardDir );
        Log.Write( 'Loading Multis' );
        Multis := TMultis.Create( baseDir, shardDir );
        Log.Write( 'Loading Speech' );
        SpeechLoader := TLoaderSpeech.Create( baseDir, shardDir );
        Log.Write( 'Loading Maps' );
        Maps := TMaps.Create( baseDir, shardDir );
        Log.Write( 'Loading Skills' );
        Skills := TSkills.Create( baseDir, shardDir );
        Log.Write( 'Loading Progress' );
        Progress := TProgress.Create( shardDir );       

        WorldCache.Free;
        WorldCache := TWorldCache.Create;
end;

procedure TData.CheckBaseDir( basedir : String );
begin
        CheckFile( basedir + 'anim.idx' );
        CheckFile( basedir + 'anim.mul' );
end;

procedure TData.CheckFile( FileName : String );
var     FileStream : TFileStream;
begin
        if not FileExists( FileName ) then begin
                Log.Write( Format( 'File "%s" does not exist.', [FileName] ), True );
                Halt;
        end
        else begin
                try
                        FileStream := TFileStream.Create( FileName, fmOpenRead );
                        FileStream.Free;
                except
                        Log.Write( Format( 'Not able to open File "%s". Maybe another process is using it.', [FileName] ), True );
                        Halt;
                end;
        end;
end;

function TData.GetAnimTexture( BaseId : LongWord; Anim : Word; Hue : Word; Frame : Word; Grayify : Boolean ) : TAnimObject;
begin
        Result := Animations.GetAnimFrameObject( BaseId, Anim, Hue, Frame, Grayify );
end;

function TData.GetAnimSitDownFrameObject( BaseId : LongWord; Hue : Word; Grayify : Boolean ) : TAnimObject;
begin
        Result := Animations.GetAnimSitDownFrameObject( BaseId, Hue, Grayify );
end;

function TData.GetCorpseID( Id, Nr : Word ) : Word;
begin
        Result := Animations.GetCorpseID( Id, Nr );
end;

function TData.GetCorpseTexture( BaseId : LongWord; Anim : Word; Hue : Word = 0; Frame : Word = 0; Grayify : Boolean = False ) : TAnimObject;
begin
        Result := Animations.GetAnimFrameObject( BaseId, Anim, Hue, Frame, Grayify );
end;

procedure TData.IncAnimCache( BaseId : LongWord; Anim : Word; Hue : Word; Frame : Word; Grayify : Boolean );
begin
        Animations.IncCache( BaseId, Anim, Hue, Frame, Grayify );
end;

procedure TData.DecAnimCache( BaseId : LongWord; Anim : Word; Hue : Word; Frame : Word; Grayify : Boolean );
begin
        Animations.DecCache( BaseId, Anim, Hue, Frame, Grayify );
end;

function TData.GetTextureTexture( Id : Word; Hue : Word ) : TTexObject;
begin
        Result := Texturen.GetTexture( Id, Hue );
end;

procedure TData.IncTextureCache( Id : Word; Hue : Word );
begin
        Texturen.IncTexture( Id, Hue );
end;

procedure TData.DecTextureCache( Id : Word; Hue : Word );
begin
        Texturen.DecTexture( Id, Hue );
end;

function TData.GetGumpTexture( Id : Word; Hue : Word ) : TTexObject;
begin
        Result := Gumps.GetGumpObject( Id, Hue );
end;

function TData.GetRadarcolTexture( Id : Word ) : TTexObject;
begin
        Result := Gumps.GetRadarcolTexture( Id );
end;

procedure TData.IncGumpCache( Id : Word; Hue : Word );
begin
        Gumps.IncCache( Id, Hue );
end;

procedure TData.DecGumpCache( Id : Word; Hue : Word );
begin
        Gumps.DecCache( Id, Hue );
end;

function TData.GetAsciiTexture( Id : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word = $FFFF; DivHeight : Integer = 0 ) : TTexObject;
begin
        Result := Fonts.GetAsciiTexture( Id, Text, Hue, MaxBreite, MaxLines, DivHeight );
end;

function TData.GetFontMaxHoehe( Id : Byte ) : Byte;
begin
        Result := Fonts.GetFontMaxHoehe( Id );
end;

function TData.GetBuchstabenBreite( FontNr : Byte; Buchstabe : Word ) : Byte;
begin
        Result := Unicode.GetBreite( FontNr, Buchstabe );
end;

function TData.GetUnicodeTexture( Id : Byte; Text : String; Hue : Word; MaxBreite : Word; MaxLines : Word; SelPos : SmallInt; Border, System : Boolean; DivHeight : Integer ) : TTexObject;
begin
        Result := Unicode.GetUnicodeTexture( Id, Text, Hue, MaxBreite, MaxLines, SelPos, Border, System, DivHeight );
end;

function TData.GetUnicodeLines( FontNr : Byte; Text : String; MaxBreite : Word; Border : Boolean ) : TStringList;
begin
        Result := Unicode.GetUnicodeLines( FontNr, Text, MaxBreite, Border );
end;

function TData.GetUnicodeHtmlTexture( Id : Byte; Text : String; MaxBreite : Word; Color : Word ) : TTexObject;
begin
        Result := Unicode.GetUnicodeHtmlTexture( Id, Text, MaxBreite, Color );
end;

function TData.GetStaticArt( Id, Hue : Word; Immediately : Boolean ) : TTexObject;
begin
        Result := Art.GetStaticArt( Id, Hue, Immediately );
end;

function TData.LoadCursor( Id, Hue : Word ) : TCustomCursor;
begin
        Result := Art.LoadCursor( Id, Hue );
end;

function TData.GetMapArt( Id, Hue : Word; Immediately : Boolean ) : TTexObject;
begin
        Result := Art.GetMapArt( Id, Hue, Immediately );
end;

procedure TData.IncMapArt( Id : Word; Hue : Word );
begin
        Art.IncMapArt( Id, Hue );
end;

procedure TData.DecMapArt( Id : Word; Hue : Word );
begin
        Art.DecMapArt( Id, Hue );
end;

procedure TData.IncStaticArt( Id : Word; Hue : Word );
begin
        Art.IncStaticArt( Id, Hue );
end;

procedure TData.DecStaticArt( Id : Word; Hue : Word );
begin
        Art.DecStaticArt( Id, Hue );
end;

function TData.GetStaticArtWidth( Id : LongWord ) : Word;
begin
        Result := Art.GetStaticArtWidth( Id );
end;

function TData.GetStaticArtHeight( Id : LongWord ) : Word;
begin
        Result := Art.GetStaticArtHeight( Id );
end;

procedure TData.LoadMapBlock( Block : TLoadBlock );
begin
        Maps.LoadBlock( Block );
end;

function TData.GetMapHeight( X, Y : Word ) : ShortInt;
begin
        Result := Maps.GetMapHeight( X, Y );
end;

function TData.GetMapHoehe : Word;
begin
        Result := Maps.GetHeight;
end;

function TData.GetMapBreite : Word;
begin
        Result := Maps.GetWidth;
end;

function TData.GetRadarTexture( BlockID : LongWord ) : TTexObject;
begin
        Result := Maps.GetRadarTexture( BlockID );
end;

function TData.GetWorldRadar : TTexObject;
begin
        Result := Maps.GetWorldRadar;
end;

function TData.InitMap( BlockID : LongWord; MapBlock : TMapBlock ) : PMapListe;
begin
        Result := Maps.InitMap( BlockID, MapBlock );
end;

function TData.GetClilocMessage( Id : LongWord ) : String;
begin
        Result := Cliloc.GetClilocMessage( Id );
end;

function TData.GetIntlocMessage( FileNr : Byte; Id : Word ) : String;
begin
        Result := Intloc.GetMessage( FileNr, Id );

        if Result = '' then begin
                Result := GetClilocMessage( 3000000 + FileNr*1000 + Id );
        end;
end;

function TData.GetSound( Id : Word ) : PMix_Chunk;
begin
        Result := Sounds.GetSound( Id );
end;

function TData.GetMusic( Id : Word ) : PMusicRecord;
begin
        Result := Music.GetMusic( Id );
end;

function TData.GetLight( Id : Word; Hue : Word ) : TTexObject;
begin
        Result := Light.GetLight( Id, Hue );
end;

procedure TData.IncLight( Id : Word; Hue : Word );
begin
        Light.IncLight( Id, Hue );
end;

procedure TData.DecLight( Id : Word; Hue : Word );
begin
        Light.DecLight( Id, Hue );
end;

function TData.GetMapRadarColor( Id : Word ) : Word;
begin
        Result := Radarcol.GetRadarColor( Id );
end;

function TData.GetStaticRadarColor( Id : Word ) : Word;
begin
        Result := Radarcol.GetRadarColor( Id + $4000 );
end;

function TData.GetKeyWords( Zeile : String ) : TList;
begin
        Result := SpeechLoader.GetKeyWords( Zeile );
end;

function TData.GetMultiItems( Id : Word ) : TList;
begin
        Result := Multis.GetItems( Id );
end;

function TData.GetMultiAsPixelBuffer( Id : Word; Color : Word ) : TPixelBuffer;
begin
        Result := Multis.GetMultiAsPixelBuffer( Id, Color );
end;

function TData.GetSkillName( Id : Word ) : String;
begin
        Result := Skills.GetSkillName( Id );
end;

function TData.SkillHasButton( Id : Word ) : Boolean;
begin
        Result := Skills.HasButton( Id );
end;

function TData.GetSkillGroup( Id : Word ) : Byte;
begin
        Result := Skills.GetSkillGroup( Id );
end;

function TData.GetSkillGroupName( Id : Byte ) : String;
begin
        Result := Skills.GetGroupName( Id );
end;

procedure TData.SetSkillGroup( Id : Word; Group : Byte );
begin
        Skills.SetSkillGroup( Id, Group );
end;

procedure TData.SetSkillGroupName( Id : Byte; Name : String );
begin
        Skills.SetGroupName( Id, Name );
end;

function TData.AddSkillGroup( Name : String ) : Byte;
begin
        Result := Skills.AddSkillGroup( Name );
end;

function TData.GetSkillGroupCount : Byte;
begin
        Result := Skills.GetSkillGroupCount;
end;

function TData.GetSkillCount : Byte;
begin
        Result := Skills.GetSkillCount;
end;

end.
