unit uGestureHandler;

interface

const   AngleTolerance = 10.0;
        DistanceTolerance = 5.0;

type    TGesturePoint = Class;
        TGestureHandler = Class
                private
                        Points : TGesturePoint;
                public
                        constructor Create;
                        destructor Free;
                        procedure AddPoint( X, Y : Integer );
                        procedure ClearPoints;
                        procedure Check;
                        procedure Save;
        end;

        TGesturePoint = Class
                public
                        X, Y : Integer;
                        LastPoint, NextPoint : TGesturePoint;
                        constructor Create( NewX, NewY : Integer );
                        destructor Free;
                        function GetAngle( GesturePoint : TGesturePoint ) : Real;
                        function Distance( GesturePoint : TGesturePoint ) : Real;
        end;

implementation

uses     Graphics, Math;

constructor TGestureHandler.Create;
begin
        Points := nil;
end;

destructor TGestureHandler.Free;
var     GesturePoint, NextGesturePoint : TGesturePoint;
begin
        GesturePoint := Points;
        while GesturePoint <> nil do begin
                NextGesturePoint := Points.NextPoint;
                GesturePoint.Free;
                GesturePoint := NextGesturePoint;
        end;
end;

procedure TGestureHandler.AddPoint( X, Y : Integer );
var     GesturePoint, NewGesturePoint : TGesturePoint;
        Diff : Real;
begin
        if Points = nil then begin
                Points := TGesturePoint.Create( X, Y );
        end else begin
                GesturePoint := Points;
                while GesturePoint.NextPoint <> nil do begin
                        GesturePoint := GesturePoint.NextPoint;
                end;

                NewGesturePoint := TGesturePoint.Create( X, Y );
                if GesturePoint.Distance( NewGesturePoint ) < DistanceTolerance then begin
                        NewGesturePoint.Free;
                        Exit;
                end;

                if GesturePoint.LastPoint <> nil then begin
                        Diff := Abs( GesturePoint.LastPoint.GetAngle( NewGesturePoint ) - GesturePoint.GetAngle( NewGesturePoint ) );
                        if (Diff < AngleTolerance) or (Abs(Diff-360) < AngleTolerance) then begin
                                GesturePoint.LastPoint.NextPoint := NewGesturePoint;
                                NewGesturePoint.LastPoint := GesturePoint.LastPoint;
                                GesturePoint.Free;
                        end else begin
                                GesturePoint.NextPoint := NewGesturePoint;
                                GesturePoint.NextPoint.LastPoint := GesturePoint;
                        end;
                end else begin
                        GesturePoint.NextPoint := NewGesturePoint;
                        GesturePoint.NextPoint.LastPoint := GesturePoint;
                end;
        end;
end;

procedure TGestureHandler.ClearPoints;
var     GesturePoint, NextGesturePoint : TGesturePoint;
begin
        GesturePoint := Points;
        while GesturePoint <> nil do begin
                NextGesturePoint := GesturePoint.NextPoint;
                GesturePoint.Free;
                GesturePoint := NextGesturePoint;
        end;
        Points := nil;
end;

procedure TGestureHandler.Check;
begin
        //Save;
end;

procedure TGestureHandler.Save;
var     Bitmap : TBitmap;
        MinX, MaxX, MinY, MaxY : Integer;
        GesturePoint : TGesturePoint;
begin
        MinX := 10000;
        MaxX := -10000;
        MinY := 10000;
        MaxY := -10000;

        GesturePoint := Points;
        while GesturePoint <> nil do begin
                if GesturePoint.X < MinX then
                        MinX := GesturePoint.X;
                if GesturePoint.X > MaxX then
                        MaxX := GesturePoint.X;
                if GesturePoint.Y < MinY then
                        MinY := GesturePoint.Y;
                if GesturePoint.Y > MaxY then
                        MaxY := GesturePoint.Y;
                GesturePoint := GesturePoint.NextPoint;
        end;

        Bitmap := TBitmap.Create;
        Bitmap.Width := MaxX - MinX + 1;
        Bitmap.Height := MaxY - MinY + 1;

        GesturePoint := Points;
        while GesturePoint <> nil do begin
                Bitmap.Canvas.Pixels[ GesturePoint.X-MinX, GesturePoint.Y-MinY ] := clBlack;
                GesturePoint := GesturePoint.NextPoint;
        end;

        Bitmap.SaveToFile( 'C:\Gesture.bmp' );
        Bitmap.FreeImage;
end;

constructor TGesturePoint.Create( NewX, NewY : Integer );
begin
        X := NewX;
        Y := NewY;
        NextPoint := nil;
        LastPoint := nil;
end;

destructor TGesturePoint.Free;
begin
end;

function TGesturePoint.GetAngle( GesturePoint : TGesturePoint ) : Real;
var     dx, dy : Integer;
begin
        dx := X - GesturePoint.X;
        dy := Y - GesturePoint.Y;

        Result := arctan2( dy, dx ) * 260 / (2*Pi) + 180;
end;

function TGesturePoint.Distance( GesturePoint : TGesturePoint ) : Real;
begin
        Result := sqrt( (GesturePoint.X-X)*(GesturePoint.X-X) + (GesturePoint.Y-Y)*(GesturePoint.Y-Y) );
end;

end.
