unit uLightManager;

interface

uses    uTexCache, uUtilities, SysUtils, uRenderer, Classes, uLoaderLight,
        dglOpenGl, Dialogs, Math, uPixelBuffer, uCounter, uTimer;

type    TLight = Class
                private
                        _Id : Byte;
                        _Color : Word;
                        _PosX, _PosY : Integer;
                        procedure SetId( Id : Byte );
                        procedure SetColor( Color : Word );
                        procedure SetPosX( PosX : Integer );
                        procedure SetPosY( PosY : Integer );
                public
                        UODrawObject : Pointer;
                        property Id : Byte read _Id write SetId;
                        property Color : Word read _Color write SetColor;
                        property PosX : Integer read _PosX write SetPosX;
                        property PosY : Integer read _PosY write SetPosY;
                        constructor Create( NPosX, NPosY : Integer; NId : Byte; NColor : Word );
                        destructor Free;
        end;

        TLightManager = Class
                private
                        Cache : TList;
                        LastGlobalLight : Real;
                        PixelBuffer : TPixelBuffer;
                        UseLight : Boolean;
                        Changed : Boolean;
                public
                        property Enabled : Boolean read UseLight;
                        constructor Create;
                        destructor Free;
                        procedure AddLight( Light : TLight );
                        procedure RemoveLight( Light : TLight );
                        procedure Clear;
                        procedure Draw;
                        procedure SetChanged;
        end;

implementation

uses    uPalanthir, uBasicTypedefs;

constructor TLightManager.Create;
begin
        Cache := TList.Create;
        LastGlobalLight := 1;
        UseLight := True;
        PixelBuffer := nil;
end;

destructor TLightManager.Free;
begin
        Cache.Free;

        if PixelBuffer <> nil then
                PixelBuffer.Destroy;        
end;

procedure TLightManager.AddLight( Light : TLight );
begin
        Cache.Add( Light );
        SetChanged;
end;

procedure TLightManager.RemoveLight( Light : TLight );
var     I : Integer;
begin
        for I := 0 to Cache.Count-1 do
                if TLight( Cache.Items[ I ] ) = Light then begin
                        TLight( Cache.Items[ I ] ).Free;
                        Cache.Delete( I );
                        break;
                end;

        SetChanged;
end;

procedure TLightManager.Clear;
var     I : Integer;
begin
        for I := 0 to Cache.Count-1 do begin
                TLight( Cache.Items[ I ] ).Free;
        end;
        Cache.Clear;

        SetChanged;
end;


procedure TLightManager.Draw;
var     I : Integer;
        Texture : TTexObject;
        Light : TLight;
        Color : Real;
        PX, PY : Integer;
begin
        if not UseLight then
                exit;

        if Palanthir.Player.PersonalLightLevel > Palanthir.GetLightLevel then
                exit;

        if (PixelBuffer <> nil) and (
           (PixelBuffer.Width <> GetNextBit( Palanthir.GameWindow.Breite )) or
           (PixelBuffer.Height <> GetNextBit( Palanthir.GameWindow.Hoehe )) ) then begin
                PixelBuffer.Destroy;
                PixelBuffer := nil;
        end;

        if PixelBuffer = nil then begin
                PixelBuffer := TPixelBuffer.Create( GetNextBit( Palanthir.GameWindow.Breite ), GetNextBit( Palanthir.GameWindow.Hoehe ), Renderer.DC, Renderer.RC, nil );
                if PixelBuffer.Init then begin
                        wglShareLists( Renderer.RC, PixelBuffer.RC );
                end
                else begin
                        PixelBuffer.Destroy;
                        PixelBuffer := nil;
                        UseLight := False;
                        exit;
                end;
        end;

        Color := ( 32 - Palanthir.GetLightLevel + Palanthir.Player.PersonalLightLevel ) / 32;
        if Changed or (Color <> LastGlobalLight) then begin
                PixelBuffer.Enable;
                glEnable( GL_TEXTURE_2D );

                glMatrixMode( GL_PROJECTION );
                glLoadIdentity;
                glViewPort( 0, 0, PixelBuffer.Width, PixelBuffer.Height );

                glMatrixMode( GL_MODELVIEW );
                glLoadIdentity;
                glOrtho( 0, PixelBuffer.Width, 0, PixelBuffer.Height, -1, 1 );

                glClearColor( Color, Color, Color, 1 );
                glClear( GL_COLOR_BUFFER_BIT );

                glEnable( GL_ALPHA_TEST );
                glAlphaFunc( GL_GREATER, 0.1 );

                glEnable( GL_BLEND );
                glBlendFunc( GL_ONE, GL_ONE );

                glColor3f( 1, 1, 1 );

                for I := 0 to Cache.Count-1 do begin
                        Light := TLight( Cache.Items[ I ] );
                        Texture := Palanthir.Data.GetLight( Light.Id, Light.Color );

                        if Texture <> nil then begin
                                if Renderer.CurrentTexID <> Texture.TexID then begin
                                        glBindTexture( GL_TEXTURE_2D, Texture.TexID );
                                        Renderer.CurrentTexID := Texture.TexID;
                                end;

                                PX := Light.PosX - ( Texture.Breite div 2 );
                                PY := Light.PosY - ( Texture.Hoehe div 2 );

                                glBegin( gl_Quads );
                                        glTexCoord2f( 0, 0 );
                                        glVertex2f( PX, PY );
                                        glTexCoord2f( 0, Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                                        glVertex2f( PX, PY + Texture.Hoehe );
                                        glTexCoord2f( Texture.Breite / GetNextBit( Texture.Breite ), Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                                        glVertex2f( PX + Texture.Breite, PY + Texture.Hoehe );
                                        glTexCoord2f( Texture.Breite / GetNextBit( Texture.Breite ), 0 );
                                        glVertex2f( PX + Texture.Breite, PY );
                                glEnd;
                        end;
                end;
                glDisable( GL_BLEND );

                PixelBuffer.Disable;

                Changed := False;
                LastGlobalLight := Color;
        end;
        PixelBuffer.Bind;

        glEnable( GL_BLEND );
        glBlendFunc( GL_ZERO, GL_SRC_COLOR );

        glBegin( gl_Quads );
                glTexCoord2f( 0, 0 );
                glVertex2f( Palanthir.GameWindow.X, Palanthir.GameWindow.Y );
                glTexCoord2f( 0, Palanthir.GameWindow.Hoehe / PixelBuffer.Height );
                glVertex2f( Palanthir.GameWindow.X, Palanthir.GameWindow.Y + Palanthir.GameWindow.Hoehe );
                glTexCoord2f( Palanthir.GameWindow.Breite / PixelBuffer.Width, Palanthir.GameWindow.Hoehe / PixelBuffer.Height );
                glVertex2f( Palanthir.GameWindow.X + Palanthir.GameWindow.Breite, Palanthir.GameWindow.Y + Palanthir.GameWindow.Hoehe );
                glTexCoord2f( Palanthir.GameWindow.Breite / PixelBuffer.Width, 0 );
                glVertex2f( Palanthir.GameWindow.X + Palanthir.GameWindow.Breite, Palanthir.GameWindow.Y );
        glEnd;

        glDisable( GL_BLEND );

        PixelBuffer.Release;
end;

procedure TLightManager.SetChanged;
begin
        Changed := True;
end;

constructor TLight.Create( NPosX, NPosY : Integer; NId : Byte; NColor : Word );
begin
        PosX := NPosX;
        PosY := NPosY;
        Id := NId;
        Color := NColor;

        Counter.IncCount( Count_Light );
end;

destructor TLight.Free;
begin
        Counter.DecCount( Count_Light );

        if UODrawObject <> nil then
                TUODrawObject( UODrawObject ).Light := nil;
end;

procedure TLight.SetId( Id : Byte );
begin
        if _Id <> Id then begin
                _Id := Id;
                Palanthir.LightManager.SetChanged;
        end;
end;

procedure TLight.SetColor( Color : Word );
begin
        if _Color <> Color then begin
                _Color := Color;
                Palanthir.LightManager.SetChanged;
        end;
end;

procedure TLight.SetPosX( PosX : Integer );
begin
        if _PosX <> PosX then begin
                _PosX := PosX;
                Palanthir.LightManager.SetChanged;
        end;
end;

procedure TLight.SetPosY( PosY : Integer );
begin
        if _PosY <> PosY then begin
                _PosY := PosY;
                Palanthir.LightManager.SetChanged;
        end;
end;

end.
