unit uLog;

interface

uses    {$IFDEF WIN32}Windows,{$ENDIF} SysUtils, Classes, Dialogs, StrLib, DateUtils, SyncObjs;

const   LOG_DEFAULT = $0001;
        LOG_PACKET = $0002;
        LOG_SPEECH = $0004;

type    TLog = Class
                private
                        Console : Boolean;
                        Flag : LongWord;
                        FileName, Dir : String;
                        DefaultLog, PacketLog, SpeechLog : TextFile;
                        CriticalSection : TCriticalSection;
                public
                        constructor Create( FileName, Dir : String; Console : Boolean = False );
                        destructor Free;
                        function Init : Boolean;
                        procedure EnableLog( LogType : LongWord );
                        procedure DisableLog( LogType : LongWord );
                        function IsEnabled( LogType : LongWord ) : Boolean;
                        procedure Write( Liste : TStringList ); overload;
                        procedure Write( Zeile : String; Show : Boolean = False ); overload;
                        procedure Write( LogType : LongWord; Liste : TStringList ); overload;
                        procedure Write( LogType : LongWord; Zeile : String; Show : Boolean = False ); overload;
                        procedure WriteDefaultLog( Zeile : String );
                        procedure WritePacketLog( Zeile : String );
                        procedure WriteSpeechLog( Zeile : String );
                        procedure WriteConsole( Liste : TStringList ); overload;
                        procedure WriteConsole( Zeile : String ); overload;
        end;

var     Log : TLog;

implementation

constructor TLog.Create( FileName, Dir : String; Console : Boolean );
begin
        Flag := 0;

        Self.Dir := Dir;
        Self.FileName := FileName;
        Self.Console := Console;

        CriticalSection := TCriticalSection.Create;

        {$IFDEF WIN32} if Console then  AllocConsole; {$ENDIF}
end;

destructor TLog.Free;
begin
        {$IFDEF WIN32} if Console then FreeConsole; {$ENDIF}

	if IsEnabled( LOG_DEFAULT ) then
       		CloseFile( DefaultLog );
        if IsEnabled( LOG_PACKET ) then
                CloseFile( PacketLog );
        if IsEnabled( LOG_SPEECH ) then
                CloseFile( SpeechLog );

        CriticalSection.Free;
end;

function TLog.Init : Boolean;
begin
        try
                if not DirectoryExists( Dir ) then
                        ForceDirectories( Dir );

                if FileExists( FileName ) then
                        DeleteFile( FileName );

                AssignFile( DefaultLog, Filename );
                {$I-}
                ReWrite( DefaultLog );
                {$I+}

                Write( 'Logging initialized.' );

		Flag := LOG_DEFAULT;
                Result := True;
        except
                Result := False;
        end;
end;

procedure TLog.EnableLog( LogType : LongWord );
var     Path : String;
        Year, Month, Day, Hour, Minute, Second, MilliSecond : Word;
begin
        if not IsEnabled( LogType ) then begin
                DecodeDateTime( Now, Year, Month, Day, Hour, Minute, Second, MilliSecond );
                Case LogType of
                        LOG_PACKET : begin
                                Path := Format( '%s%.4d_%.2d_%.2d__%.2d_%.2d_%.2d_packetlog.txt', [Dir,Year,Month,Day,Hour,Minute,Second] );
                                AssignFile( PacketLog, Path );
                                {$I-}
                                ReWrite( PacketLog );
                                {$I+}
                                end;
                        LOG_SPEECH : begin
                                Path := Format( '%s%.4d_%.2d_%.2d__%.2d_%.2d_%.2d_speechlog.txt', [Dir,Year,Month,Day,Hour,Minute,Second] );
                                AssignFile( SpeechLog, Path );
                                {$I-}
                                ReWrite( SpeechLog );
                                {$I+}
                                end;
                end;
                Flag := Flag or LogType;
        end;
end;

procedure TLog.DisableLog( LogType : LongWord );
begin
        if IsEnabled( LogType ) then begin
                Case LogType of
                        LOG_PACKET : CloseFile( PacketLog );
                        LOG_SPEECH : CloseFile( SpeechLog );
                end;
                Flag := Flag and (not LogType);
        end;
end;

function TLog.IsEnabled( LogType : LongWord ) : Boolean;
begin
        Result := Flag and LogType = LogType;
end;

procedure TLog.Write( Liste : TStringList );
begin
        Write( LOG_DEFAULT, Liste );
end;

procedure TLog.Write( Zeile : String; Show : Boolean );
begin
        if Zeile <> '' then
                Write( LOG_DEFAULT, Zeile, Show );
end;

procedure TLog.Write( LogType : LongWord; Liste : TStringList );
var     I : Integer;
begin
        if (not IsEnabled( LogType )) or (Liste = nil) then
                exit;

        if Liste.Count > 0 then
                for I := 0 to Liste.Count-1 do begin
                        Case LogType of
                                LOG_DEFAULT : WriteDefaultLog( Liste.Strings[ I ] );
                                LOG_PACKET : WritePacketLog( Liste.Strings[ I ] );
                                LOG_SPEECH : WriteSpeechLog( Liste.Strings[ I ] );
                        end;
                end;
        Liste.Free;
end;

procedure TLog.Write( LogType : LongWord; Zeile : String; Show : Boolean = False );
var     Text : String;
        Year, Month, Day, Hour, Minute, Second, MilliSecond : Word;
begin
        if (not IsEnabled( LogType )) or (Zeile = '') then
                exit;

        DecodeDateTime( Now, Year, Month, Day, Hour, Minute, Second, MilliSecond );
        Text := Format( '%d.%d.%d %d:%d:%d:%d: %s', [Day,Month,Year,Hour,Minute,Second,MilliSecond,Zeile] );

        Case LogType of
                LOG_DEFAULT : WriteDefaultLog( Text );
                LOG_PACKET : WritePacketLog( Text );
                LOG_SPEECH : WriteSpeechLog( Text );
        end;

        if Show then
                ShowMessage( Text );
end;

procedure TLog.WriteDefaultLog( Zeile : String );
begin
        CriticalSection.Enter;
        WriteLn( DefaultLog, Zeile );
        {$IFDEF WIN32} if Console then WriteLn( Zeile ); {$ENDIF}
        Flush( DefaultLog );
        CriticalSection.Leave;
end;

procedure TLog.WritePacketLog( Zeile : String );
begin
        CriticalSection.Enter;
        WriteLn( PacketLog, Zeile );
        Flush( PacketLog );
        CriticalSection.Leave;
end;

procedure TLog.WriteSpeechLog( Zeile : String );
begin
        CriticalSection.Enter;
        WriteLn( SpeechLog, Zeile );
        Flush( SpeechLog );
        CriticalSection.Leave;
end;

procedure TLog.WriteConsole( Zeile : String );
begin
        CriticalSection.Enter;
        {$IFDEF WIN32} if Console then WriteLn( Zeile ); {$ENDIF}
        CriticalSection.Leave;
end;

procedure TLog.WriteConsole( Liste : TStringList );
var     I : Integer;
begin
        CriticalSection.Enter;
        {$IFDEF WIN32} if Console then for I := 0 to Liste.Count-1 do WriteLn( Liste.Strings[ I ] ); {$ENDIF}
        CriticalSection.Leave;
        Liste.Free;
end;

end.
