unit uOpenGLRenderer;

interface

uses    Classes, SysUtils, dglOpenGL,
        uBaseRenderer, uByteCache, SDL, uLog, uGeometry, uBitMask;

type    TOpenGLWindow = Class( TBaseWindow )
                private
                        mBitPerPixel : LongWord;
                        mFlags : LongWord;
                        mSurface : pSDL_Surface;
                public
                        constructor Create( vFullScreen : Boolean ); override;
                        destructor Free; override;
                        procedure SetSize( vWidth, vHeight : Word ); override;
                        procedure AllowResize( vAllowResize : Boolean ); override;
                        procedure SetCaption( vCaption : String ); override;
                        procedure BringToFront; override;
                        function Poll : Boolean; override;
        end;

        TOpenGLTexture = Class( TBaseTexture )
                private
                        mTexId : Integer;
                public
                        property pTexId : Integer read mTexId;
                        constructor Create( vType : Byte ); override;
                        destructor Free; override;
                        function Load( vByteCache : TByteCache; vHasAlpha : Boolean = False ) : Boolean; override;
        end;

        TOpenGLRenderer = Class( TBaseRenderer )
                protected
                        mBoundingBoxList : TList;
                        mCurrentTexId : Integer;
                        mPixelBufferList : TList;
                        mDC, mRC : LongWord;
                public
                        property pDC : LongWord read mDC;
                        property pRC : LongWord read mRC;
                        constructor Create( vWindow : TBaseWindow ); override;
                        destructor Free; override;
                        procedure SetSize( vWidth, vHeight : Word ); override;
                        procedure StartDraw; override;
                        procedure FinishDraw; override;
                        function CreateTexture( vType : Byte ) : TBaseTexture; override;
                        function DrawTexture( vTex : TBaseTexture; vCoord : TRect3D; vTexCoord : TRect2D ) : Boolean; override;
                        function DrawRectangle( vCoord : TRect3D; vRed, vGreen, vBlue : Byte; vAlpha : Byte = 255 ) : Boolean; override;
                        procedure AddBoundingBox( vRect : TRect2D ); override;
                        procedure RemoveLastBoundingBox; override;
                        procedure EnablePixelBuffer( PixelBuffer : TBasePixelBuffer ); override;
                        procedure DisablePixelBuffer( PixelBuffer : TBasePixelBuffer ); override;
                        procedure BindPixelBuffer( PixelBuffer : TBasePixelBuffer ); override;
                        procedure ReleasePixelBuffer( PixelBuffer : TBasePixelBuffer ); override;
        end;

        TOpenGLPixelBuffer = Class( TBasePixelBuffer )
                private
                        mHandle : LongWord;
                        mLoaded : Boolean;
                        mTexId : Integer;
                        mDC, mRC : LongWord;
                public
                        property pDC : LongWord read mDC;
                        property pRC : LongWord read mRC;
                        property pTexId : Integer read mTexId;
                        property pHandle : LongWord read mHandle;
                        procedure Enable; override;
                        procedure Disable; override;
                        procedure Bind; override;
                        procedure Release; override;
                        function Init : Boolean; override;                        
                        constructor Create( vWidth, vHeight : Word; vRenderer : TBaseRenderer ); override;
                        destructor Free; override;
        end;

implementation

uses    uPalanthir;

constructor TOpenGLWindow.Create( vFullScreen : Boolean );
var     videoInfo : PSDL_VideoInfo;
begin
        inherited Create( vFullScreen );

        mFlags := SDL_OPENGL;
        mFlags := mFlags and not ( SDL_NOFRAME or SDL_RESIZABLE );

        if SDL_Init( SDL_INIT_VIDEO ) = -1 then begin
                Log.Write( 'Could not initialize SDL.' );
                Exit;
        end else begin
                Log.Write( 'SDL initialized.' );
        end;

        videoInfo := SDL_GetVideoInfo;

        if videoInfo = nil then begin
                Log.Write( 'No video modes found.' );
                Exit;
        end;

        mFlags := mFlags or SDL_DOUBLEBUF or SDL_HWPALETTE;
	      if videoInfo.hw_available <> 0 then begin
    	        mFlags := mFlags or SDL_HWSURFACE
	      end else begin
            	mFlags := mFlags or SDL_SWSURFACE;
        end;

	      if videoInfo.blit_hw <> 0 then begin
            	mFlags := mFlags or SDL_HWACCEL;
        end;

        if vFullScreen then begin
                mFlags := mFlags or SDL_FULLSCREEN;
        end;

        SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
        SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
        SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );
        SDL_GL_SetAttribute( SDL_GL_ALPHA_SIZE, 8 );

        SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 32 );
        SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

        SetCaption( 'OpenGL Window' );

        mBitPerPixel := 32;        

        mSurface := SDL_SetVideoMode( mWidth, mHeight, mBitPerPixel, mFlags );
        if mSurface = nil then begin
                Log.Write( 'Could not open main window.' );
                Exit;
        end else begin
                mInitialized := True;
        end;

        mRenderer := TOpenGLRenderer.Create( Self );
end;

destructor TOpenGLWindow.Free;
begin
        //vor SDL renderer aufrumen
        mRenderer.Free;

        SDL_Quit;

        inherited Free;
end;

procedure TOpenGLWindow.SetSize( vWidth, vHeight : Word );
begin
        if mAllowResize then begin
                mWidth := vWidth;
                mHeight := vHeight;

                mSurface := SDL_SetVideoMode( mWidth, mHeight, mBitPerPixel, mFlags );
        end;
end;

procedure TOpenGLWindow.AllowResize( vAllowResize : Boolean );
begin
        if vAllowResize <> mAllowResize then begin
                mAllowResize := vAllowResize;
                if vAllowResize then begin
                        mFlags := mFlags or SDL_RESIZABLE;
                end else begin
                        mFlags := mFlags and not SDL_RESIZABLE;
                end;

                mSurface := SDL_SetVideoMode( mWidth, mHeight, mBitPerPixel, mFlags );
        end;
end;

procedure TOpenGLWindow.SetCaption( vCaption : String );
begin
        SDL_WM_SetCaption( PChar( vCaption ), nil );
end;

procedure TOpenGLWindow.BringToFront;
begin
end;

function TOpenGLWindow.Poll : Boolean;
var     Event : TSDL_Event;
begin
        Result := True;

        while ( SDL_PollEvent( @Event ) = 1 ) do begin
                case Event.type_ of
                        SDL_QUITEV : begin
                                Result := False;
                        end;

                        SDL_VIDEORESIZE : begin
                                SetSize( Event.resize.w, Event.resize.h );
                        end;
                end;
        end;
end;

constructor TOpenGLTexture.Create( vType : Byte );
begin
        inherited Create( vType );
        mTexId := -1;
end;

destructor TOpenGLTexture.Free;
begin
        if mTexId <> -1 then begin
                glDeleteTextures( 1, @mTexId );
        end;

        inherited Free;
end;

function TOpenGLTexture.Load( vByteCache : TByteCache; vHasAlpha : Boolean ) : Boolean;
begin
        Result := False;

        mHasAlpha := vHasAlpha;
        mWidth := vByteCache.RealWidth;
        mHeight := vByteCache.RealHeight;

        if vByteCache.GetData <> nil then begin
                glGenTextures( 1, @mTexId );
                glBindTexture( GL_TEXTURE_2D, mTexId );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
                glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, vByteCache.Width, vByteCache.Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, vByteCache.GetData );

                mBitMask.Init32( vByteCache );
        end;
end;

constructor TOpenGLRenderer.Create( vWindow : TBaseWindow );
begin
        inherited Create( vWindow );

        mCurrentTexId := -1;
        mBoundingBoxList := TList.Create;
        mPixelBufferList := TList.Create;

        InitOpenGL;

        glViewPort( 0, 0, mWidth, mHeight );
        glDisable( GL_DITHER );
        glEnable( GL_DEPTH_TEST );
        glDepthFunc( GL_GEQUAL );
        glDisable( GL_BLEND );
        glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
        
        Log.Write( glGetString( GL_VENDOR ) );
        Log.Write( glGetString( GL_RENDERER ) );
        Log.Write( glGetString( GL_VERSION ) );
        Log.Write( glGetString( GL_EXTENSIONS ) );

        mDC := wglGetCurrentDC;
        mRC := wglGetCurrentContext;
end;

destructor TOpenGLRenderer.Free;
begin
        mBoundingBoxList.Free;
        mPixelBufferList.Free;
        
        inherited Free;
end;

procedure TOpenGLRenderer.SetSize( vWidth, vHeight : Word );
begin
        inherited SetSize( vWidth, vHeight );
end;

procedure TOpenGLRenderer.StartDraw;
begin
        glClear( GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT or GL_STENCIL_BUFFER_BIT );
        glClearColor( 0, 0, 0, 0 );
        glClearDepth( 0 );        

        glMatrixMode( GL_PROJECTION );
        glLoadIdentity;
        glOrtho( 0, mWidth, mHeight, 0, Low( Integer ), High( Integer ) );

        glMatrixMode( GL_MODELVIEW );
        glLoadIdentity;

        glEnable( GL_TEXTURE_2D );
        glDisable( GL_ALPHA_TEST );
end;

procedure TOpenGLRenderer.FinishDraw;
begin
        SDL_GL_SwapBuffers;
end;

function TOpenGLRenderer.CreateTexture( vType : Byte ) : TBaseTexture;
begin
        Result := TOpenGLTexture.Create( vType );
end;

function TOpenGLRenderer.DrawTexture( vTex : TBaseTexture; vCoord : TRect3D; vTexCoord : TRect2D ) : Boolean;
var     Tex : TOpenGLTexture;
begin
        Tex := TOpenGLTexture( vTex );

        if Tex.pTexId <> mCurrentTexId then begin
                glBindTexture( GL_TEXTURE_2D, Tex.pTexId );
                mCurrentTexId := Tex.pTexId;
        end;

        if Tex.pHasAlpha then begin
                glEnable( GL_BLEND );
        end;

        glBegin( GL_Quads );
                glTexCoord2f( vTexCoord.X1, vTexCoord.Y1 );
                glVertex3f( vCoord.X1, vCoord.Y1, vCoord.Z1 );
                glTexCoord2f( vTexCoord.X1, vTexCoord.Y2 );
                glVertex3f( vCoord.X1, vCoord.Y2, vCoord.Z1 );
                glTexCoord2f( vTexCoord.X2, vTexCoord.Y2 );
                glVertex3f( vCoord.X2, vCoord.Y2, vCoord.Z1 );
                glTexCoord2f( vTexCoord.X2, vTexCoord.Y1 );
                glVertex3f( vCoord.X2, vCoord.Y1, vCoord.Z1 );
        glEnd;

        Result := False;
end;

function TOpenGLRenderer.DrawRectangle( vCoord : TRect3D; vRed, vGreen, vBlue, vAlpha : Byte ) : Boolean;
begin
        glDisable( GL_TEXTURE_2D );

        if vAlpha <> 255 then begin
                glEnable( GL_BLEND );
        end;

        glColor4f( vRed / 255, vGreen / 255, vBlue / 255, vAlpha / 255 );
        glBegin( gl_Quads );
                glVertex3f( vCoord.X1, vCoord.Y1, vCoord.Z1 );
                glVertex3f( vCoord.X1, vCoord.Y2, vCoord.Z1 );
                glVertex3f( vCoord.X2, vCoord.Y2, vCoord.Z1 );
                glVertex3f( vCoord.X2, vCoord.Y1, vCoord.Z1 );
        glEnd;

        if vAlpha <> 255 then begin
                glDisable( GL_BLEND );
        end;

        glEnable( GL_TEXTURE_2D );

        Result := True;
end;

procedure TOpenGLRenderer.AddBoundingBox( vRect : TRect2D );
var     Rect : PRect2D;
begin
        if mBoundingBoxList.Count = 0 then begin
                glEnable( GL_SCISSOR_TEST );
        end;

        New( Rect );
        Rect.StartPoint := vRect.StartPoint;
        Rect.EndPoint := vRect.EndPoint;

        glScissor( Rect.X1, pHeight-Rect.Y2, Rect.X2-Rect.X1+1, Rect.Y2-Rect.Y1+1 );
        mBoundingBoxList.Add( Rect );
end;

procedure TOpenGLRenderer.RemoveLastBoundingBox;
var     Rect : PRect2D;
begin
        mBoundingBoxList.Delete( mBoundingBoxList.Count-1 );

        if mBoundingBoxList.Count > 0 then begin
                Rect := PRect2D( mBoundingBoxList.Items[ mBoundingBoxList.Count-1 ] );
                glScissor( Rect.X1, pHeight-Rect.Y2, Rect.X2-Rect.X1+1, Rect.Y2-Rect.Y1+1 );
        end else begin
                gLDisable( GL_SCISSOR_TEST );
        end;
end;

procedure TOpenGLRenderer.EnablePixelBuffer( PixelBuffer : TBasePixelBuffer );
var     OpenGLPixelBuffer : TOpenGLPixelBuffer;
begin
        OpenGLPixelBuffer := TOpenGLPixelBuffer( PixelBuffer );

        mPixelBufferList.Add( PixelBuffer );
        wglMakeCurrent( OpenGLPixelBuffer.pDC, OpenGLPixelBuffer.pRC );
end;

procedure TOpenGLRenderer.DisablePixelBuffer( PixelBuffer : TBasePixelBuffer );
var     OpenGLPixelBuffer : TOpenGLPixelBuffer;
begin
        if mPixelBufferList.Items[ mPixelBufferList.Count-1 ] <> PixelBuffer then begin
            Log.Write( 'Only last enabled pixelbuffer can be disabled' );
        end;

        mPixelBufferList.Delete( mPixelBufferList.Count-1 );
        if mPixelBufferList.Count > 0 then begin
            OpenGLPixelBuffer := TOpenGLPixelBuffer( mPixelBufferList.Items[ mPixelBufferList.Count-1 ] );
            wglMakeCurrent( OpenGLPixelBuffer.pDC, OpenGLPixelBuffer.pRC );
        end else begin
            wglMakeCurrent( pDC, pRC );
        end;
end;

procedure TOpenGLRenderer.BindPixelBuffer( PixelBuffer : TBasePixelBuffer );
var     OpenGLPixelBuffer : TOpenGLPixelBuffer;
begin
        OpenGLPixelBuffer := TOpenGLPixelBuffer( PixelBuffer );

        glBindTexture( GL_TEXTURE_2D, OpenGLPixelBuffer.pTexId );
        mCurrentTexId := OpenGLPixelBuffer.pTexId;
        wglBindTexImageARB( OpenGLPixelBuffer.pHandle, WGL_FRONT_LEFT_ARB );
end;

procedure TOpenGLRenderer.ReleasePixelBuffer( PixelBuffer : TBasePixelBuffer );
var     OpenGLPixelBuffer : TOpenGLPixelBuffer;
begin
        OpenGLPixelBuffer := TOpenGLPixelBuffer( PixelBuffer );
        wglReleaseTexImageARB( OpenGLPixelBuffer.pHandle, WGL_FRONT_LEFT_ARB );
end;

constructor TOpenGLPixelBuffer.Create( vWidth, vHeight : Word; vRenderer : TBaseRenderer );
begin
        inherited Create( vWidth, vHeight, vRenderer );

        mDC := $FFFFFFFF;
        mRC := $FFFFFFFF;
        mTexId := -1;
        mLoaded := False;
end;

destructor TOpenGLPixelBuffer.Free;
begin
        try
                if mTexId <> -1 then
                        glDeleteTextures( 1, @mTexId );
                if mRC <> $FFFFFFFF then
                        wglDeleteContext( mRC );
                if mDC <> $FFFFFFFF then begin
                        wglReleasePbufferDCARB( mHandle, mDC );
                        wglDestroyPbufferARB( mHandle );
                end;
        except
                Log.Write( 'Error destroying Pixelbuffer' );
        end;

        inherited Free;
end;

procedure TOpenGLPixelBuffer.Enable;
begin
        TOpenGLRenderer( mRenderer ).EnablePixelBuffer( Self );
end;

procedure TOpenGLPixelBuffer.Disable;
begin
        TOpenGLRenderer( mRenderer ).DisablePixelBuffer( Self );
end;

procedure TOpenGLPixelBuffer.Bind;
begin
        TOpenGLRenderer( mRenderer ).BindPixelBuffer( Self );
end;

procedure TOpenGLPixelBuffer.Release;
begin
        TOpenGLRenderer( mRenderer ).ReleasePixelBuffer( Self );
end;

function TOpenGLPixelBuffer.Init : Boolean;
const   PixelFormatAttribs : array[0..14] of TGLUInt = ( WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
                                                  WGL_DRAW_TO_PBUFFER_ARB, GL_TRUE,
                                                  WGL_COLOR_BITS_ARB, 24,
                                                  WGL_ALPHA_BITS_ARB, 8,
                                                  WGL_DEPTH_BITS_ARB, 24,
                                                  WGL_STENCIL_BITS_ARB, 1,
                                                  WGL_DOUBLE_BUFFER_ARB, GL_FALSE, 0 );

        PixelBufferAttribs : array[0..4] of TGLUInt = ( WGL_TEXTURE_FORMAT_ARB, WGL_TEXTURE_RGBA_ARB,
                                                 WGL_TEXTURE_TARGET_ARB, WGL_TEXTURE_2D_ARB, 0 );
        EmptyF : TGLFLoat = 0;
var     PFormat : array[0..64] of TGLUInt;
        NumPFormat : TGLUInt;
        TempW, TempH : TGLUInt;
        TempDC : LongWord;
        P : Pointer;
        CurrentTexture : Integer;
begin
        Result := False;

        Log.Write( Format( 'Initializing Pixelbuffer Width: %d Height: %d', [mWidth,mHeight] ) );

        if not WGL_ARB_render_texture then begin
                Log.Write( 'Extension not loaded: WGL_ARB_render_texture' );
                Exit;
        end;

        TempDC := TOpenGLRenderer( mRenderer ).pDC;

        P := @wglChoosePixelFormatARB;

        if P = nil then begin
                Log.Write( 'PixelBuffer: wglChoosePixelFormatARB func not found' );
                exit;
        end;

        if not wglChoosePixelFormatARB( TempDC, @PixelFormatAttribs, @EmptyF, Length( PFormat ), @PFormat, @NumPFormat ) then begin
                Log.Write( 'PixelBuffer: No suitable pixelformat found.' );
                exit;
        end;

        mHandle := wglCreatePBufferARB( TempDC, PFormat[ 0 ], mWidth, mHeight, @PixelBufferAttribs );
        if mHandle = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t obtain valid handle' );
                exit;
        end;

        wglQueryPbufferARB( mHandle, WGL_PBUFFER_WIDTH_ARB, @TempW );
        wglQueryPbufferARB( mHandle, WGL_PBUFFER_HEIGHT_ARB, @TempH );

        mDC := wglGetPBufferDCARB( mHandle );
        if mDC = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t obtain valid DC for PBuffer' );
                exit;
        end;

        mRC := wglCreateContext( mDC );
        if mRC = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t create rendercontext for PBuffer' );
                exit;
        end;

        glGenTextures( 1, @mTexId );
        glBindTexture( GL_TEXTURE_2D, mTexId );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

        glGetIntegerv( GL_TEXTURE_BINDING_2D, @CurrentTexture );
        try
                glBindTexture( GL_TEXTURE_2D, mTexId );
                wglBindTexImageARB( mHandle, WGL_FRONT_LEFT_ARB );
                wglReleaseTexImageARB( mHandle, WGL_FRONT_LEFT_ARB );

                mLoaded := True;
                Result := True;

                wglShareLists( TOpenGLRenderer( mRenderer ).pRC, pRC );
        except
                Log.Write( 'Trying Pixelbuffer failed.' );
        end;

        glBindTexture( GL_TEXTURE_2D, CurrentTexture );        
end;

end.
