unit uPixelBuffer;

interface

uses    Windows, Classes, SysUtils, dglOpenGL, uUtilities, uLog, uRenderer,
        uBitMask;

type    TPixelBuffer = class( TComponent )
                private
                        Handle : TGLUInt;                 
                        Loaded : Boolean;
                public
                        BitMask : TBitMask;
                        DC : HDC;
                        RC : HGLRC;
                        ParentDC : HDC;
                        ParentRC : HGLRC;
                        Width : TGLUInt;
                        Height : TGLUInt;
                        TextureID : TGLUInt;
                        RelX, RelY : Integer;
                        procedure Enable;
                        procedure Disable;
                        procedure Bind;
                        procedure Release;
                        function Init : Boolean;                        
                        constructor Create( pWidth, pHeight : Integer; pParentDC, pParentRC : Cardinal; pAOwner : TComponent ); reintroduce;
                        destructor Destroy; override;
        end;

implementation

uses    uPalanthir;

constructor TPixelBuffer.Create( pWidth, pHeight : Integer; pParentDC, pParentRC : Cardinal; pAOwner : TComponent );
begin
        inherited Create( pAOWner );
        ParentDC := pParentDC;
        ParentRC := pParentRC;
        Width := pWidth;
        Height := pHeight;
        DC := $FFFFFFFF;
        RC := $FFFFFFFF;
        TextureID := $FFFFFFFF;
        Loaded := False;
        BitMask := nil;
end;

function TPixelBuffer.Init : Boolean;
const   PixelFormatAttribs : array[0..14] of TGLUInt = ( WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
                                                  WGL_DRAW_TO_PBUFFER_ARB, GL_TRUE,
                                                  WGL_COLOR_BITS_ARB, 24,
                                                  WGL_ALPHA_BITS_ARB, 8,
                                                  WGL_DEPTH_BITS_ARB, 24,
                                                  WGL_STENCIL_BITS_ARB, 1,
                                                  WGL_DOUBLE_BUFFER_ARB, GL_FALSE, 0 );

        PixelBufferAttribs : array[0..4] of TGLUInt = ( WGL_TEXTURE_FORMAT_ARB, WGL_TEXTURE_RGBA_ARB,
                                                 WGL_TEXTURE_TARGET_ARB, WGL_TEXTURE_2D_ARB, 0 );
        EmptyF : TGLFLoat = 0;
var     PFormat : array[0..64] of TGLUInt;
        NumPFormat : TGLUInt;
        TempW, TempH : TGLUInt;
        TempDC : TGLUInt;
        P : Pointer;
begin
        Result := False;

        Log.Write( Format( 'Initializing Pixelbuffer Width: %d Height: %d', [Width,Height] ) );

        if not WGL_ARB_render_texture then begin
                Log.Write( 'Extension not loaded: WGL_ARB_render_texture' );
                exit;
        end;

        TempDC := ParentDC;
        if TempDC = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t obtain valid device context.' );
                exit;
        end;

        P := @wglChoosePixelFormatARB;

        if P = nil then begin
                Log.Write( 'PixelBuffer: wglChoosePixelFormatARB func not found' );
                exit;
        end;

        if not wglChoosePixelFormatARB( TempDC, @PixelFormatAttribs, @EmptyF, Length( PFormat ), @PFormat, @NumPFormat ) then begin
                Log.Write( 'PixelBuffer: No suitable pixelformat found.' );
                exit;
        end;

        Handle := wglCreatePBufferARB( TempDC, PFormat[ 0 ], Width, Height, @PixelBufferAttribs );
        if Handle = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t obtain valid handle' );
                exit;
        end;

        wglQueryPbufferARB( Handle, WGL_PBUFFER_WIDTH_ARB, @TempW );
        wglQueryPbufferARB( Handle, WGL_PBUFFER_HEIGHT_ARB, @TempH );

        DC := wglGetPBufferDCARB( Handle );
        if DC = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t obtain valid DC for PBuffer' );
                exit;
        end;

        RC := wglCreateContext( DC );
        if RC = 0 then begin
                Log.Write( 'PixelBuffer: Couldn''t create rendercontext for PBuffer' );
                exit;
        end;

        glGenTextures( 1, @TextureID );
        glBindTexture( GL_TEXTURE_2D, TextureID );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

        try
                glBindTexture( GL_TEXTURE_2D, TextureID );
                Renderer.CurrentTexID := TextureID;
                wglBindTexImageARB( Handle, WGL_FRONT_LEFT_ARB );
                wglReleaseTexImageARB( Handle, WGL_FRONT_LEFT_ARB );
        except
                Log.Write( 'Trying Pixelbuffer failed.' );
                exit;
        end;

        Loaded := True;
        Result := True;
end;

destructor TPixelBuffer.Destroy;
begin
        try
                if TextureID <> $FFFFFFFF then
                        glDeleteTextures( 1, @TextureID );
                if RC <> $FFFFFFFF then
                        wglDeleteContext( RC );
                if DC <> $FFFFFFFF then begin
                        wglReleasePbufferDCARB( Handle, DC );
                        wglDestroyPbufferARB( Handle );
                end;
                if BitMask <> nil then
                        BitMask.Free;
        except
                Log.Write( 'Error destroying Pixelbuffer' );
        end;
        inherited Destroy;
end;

procedure TPixelBuffer.Enable;
begin
        wglMakeCurrent( DC, RC );
        Renderer.CurrentTexID := Invalid_Texture;
end;

procedure TPixelBuffer.Disable;
begin
        wglMakeCurrent( ParentDC, ParentRC );
        Renderer.CurrentTexID := Invalid_Texture;        
end;

procedure TPixelBuffer.Bind;
begin
        glBindTexture( GL_TEXTURE_2D, TextureID );
        Renderer.CurrentTexID := TextureID;
        wglBindTexImageARB( Handle, WGL_FRONT_LEFT_ARB );
end;

procedure TPixelBuffer.Release;
begin
        wglReleaseTexImageARB( Handle, WGL_FRONT_LEFT_ARB );
end;

end.
