unit uPos;

interface

uses    uUtilities, uCounter, Classes, uLoaderTiledata;

type    TPos = Class
                private
                        function IsBetween( Number, LowerBound, UpperBound : Double; Tolerance : Double = 0.5 ) : Boolean;
                        function CreatePointListTo( Pos : TPos ) : TList;
                        function GetBlockingTiles : TList;
                        function CheckBlockingTiles( List : TList; Target : TPos ) : Boolean;
                public
                        X, Y : Word;
                        Z : ShortInt;
                        Map : Word;
                        constructor Create( PX : Word = 0; PY : Word = 0; PZ : ShortInt = 0; PM : Word = 0 ); overload;
                        constructor Create( Pos : TPos ); overload;
                        destructor Free;
                        function GetBlockID : Integer;
                        procedure SetPos( Pos : TPos ); overload;
                        procedure SetPos( X, Y : Word; Z : ShortInt ); overload;
                        function Equals( Pos : TPos ) : Boolean;
                        function DistanceTo( Pos : TPos ) : Real;
                        function IsInRechteckRange( Pos : TPos; Range : Integer ) : Boolean;
                        function LineOfSight( Pos : TPos ) : Boolean;
        end;

        P_Pos = ^TPos;

        TBlockingItem = Record
                Z : ShortInt;
                Height : Byte;
        end;
        PBlockingItem = ^TBlockingItem;

implementation

uses    uPalanthir, uBasicTypedefs, uItem;

function TPos.IsBetween( Number, LowerBound, UpperBound : Double; Tolerance : Double ) : Boolean;
var     Swap : Double;
begin
        if LowerBound > UpperBound then begin
                Swap := LowerBound;
                LowerBound := UpperBound;
                UpperBound := Swap;
        end;

        Result := ( Number > LowerBound - Tolerance ) and ( Number < UpperBound + Tolerance );
end;

function TPos.CreatePointListTo( Pos : TPos ) : TList;
var     DiffX, DiffY, DiffZ : Integer;
        Len : Double;
        StepX, StepY, StepZ : Double;
        Point : TPos;
        CurrentX, CurrentY, CurrentZ : Double;
begin
        Result := TList.Create;

        DiffX := Pos.X - Self.X;
        DiffY := Pos.Y - Self.Y;
        DiffZ := Pos.Z - Self.Z;

        Len := sqrt( DiffX*DiffX + DiffY*DiffY + DiffZ*DiffZ );

        StepX := DiffX / Len;
        StepY := DiffY / Len;
        StepZ := DiffZ / Len;

        CurrentX := Self.X;
        CurrentY := Self.Y;
        CurrentZ := Self.Z;

        while IsBetween( CurrentX, Pos.X, Self.X ) and IsBetween( CurrentY, Pos.Y, Self.Y ) and IsBetween( CurrentZ, Pos.Z, Self.Z ) do begin
                Point := TPos.Create( Round( CurrentX ), Round( CurrentY ), Round( CurrentZ ), Self.Map );

                if (Result.Count = 0) or (not Point.Equals( TPos( Result.Items[ Result.Count-1 ] ) ) ) then begin
                        Result.Add( Point );
                end else begin
                        Point.Free;
                end;

                CurrentX := CurrentX + StepX;
                CurrentY := CurrentY + StepY;
                CurrentZ := CurrentZ + StepZ;
        end;

        if (Result.Count <> 0) and (not Pos.Equals( TPos( Result.Items[ Result.Count-1 ] ) ) ) then begin
                Result.Add( TPos.Create( Pos ) );
        end;
end;

function TPos.GetBlockingTiles : TList;
var     Tiledata : TTiledata;
        BlockingItem : PBlockingItem;
        MapItem : TMapItem;
        StaticItem : TStaticItem;
        Item : TItem;
        MultiItem : TMultiItem;
        StaticList, ItemList, MultiList : TList;
        I : Integer;
begin
        Result := TList.Create;

        Tiledata := Palanthir.Data.Tiledata;

        MapItem := Palanthir.Data.WorldCache.GetMapItem( Self.X, Self.Y );
        if MapItem <> nil then begin
                New( BlockingItem );
                BlockingItem.Z := MapItem.Pos.Z;
                BlockingItem.Height := 0;

                if MapItem.ID <> $2 then
                        Result.Add( BlockingItem )
                else
                        FreeMem( BlockingItem );
        end;

        StaticList := Palanthir.Data.WorldCache.GetStaticObjects( Self.X, Self.Y );
        for I := 0 to StaticList.Count-1 do begin
                StaticItem := TStaticItem( StaticList.Items[ I ] );

                if not ( Tiledata.GetStaticFlag( StaticItem.ID, dNOSHOOT ) or Tiledata.GetStaticFlag( StaticItem.ID, dWINDOW ) ) then
                        continue;

                new( BlockingItem );
                BlockingItem.Z := StaticItem.Pos.Z;

                if Tiledata.GetStaticFlag( StaticItem.ID, dBRIDGE ) then
                        BlockingItem.Height := Tiledata.GetStaticHeight( StaticItem.ID ) div 2
                else
                        BlockingItem.Height := Tiledata.GetStaticHeight( StaticItem.ID );

                if StaticItem.Id <> $1 then
                        Result.Add( BlockingItem )
                else
                        FreeMem( BlockingItem );
        end;
        StaticList.Free;

        ItemList := Palanthir.Data.WorldCache.GetItems( Self.X, Self.Y );
        for I := 0 to ItemList.Count-1 do begin
                Item := TItem( ItemList.Items[ I ] );

                if not ( Tiledata.GetStaticFlag( Item.ID, dNOSHOOT ) or Tiledata.GetStaticFlag( Item.ID, dWINDOW ) ) then
                        continue;

                new( BlockingItem );
                BlockingItem.Z := Item.Pos.Z;

                if Tiledata.GetStaticFlag( Item.ID, dBRIDGE ) then
                        BlockingItem.Height := Tiledata.GetStaticHeight( Item.ID ) div 2
                else
                        BlockingItem.Height := Tiledata.GetStaticHeight( Item.ID );

                if Item.Id <> $1 then
                        Result.Add( BlockingItem )
                else
                        FreeMem( BlockingItem );
        end;
        ItemList.Free;

        MultiList := Palanthir.Data.WorldCache.GetMultiItems( Self.X, Self.Y );
        for I := 0 to MultiList.Count-1 do begin
                MultiItem := TMultiItem( MultiList.Items[ I ] );

                if not ( Tiledata.GetStaticFlag( MultiItem.ID, dNOSHOOT ) or Tiledata.GetStaticFlag( MultiItem.ID, dWINDOW ) ) then
                        continue;

                new( BlockingItem );
                BlockingItem.Z := MultiItem.Pos.Z;

                if Tiledata.GetStaticFlag( MultiItem.ID, dBRIDGE ) then
                        BlockingItem.Height := Tiledata.GetStaticHeight( MultiItem.ID ) div 2
                else
                        BlockingItem.Height := Tiledata.GetStaticHeight( MultiItem.ID );

                if MultiItem.Id <> $1 then
                        Result.Add( BlockingItem )
                else
                        FreeMem( BlockingItem );
        end;
        MultiList.Free;
end;

function TPos.CheckBlockingTiles( List : TList; Target : TPos ) : Boolean;
var     I : Integer;
        BlockingItem : PBlockingItem;
begin
        Result := False;

        for I := 0 to List.Count-1 do begin
                BlockingItem := PBlockingItem( List.Items[ I ] );

                if ( Self.Z >= BlockingItem.Z ) and ( Self.Z <= BlockingItem.Z + BlockingItem.Height ) then begin
                        if ( Self.X <> Target.X ) or ( Self.Y <> Target.Y ) or ( BlockingItem.Z > Target.Z ) or ( BlockingItem.Z + BlockingItem.Height < Target.Z ) then begin
                                Result := True;
                                Exit;
                        end;
                end;
        end;
end;

constructor TPos.Create( PX, PY : Word; PZ : ShortInt; PM : Word );
begin
        X := PX;
        Y := PY;
        Z := PZ;
        Map := PM;

        Counter.IncCount( Count_Pos );
end;

constructor TPos.Create( Pos : TPos );
begin
        X := Pos.X;
        Y := Pos.Y;
        Z := Pos.Z;

        Map := Pos.Map;

        Counter.IncCount( Count_Pos );
end;

destructor TPos.Free;
begin
        Counter.DecCount( Count_Pos );
end;

function TPos.GetBlockID : Integer;
begin
        Result := ( X div 8 ) * GetMapHoehe + ( Y div 8 );
end;

procedure TPos.SetPos( Pos : TPos );
begin
        X := Pos.X;
        Y := Pos.Y;
        Z := Pos.Z;
        Map := Pos.Map;
end;

procedure TPos.SetPos( X, Y : Word; Z : ShortInt );
begin
        Self.X := X;
        Self.Y := Y;
        Self.Z := Z;
end;

function TPos.Equals( Pos : TPos ) : Boolean;
begin
        if (Pos.X = X) and (Pos.Y = Y) and (Pos.Z = Z) and (Pos.Map = Map) then
                Result := True
        else
                Result := False;
end;

function TPos.DistanceTo( Pos : TPos ) : Real;
begin
        Result := Sqrt( (Pos.X-X)*(Pos.X-X) + (Pos.Y-Y)*(Pos.Y-Y) );
end;

function TPos.IsInRechteckRange( Pos : TPos; Range : Integer ) : Boolean;
begin
        if Pos.Map <> Map then begin
                Result := False;
                exit;
        end;

        Result := ( X >= ( Pos.X - Range ) ) and
                  ( X <= ( Pos.X + Range ) ) and
                  ( Y >= ( Pos.Y - Range ) ) and
                  ( Y <= ( Pos.Y + Range ) );
end;

function TPos.LineOfSight( Pos : TPos ) : Boolean;
var     PointList, BlockingList : TList;
        LastX, LastY : Integer;
        Point : TPos;
        I, J : Integer;
begin
        if (Self.Map <> Pos.Map) or (Pos.DistanceTo( Self ) > 25) then begin
                Result := False;
                Exit;
        end;

        if Pos.Equals( Self ) then begin
                Result := True;
                Exit;
        end;

        PointList := CreatePointListTo( Pos );
        LastX := -1;
        LastY := -1;
        BlockingList := TList.Create;

        for I := 0 to PointList.Count-1 do begin
                Point := TPos( PointList.Items[ I ] );

                if (Point.X <> LastX) or (Point.Y <> LastY) then begin
                        for J := 0 to BlockingList.Count-1 do begin
                                Dispose( PBlockingItem( BlockingList.Items[ J ] ) );
                        end;
                        BlockingList.Free;
                        BlockingList := Point.GetBlockingTiles;

                        LastX := Point.X;
                        LastY := Point.Y;
                end;

                if Point.CheckBlockingTiles( BlockingList, Pos ) then begin
                        Result := False;
                        Exit;
                end;
        end;

        for J := 0 to BlockingList.Count-1 do begin
                Dispose( PBlockingItem( BlockingList.Items[ J ] ) );
        end;
        BlockingList.Free;

        Result := True;
end;

end.
