unit uRenderer;

interface

uses    SysUtils, Classes, Windows, dglOpenGL, Math, uTexCache, uUtilities,
        uTimer, uCounter, uLog, uConfig, SyncObjs;

type    TPoint = Record
                X, Y : Integer;
        end;

        TRenderer = Class
                private
                        Breite, Hoehe : Word;
                        NBreite, NHoehe : Word;
                        _Handle : HWND;
                        BBX, BBY, BBWidth, BBHeight : Integer;
                        procedure InternalDrawPixels( X, Y : Integer; Texture : TTexObject; Transparent : Word; MinX, MinY, MaxX, MaxY : Integer; Gespiegelt : Boolean; ZoomX, ZoomY, Light : Real; StartX, StartY : Word );
                public
                        CurrentTexID : LongWord;
                        DC : HDC;
                        RC : HGLRC;
                        constructor Create( Handle : HWND );
                        destructor Free;
                        procedure SetSize( NBreite, NHoehe : Word );
                        procedure ChangeSize( NBreite, NHoehe : Word );
                        procedure SetResizable( Resize : Boolean );
                        procedure StartZeichnen;
                        procedure FinishZeichnen;
                        procedure DrawRectangle( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue : Byte; Alpha : Byte = 255 );
                        procedure DrawRectangleGradientHorizontal( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue, Red2, Green2, Blue2 : Byte );
                        procedure DrawRectangleGradientVertical( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue, Red2, Green2, Blue2 : Byte );
                        procedure DrawPixels( X, Y : Integer; Texture : TTexObject; Light : Real = 1 );
                        procedure DrawPixelsTransp( X, Y : Integer; Texture : TTexObject; Transparenz : Word; Light : Real = 1 );
                        procedure DrawPixelsZoom( X, Y : Integer; Texture : TTexObject; Gespiegelt : Boolean; ZoomX : Real = 1; ZoomY : Real = 1; Light : Real = 1 );
                        procedure DrawPixelsMax( X, Y : Integer; Texture : TTexObject; MaxX, MaxY : Integer; Light : Real = 1 );
                        procedure DrawPixelsMinMaxY( X, Y : Integer; Texture : TTexObject; MinY, MaxY : Integer; Light : Real = 1 );
                        procedure DrawPixelsMinMax( X, Y : Integer; Texture : TTexObject; MinY, MaxX, MaxY : Integer; Light : Real = 1 );
                        procedure DrawPixelsStartMax( X, Y : Integer; Texture : TTexObject; StartX, StartY, MaxX, MaxY : Integer; Light : Real = 1 );
                        procedure DrawPixelsHoeheBreite( X, Y : Integer; Texture : TTexObject; Hoehe : Word; Breite : Word; Gespiegelt : Boolean; Light : Real = 1 );
                        function GetWindowBreite : Word;
                        function GetWindowHoehe : Word;
                        function CreateTexture32( Typ : Byte; Breite, Hoehe : Word; Pixel : PByteArray; Id : LongWord = 0; Hue : Word = 0; Frame : Word = 0 ) : TTexObject;
                        procedure InitTexture32( Texture : TTexObject; Breite, Hoehe : Word; Pixel : PByteArray );
                        procedure SetBoundingBox( X, Y : Integer; Width, Height : Word );
                        procedure ReleaseBoundingBox;
        end;

var     Renderer : TRenderer;

implementation

uses    uClientForm, uPalanthir;

constructor TRenderer.Create( Handle : HWND );
var     LightPos, LightValues, Values : Array[0..3] of GLFloat;
begin
        _Handle := Handle;

        InitOpenGL;
        DC := GetDC( _Handle );
        RC := CreateRenderingContext( DC, [opDoubleBuffered], 32, 24, 1, 0, 0, 0 );
        ActivateRenderingContext( DC, RC );
        glClearColor( 0, 0, 0, 0 );
        glDisable( GL_DITHER );
        CurrentTexID := Invalid_Texture;

        glEnable( GL_ALPHA_TEST );
        glAlphaFunc( GL_GREATER, 0.1 );

        glEnable( GL_LIGHT0 );

        LightPos[ 0 ] := -1;
        LightPos[ 1 ] := -1;
        LightPos[ 2 ] := 0.5;
        LightPos[ 3 ] := 0;
        glLightfv( GL_LIGHT0, GL_POSITION, @LightPos[0] );

        LightValues[ 0 ] := 2;
        LightValues[ 1 ] := 2;
        LightValues[ 2 ] := 2;
        LightValues[ 3 ] := 1;
        glLightfv( GL_LIGHT0, GL_AMBIENT, @LightValues[0] );

        Values[ 0 ] := 1;
        Values[ 1 ] := 1;
        Values[ 2 ] := 1;
        Values[ 3 ] := 1;
        glLightModelfv( GL_LIGHT_MODEL_AMBIENT, @Values[0] );

        glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE );

        Log.Write( glGetString( GL_VENDOR ) );
        Log.Write( glGetString( GL_RENDERER ) );
        Log.Write( glGetString( GL_VERSION ) );
        Log.Write( glGetString( GL_EXTENSIONS ) );
end;

destructor TRenderer.Free;
begin
        try
                DeActivateRenderingContext;
                DestroyRenderingContext( RC );
                ReleaseDC( _Handle, DC );
        except
                Log.Write( 'Error freeing renderer.' );
        end;
end;

procedure TRenderer.SetSize( NBreite, NHoehe : Word );
begin
        Self.NBreite := NBreite;
        Self.NHoehe := NHoehe;

        if Palanthir.State = State_Playing then begin
                Config.SetInteger( 'ScreenHeight', NHoehe );
                Config.SetInteger( 'ScreenWidth', NBreite );
        end;
end;

procedure TRenderer.ChangeSize( NBreite, NHoehe : Word );
begin
        SetSize( NBreite, NHoehe );
end;

procedure TRenderer.SetResizable( Resize : Boolean );
begin
        ClientForm.SetResize( Resize );
end;

procedure TRenderer.StartZeichnen;
begin
        if (Breite <> NBreite) or (Hoehe <> NHoehe) then begin
                Breite := NBreite;
                Hoehe := NHoehe;
                ClientForm.CustomResize( NBreite, NHoehe );
        end;

        glViewPort( 0, 0, Breite, Hoehe );

        glLoadIdentity;
        glClear( GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT or GL_STENCIL_BUFFER_BIT );
        glClearColor( 0, 0, 0, 0 );

        glMatrixMode( GL_PROJECTION );
        glLoadIdentity;
        glOrtho( 0, Breite, Hoehe, 0, -1, 1 );

        glMatrixMode( GL_TEXTURE );
        glLoadIdentity;

        glMatrixMode( GL_MODELVIEW );
        glLoadIdentity;
        //gluLookAt( I, 0.0, 1.0, I, 0.0, 0.0, 0.0, 1.0, 0.0);

        glEnable( GL_TEXTURE_2D );
end;

procedure TRenderer.FinishZeichnen;
begin
        Timer.Start( Timer_SwapBuffers );
        SwapBuffers( DC );
        Timer.Stop( Timer_SwapBuffers );
end;

procedure TRenderer.InternalDrawPixels( X, Y : Integer; Texture : TTexObject; Transparent : Word; MinX, MinY, MaxX, MaxY : Integer; Gespiegelt : Boolean; ZoomX, ZoomY, Light : Real; StartX, StartY : Word );
var     PosXMin, PosXMax, PosYMin, PosYMax : Integer;
        XMin, XMax, YMin, YMax : Integer;
        TP : Real;
        BlendingEnabled : Boolean;
begin
        Timer.Continue( Timer_Renderer );
        Timer.Continue( Timer_RendererCalc );

        if (Texture = nil) or (Texture.TexID = -1) then begin
                Timer.Pause( Timer_Renderer );
                Timer.Pause( Timer_RendererCalc );
                exit;
        end;

        if ( X >= MaxX ) or ( X + Texture.Breite < MinX ) or ( Y >= MaxY ) or ( Y + Texture.Hoehe < MinY ) then begin
                Timer.Pause( Timer_Renderer );
                Timer.Pause( Timer_RendererCalc );
                exit;
        end;

        TP := 1;
        BlendingEnabled := False;

        if Texture.HasAlpha then begin
                TP := 1.0;

                glEnable( GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                BlendingEnabled := True;
        end
        else if Transparent = ItemTransparenz then begin
                TP := 0.7;

                glEnable( GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                BlendingEnabled := True;
        end
        else if Transparent = FoliageTransparenz then begin
                TP := 0.3;

                glEnable( GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                BlendingEnabled := True;
        end
        else if Transparent < 255 then begin
                TP := Transparent / 255;

                glEnable( GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                BlendingEnabled := True;
        end;

        glColor4f( Light, Light, Light, TP );

        if CurrentTexID <> Texture.TexID then begin
                glBindTexture( GL_TEXTURE_2D, Texture.TexID );
                CurrentTexID := Texture.TexID;
        end;

        XMin := Max( X, MinX );
        PosXMin := StartX + XMin - X;

        YMin := Max( Y, MinY );
        PosYMin := StartY + YMin - Y;

        XMax := Min( X + Texture.Breite, MaxX );
        PosXMax := StartX + XMax - X;

        YMax := Min( Y + Texture.Hoehe, MaxY );
        PosYMax := StartY + YMax - Y;

        if ZoomX <> 1 then begin
                XMin := XMin + Round( Texture.Breite * ( 1 - ZoomX ) / 2 );
                XMax := XMax - Round( Texture.Breite * ( 1 - ZoomX ) / 2 );
        end;
        if ZoomY <> 1 then begin
                YMin := YMin + Round( Texture.Hoehe * ( 1 - ZoomY ) );
        end;

        Timer.Pause( Timer_RendererCalc );

        if not Gespiegelt then begin
                Timer.Continue( Timer_Draw );
                glBegin( gl_Quads );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMin );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMin );
                glEnd;
                Timer.Pause( Timer_Draw );
        end
        else begin
                Timer.Continue( Timer_Draw );
                glBegin( gl_Quads );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMin );
                        gltexcoord2f( PosXMin / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMax, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMax / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMax );
                        gltexcoord2f( PosXMax / GetNextBit( Texture.Breite ), PosYMin / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( XMin, YMin );
                glEnd;
                Timer.Pause( Timer_Draw );
        end;

        if BlendingEnabled then begin
                glDisable( GL_BLEND );
        end;

        Timer.Pause( Timer_Renderer );
end;

procedure TRenderer.DrawRectangle( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue, Alpha : Byte );
begin
        glDisable( GL_TEXTURE_2D );
        glColor3f( Red / 255, Green / 255, Blue / 255 );

        glBegin( gl_Quads );
                glvertex2f( X, Y );
                glvertex2f( X + Breite, Y );
                glvertex2f( X + Breite, Y + Hoehe );
                glvertex2f( X, Y + Hoehe );
        glEnd;

        glEnable( GL_TEXTURE_2D );
end;

procedure TRenderer.DrawRectangleGradientHorizontal( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue, Red2, Green2, Blue2 : Byte );
begin
        glDisable( GL_TEXTURE_2D );

        glBegin( gl_Quads );
                glColor3f( Red / 255, Green / 255, Blue / 255 );
                glvertex2f( X, Y );
                glColor3f( Red2 / 255, Green2 / 255, Blue2 / 255 );
                glvertex2f( X + Breite, Y );
                glvertex2f( X + Breite, Y + Hoehe );
                glColor3f( Red / 255, Green / 255, Blue / 255 );
                glvertex2f( X, Y + Hoehe );
        glEnd;

        glEnable( GL_TEXTURE_2D );
end;

procedure TRenderer.DrawRectangleGradientVertical( X, Y : Integer; Hoehe, Breite : Word; Red, Green, Blue, Red2, Green2, Blue2 : Byte );
begin
        glDisable( GL_TEXTURE_2D );

        glBegin( gl_Quads );
                glColor3f( Red / 255, Green / 255, Blue / 255 );
                glvertex2f( X, Y );
                glvertex2f( X + Breite, Y );
                glColor3f( Red2 / 255, Green2 / 255, Blue2 / 255 );
                glvertex2f( X + Breite, Y + Hoehe );
                glvertex2f( X, Y + Hoehe );
        glEnd;

        glEnable( GL_TEXTURE_2D );
end;

procedure TRenderer.DrawPixels( X, Y : Integer; Texture : TTexObject; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, 0, Breite, Hoehe, False, 1, 1, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsTransp( X, Y : Integer; Texture : TTexObject; Transparenz : Word; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, Transparenz, 0, 0, Breite, Hoehe, False, 1, 1, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsZoom( X, Y : Integer; Texture : TTexObject; Gespiegelt : Boolean; ZoomX, ZoomY, Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, 0, Breite, Hoehe, Gespiegelt, ZoomX, ZoomY, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsMax( X, Y : Integer; Texture : TTexObject; MaxX, MaxY : Integer; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, 0, MaxX, MaxY, False, 1, 1, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsMinMaxY( X, Y : Integer; Texture : TTexObject; MinY, MaxY : Integer; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, MinY, Breite, MaxY, False, 1, 1, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsMinMax( X, Y : Integer; Texture : TTexObject; MinY, MaxX, MaxY : Integer; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, MinY, MaxX, MaxY, False, 1, 1, Light, 0, 0 );
end;

procedure TRenderer.DrawPixelsStartMax( X, Y : Integer; Texture : TTexObject; StartX, StartY, MaxX, MaxY : Integer; Light : Real );
begin
        InternalDrawPixels( X, Y, Texture, 255, 0, 0, MaxX, MaxY, False, 1, 1, Light, StartX, StartY );
end;

procedure TRenderer.DrawPixelsHoeheBreite( X, Y : Integer; Texture : TTexObject; Hoehe : Word; Breite : Word; Gespiegelt : Boolean; Light : Real = 1 );
var     BlendingEnabled : Boolean;
begin
        Timer.Continue( Timer_Renderer );
        Timer.Continue( Timer_RendererCalc );

        BlendingEnabled := False;

        if (Texture = nil) or (Texture.TexID = -1) then begin
                Timer.Pause( Timer_Renderer );
                Timer.Pause( Timer_RendererCalc );
                exit;
        end;

        if ( X >= Self.Breite ) or ( X + Texture.Breite < 0 ) or ( Y >= Self.Hoehe ) or ( Y + Texture.Hoehe < 0 ) then begin
                Timer.Pause( Timer_Renderer );
                Timer.Pause( Timer_RendererCalc );
                exit;
        end;

        if Texture.HasAlpha then begin
                glEnable( GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                BlendingEnabled := True;
        end;

        glColor4f( Light, Light, Light, 1 );

        if CurrentTexID <> Texture.TexID then begin
                glBindTexture( GL_TEXTURE_2D, Texture.TexID );
                CurrentTexID := Texture.TexID;
        end;

        Timer.Pause( Timer_RendererCalc );

        Timer.Continue( Timer_Draw );
        if not Gespiegelt then begin
                glBegin( gl_Quads );
                        gltexcoord2f( 0, 0 );
                        glvertex2f( X, Y );
                        gltexcoord2f( 0, Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( X, Y + Hoehe );
                        gltexcoord2f( Texture.Breite / GetNextBit( Texture.Breite ), Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( X + Breite, Y + Hoehe );
                        gltexcoord2f( Texture.Breite / GetNextBit( Texture.Breite ), 0 );
                        glvertex2f( X + Breite, Y );
                glEnd;
        end
        else begin
                glBegin( gl_Quads );
                        gltexcoord2f( 0, 0 );
                        glvertex2f( X + Breite, Y );
                        gltexcoord2f( 0, Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( X + Breite, Y + Hoehe );
                        gltexcoord2f( Texture.Breite / GetNextBit( Texture.Breite ), Texture.Hoehe / GetNextBit( Texture.Hoehe ) );
                        glvertex2f( X , Y + Hoehe );
                        gltexcoord2f( Texture.Breite / GetNextBit( Texture.Breite ), 0 );
                        glvertex2f( X , Y );
                glEnd;
        end;

        if BlendingEnabled then begin
                glDisable( GL_BLEND );
        end;

        Timer.Pause( Timer_Draw );
        
        Timer.Pause( Timer_Renderer );
end;

function TRenderer.GetWindowBreite : Word;
begin
        Result := Breite;
end;

function TRenderer.GetWindowHoehe : Word;
begin
        Result := Hoehe;
end;

function TRenderer.CreateTexture32( Typ : Byte; Breite, Hoehe : Word; Pixel : PByteArray; Id : LongWord; Hue : Word; Frame : Word ) : TTexObject;
var     RBreite, RHoehe : Word;
begin
        Result := TTexObject.Create( Typ );
        Result.Breite := Breite;
        Result.Hoehe := Hoehe;
        Result.ID := Id;
        Result.Hue := Hue;
        Result.Frame := Frame;

        if Pixel <> nil then begin
                RBreite := GetNextBit( Breite );
                RHoehe := GetNextBit( Hoehe );

                glGenTextures( 1, @Result.TexID );
                glBindTexture( GL_TEXTURE_2D, Result.TexID );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
                glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, RBreite, RHoehe, 0, GL_RGBA, GL_UNSIGNED_BYTE, Pixel );
                Result.InitBitMask32( Pixel );
        end;
end;

procedure TRenderer.InitTexture32( Texture : TTexObject; Breite, Hoehe : Word; Pixel : PByteArray );
var     RBreite, RHoehe : Word;
begin
        Texture.Breite := Breite;
        Texture.Hoehe := Hoehe;

        if Pixel <> nil then begin
                RBreite := GetNextBit( Breite );
                RHoehe := GetNextBit( Hoehe );

                glGenTextures( 1, @Texture.TexID );
                glBindTexture( GL_TEXTURE_2D, Texture.TexID );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
                glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
                glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, RBreite, RHoehe, 0, GL_RGBA, GL_UNSIGNED_BYTE, Pixel );
                Texture.InitBitMask32( Pixel );
        end;
end;

procedure TRenderer.SetBoundingBox( X, Y : Integer; Width, Height : Word );
var     Params : Array[0..3] of Integer;
begin
        glGetIntegerv( GL_SCISSOR_BOX, @Params );
        BBX := Params[0];
        BBY := Params[1];
        BBWidth := Params[2];
        BBHeight := Params[3];

        glEnable( GL_SCISSOR_TEST );
        glScissor( X, Renderer.GetWindowHoehe-Y-Height, Width, Height );
end;

procedure TRenderer.ReleaseBoundingBox;
begin
        glDisable( GL_SCISSOR_TEST );
        glScissor( BBX, BBY, BBWidth, BBHeight );
end;

end.
