unit uScriptManager;

interface

uses    Python, uPyPalanthir, uLog, SysUtils, Classes, Dialogs, StrUtils,
        uPythonScript, uPythonFunction, uPacket, uPyPacket, uPyItem,
        uPyChar, uItem, uChar, uPyGump, uGumps, uBasicTypeDefs, uPyUODrawObject;

type    TScriptManager = Class
                private
                        Functions : TList;
                        PacketHandlers : TList;
                        PythonPath, Path : String;
                        Scripts : TList;                        
                        procedure LoadScriptFromDir( Dir : String );
                        procedure LoadScript( ScriptName : String );
                public
                        constructor Create;
                        destructor Free;
                        procedure StartPython( Path, PythonPath : String );
                        procedure StopPython;
                        procedure LoadScripts;
                        procedure UnLoadScripts;
                        procedure ReloadScripts;
                        procedure GetError( Name : String = '' );
                        function HandleEvent( Event : String; Args : PPyObject = nil; ForAllScripts : Boolean = False ) : Boolean;
                        procedure AddFunction( Func : TPythonFunction );
                        procedure RemoveFunction( Func : TPythonFunction );
                        procedure RegisterPacketHandler( PacketId : Byte; PyFunction : PPyObject );

                        function OnClientReady : Boolean;
                        function OnRecievePacket( Packet : UOPacket ) : Boolean;
                        function OnAddItemToCont( Container, Item : TItem ) : Boolean;
                        function OnAddItemToChar( Container : TChar; Item : TItem ) : Boolean;
                        function OnRemoveItemFromCont( Container, Item : TItem ) : Boolean;
                        function OnRemoveItemFromChar( Container : TChar; Item : TItem ) : Boolean;
                        function OnAddGump( Gump : TGump ) : Boolean;
                        function OnRemoveGump( Gump : TGump ) : Boolean;
                        function OnSpeech( Text : String ) : Boolean;
                        function OnClick( UODrawObject : TUODrawObject ) : Boolean;
                        function OnMove( Direction : Byte ) : Boolean;
                        function OnMouseStayOver( UODrawObject : TUODrawObject ) : Boolean;
                        function OnGameEnter : Boolean;
                        function OnGameLeave : Boolean;
                        function OnWarModeChange( WarMode : Boolean ) : Boolean;
        end;

        TPythonPacketHandler = Class
                private
                        _PacketId : Byte;
                        _PythonFunction : TPythonFunction;
                public
                        property PacketId : Byte read _PacketId;
                        property PythonFunction : TPythonFunction read _PythonFunction;
                        constructor Create( PacketId : Byte; PythonFunction : TPythonFunction );
                        destructor Free;
        end;

implementation


constructor TScriptManager.Create;
begin
        Scripts := TList.Create;
        Functions := TList.Create;
        PacketHandlers := TList.Create;
end;

destructor TScriptManager.Free;
begin
        UnLoadScripts;
        Scripts.Free;
        Functions.Free;
        PacketHandlers.Free;
end;

procedure TScriptmanager.StartPython( Path, PythonPath : String );
var     Buf : PChar;
        Argv : PPChar;
        PythonLog : PPyObject;
        SearchPath : PPyObject;
        Site : PPyObject;
begin
        Self.Path := Path;
        Self.PythonPath := PythonPath;

        Py_SetProgramName( PChar( Path ) );

        Py_NoSiteFlag^ := 1;

        Log.Write( 'Calling Py_Initialize' );
        Py_Initialize;

        GetMem( Buf, sizeof( PChar ) );
        Argv := PPChar( Buf );
        Argv^[0] := PChar( Path );
        PySys_SetArgv( 1, Argv );

        Log.Write( 'Configuring Python Searchpath' );
        SearchPath := PyList_New( 0 );
        PyList_Insert( SearchPath, 0, PyString_FromString( PChar( PythonPath ) ) );
        PySys_SetObject( 'path', SearchPath );

        Log.Write( 'Opening pythonlog' );
        PythonLog := PyFile_FromString( 'log\python.log', 'w' );
        if PythonLog <> nil then begin
                Py_INCREF( PythonLog );
                PySys_SetObject( 'stderr', PythonLog );
		Py_INCREF( PythonLog );
		PySys_SetObject( 'stdout', PythonLog );
		Py_DECREF( PythonLog );
        end;

        Log.Write( 'Importing Modules' );
        Site := PyImport_ImportModule( 'site' );
	Py_XDECREF( Site );

        Init_pyPalanthir;
end;

procedure TScriptManager.StopPython;
begin
        try
                Py_Finalize;
        except
        end;
end;

procedure TScriptManager.LoadScripts;
begin
        LoadScriptFromDir( PythonPath );
        HandleEvent( 'onLoad', nil, True );
end;

procedure TScriptManager.LoadScriptFromDir( Dir : String );
var     SearchDir, SearchFile : TSearchRec;
        FileName : String;
begin
        if FindFirst( Dir + '*', faAnyFile, SearchFile ) = 0 then begin
                repeat
                        if (SearchFile.Name <> '.') and (SearchFile.Name <> '..') and FileExists( Dir + SearchFile.Name ) then begin
                                FileName := SearchFile.Name;
                                if ExtractFileExt( FileName ) = '.py' then begin
                                        FileName := ChangeFileExt( FileName, '' );
                                        FileName := Copy( Dir, Length( PythonPath )+1, Length( Dir )-Length( PythonPath ) ) + FileName;
                                        FileName := AnsiReplaceStr( FileName, '\', '.' );
                                        FileName := AnsiReplaceStr( FileName, '/', '.' );                                        
                                        LoadScript( FileName );
                                end;
                        end;
                until FindNext( SearchFile ) <> 0;
        end;
        FindClose( SearchFile );

        if FindFirst( Dir + '*', faDirectory, SearchDir ) = 0 then begin
                repeat
                        if (SearchDir.Name <> '.') and (SearchDir.Name <> '..') and DirectoryExists( Dir + SearchDir.Name ) then begin
                                LoadScriptFromDir( Dir + SearchDir.Name + '\' );
                        end;
                until FindNext( SearchDir ) <> 0;
        end;
        FindClose( SearchDir );        
end;

procedure TScriptManager.LoadScript( ScriptName : String );
var     Script : TPythonScript;
begin
        Script := TPythonScript.Create;

        if Script.Load( ScriptName ) then begin
                Scripts.Add( Script );
        end
        else begin
                Log.Write( Format( 'Loading script %s failed.', [ScriptName] ) );
        end;
end;

procedure TScriptManager.UnLoadScripts;
var     I : Integer;
        Script : TPythonScript;
begin
        HandleEvent( 'onUnload', nil, True );
        
        for I := Scripts.Count-1 downto 0 do begin
                Script := TPythonScript( Scripts.Items[ I ] );
                Script.UnLoad;
                Script.Free;
                
                Scripts.Delete( I );
        end;


        for I := 0 to PacketHandlers.Count-1 do begin
                TPythonPacketHandler( PacketHandlers.Items[ I ] ).Free;
        end;
        PacketHandlers.Clear;
end;

procedure TScriptManager.ReloadScripts;
var     I : Integer;
begin
        Log.Write( 'Reloading ScriptManager' );
        Log.Write( 'Unloading ScriptManager' );
        UnloadScripts;

        Log.Write( 'Stopping Python' );
        StopPython;

        Log.Write( 'Starting Python' );
        StartPython( Path, PythonPath );

        Log.Write( 'Loading ScriptManager' );
        LoadScripts;

        Log.Write( 'Reloading PythonFunctions' );
        for I := 0 to Functions.Count-1 do begin
                TPythonFunction( Functions.Items[ I ] ).Reload;
        end;
        Log.Write( 'Reload of ScriptManager done' );
end;

procedure TScriptManager.GetError( Name : String );
var     Exception, Value, Traceback : PPyObject;
        ExceptionName, Error, Frame : PPyObject;
        Code, NewTraceback : PPyObject;
        PyFileName, PyFunctionName, PyLine : PPyObject;
        FileName, FunctionName : String;
        LineNr : LongWord;
begin
        if PyErr_Occurred <> nil then begin
                PyErr_Fetch( @Exception, @Value, @Traceback );
                PyErr_NormalizeException( @Exception, @Value, @Traceback );

                PySys_SetObject( 'last_type', Exception );
		PySys_SetObject( 'last_value', Value );
		PySys_SetObject( 'last_traceback', Traceback );

                ExceptionName := PyObject_GetAttrString( Exception, '__name__' );

                if Name <> '' then begin
                        Log.Write( Format( 'Error in script %s: %s', [Name,PyString_AsString( ExceptionName )] ) );
                end
                else begin
                        Log.Write( Format( 'Error: %s', [PyString_AsString( ExceptionName )] ) );
                end;

                if Value <> nil then begin
                        Error := PyObject_Str( Value );

                        if Error <> nil then begin
                                Log.Write( Format( '%s: %s', [PyString_AsString( ExceptionName ),PyString_AsString( Error )] ) );
                                Py_XDECREF( Error );
                        end;
                end;

                if PyErr_GivenExceptionMatches( Exception, PyExc_SyntaxError^ ) <> 0 then begin
			Py_XDECREF( Traceback );
			Traceback := nil;
		end;

                while Traceback <> nil do begin
                        if PyObject_HasAttrString( Traceback, 'tb_frame' ) = 0 then
                                break;

                        Frame := PyObject_GetAttrString( Traceback, 'tb_frame' );

                        if PyObject_HasAttrString( Frame, 'f_code' ) = 0 then begin
                                Py_XDECREF( Frame );
                                break;
                        end;

                        Code := PyObject_GetAttrString( Frame, 'f_code' );

                        if (PyObject_HasAttrString( Code, 'co_filename' ) = 0) or (PyObject_HasAttrString( Code, 'co_name' ) = 0) then begin
                                Py_XDECREF( Frame );
                                Py_XDECREF( Code );
                                break;
                        end;

                        PyFileName := PyObject_GetAttrString( Code, 'co_filename' );
                        PyFunctionName := PyObject_GetAttrString( Code, 'co_name' );

                        FileName := PyString_AsString( PyFileName );
                        FunctionName := PyString_AsString( PyFunctionName );

                        Py_XDECREF( PyFileName );
                        Py_XDECREF( PyFunctionName );

                        Py_XDECREF( Frame );
                        Py_XDECREF( Code );

                        PyLine := PyObject_GetAttrString( Traceback, 'tb_lineno' );
                        LineNr := PyInt_AsLong( PyLine );
                        Py_XDECREF( PyLine );

                        Log.Write( Format( 'File %s, %d, %s', [FileName,LineNr,FunctionName] ) );

                        NewTraceback := PyObject_GetAttrString( Traceback, 'tb_next' );
                        Py_XDECREF( Traceback );
                        Traceback := NewTraceback;
                end;

                Py_XDECREF( Exception );
                Py_XDECREF( Value );
                Py_XDECREF( Traceback );
                Py_XDECREF( ExceptionName );
        end;
end;

function TScriptManager.HandleEvent( Event : String; Args : PPyObject; ForAllScripts : Boolean ) : Boolean;
var     PythonScript : TPythonScript;
        I : Integer;
begin
        Result := False;
        for I := 0 to Scripts.Count-1 do begin
                PythonScript := TPythonScript( Scripts.Items[ I ] );
                Result := PythonScript.HandleEvent( Event, Args );

                if Result and (not ForAllScripts) then
                        break;
        end;
end;

procedure TScriptManager.AddFunction( Func : TPythonFunction );
begin
        Functions.Add( Func );
end;

procedure TScriptManager.RemoveFunction( Func : TPythonFunction );
var     I : Integer;
begin
        for I := 0 to Functions.Count-1 do begin
                if TPythonFunction( Functions.Items[ I ] ) = Func then begin
                        Functions.Delete( I );
                        Break;
                end;
        end;
end;

procedure TScriptManager.RegisterPacketHandler( PacketId : Byte; PyFunction : PPyObject );
var     PythonPacketHandler : TPythonPacketHandler;
        PythonFunction : TPythonFunction;
begin
        PythonFunction := TPythonFunction.Create( PyFunction );
        PythonPacketHandler := TPythonPacketHandler.Create( PacketId, PythonFunction );
        PacketHandlers.Add( PythonPacketHandler );
end;

{
        \event onClientReady
        \condition Triggered after startup before opening the logingump
}

function TScriptManager.OnClientReady : Boolean;
begin
        Result := HandleEvent( 'onClientReady' );
end;

{
        \event onRecievePacket
        \param packet A <object id="packet">packet</object> object.
        \return Return 1 if you dont wont the core to handle the packet
        \condition Triggered when a new packet is recieved from the server
}

function TScriptManager.OnRecievePacket( Packet : UOPacket ) : Boolean;
var     Args : PPyObject;
        I : Integer;
        PythonPacketHandler : TPythonPacketHandler;
begin
        Args := Py_BuildValue( '(O&)', [@pyGetPacketObject,Packet] );

        Result := False;
        for I := 0 to PacketHandlers.Count-1 do begin
                PythonPacketHandler := TPythonPacketHandler( PacketHandlers.Items[ I ] );
                if PythonPacketHandler.PacketId = Packet.GetByte( 0 ) then begin
                        Result := PythonPacketHandler.PythonFunction.CallFunction( Args );
                        if Result then
                                break;
                end;
        end;

        if not Result then
                Result := HandleEvent( 'onRecievePacket', Args );
        Py_XDECREF( Args );
end;

{
        \event onAddItemToCont
        \param container A <object id="item">item</object> object.
        \param item A <object id="item">item</object> object.
        \condition Triggered after an item is added to a container.
}

function TScriptManager.OnAddItemToCont( Container, Item : TItem ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&O&)', [@pyGetItem,Container,@pyGetItem,Item] );
        Result := HandleEvent( 'onAddItemToCont', Args );
        Py_XDECREF( Args );
end;

{
        \event onAddItemToChar
        \param char A <object id="char">char</object> object.
        \param item A <object id="item">item</object> object.
        \condition Triggered after an item is added to a char.
}

function TScriptManager.OnAddItemToChar( Container : TChar; Item : TItem ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&O&)', [@pyGetChar,Container,@pyGetItem,Item] );
        Result := HandleEvent( 'onAddItemToChar', Args );
        Py_XDECREF( Args );
end;

{
        \event onRemoveItemFromCont
        \param container A <object id="item">item</object> object.
        \param item A <object id="item">item</object> object.
        \condition Triggered after an item removed from a container.
}

function TScriptManager.OnRemoveItemFromCont( Container, Item : TItem ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&O&)', [@pyGetItem,Container,@pyGetItem,Item] );
        Result := HandleEvent( 'onRemoveItemFromCont', Args );
        Py_XDECREF( Args );
end;

{
        \event onRemoveItemFromChar
        \param char A <object id="char">char</object> object.
        \param item A <object id="item">item</object> object.
        \condition Triggered after an item removed from a char.
}

function TScriptManager.OnRemoveItemFromChar( Container : TChar; Item : TItem ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&O&)', [@pyGetChar,Container,@pyGetItem,Item] );
        Result := HandleEvent( 'onRemoveItemFromChar', Args );
        Py_XDECREF( Args );
end;

{
        \event onAddGump
        \param gump A <object id="gump">gump</object> object.
        \return Return 1 if you dont wont the core to add the gump to the drawlist.
        \condition Triggered when a gump is added to the drawlist.
}

function TScriptManager.OnAddGump( Gump : TGump ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&)', [@pyGetGump,Gump] );
        Result := HandleEvent( 'onAddGump', Args );
        Py_XDECREF( Args );
end;

{
        \event onRemoveGump
        \param gump A <object id="gump">gump</object> object.
        \return Return 1 if you dont wont the core to remove the gump the from the drawlist (and delete it afterwards).
        \condition Triggered when a gump is removed from the drawlist.
}

function TScriptManager.OnRemoveGump( Gump : TGump ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&)', [@pyGetGump,Gump] );
        Result := HandleEvent( 'onRemoveGump', Args );
        Py_XDECREF( Args );
end;

{
        \event onCommand
        \param text
        \return Return 1 if you dont wont the core to further evaluate the speechrequest.
        \condition Triggered when a you type a command (text that starts with "?").
}

{
        \event onSpeech
        \param text
        \return Return 1 if you dont wont the core to send the text as speechrequest to the server.
        \condition Triggered when a you send some text.
}

function TScriptManager.OnSpeech( Text : String ) : Boolean;
var     Args : PPyObject;
        Command : String;
        TmpText : PChar;
begin
        Result := False;
        if Text[ 1 ] = '?' then begin
                Command := Copy( Text, 2, Length( Text )-1 );
                TmpText := PChar( Command );
                Args := Py_BuildValue( '(s)', [Command] );
                Result := HandleEvent( 'onCommand', Args );
                Py_XDECREF( Args );
        end;

        if not Result then begin
                TmpText := PChar( Text );
                Args := Py_BuildValue( '(s)', [Text] );
                Result := HandleEvent( 'onSpeech', Args );
                Py_XDECREF( Args );
        end;
end;

{
        \event onClick
        \param object
        \return Return 1 if you dont wont the core to further handle the click.
        \condition Triggered when a you click on an object (map,statics,item,char)
}

function TScriptManager.OnClick( UODrawObject : TUODrawObject ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(O&)', [@pyGetUODrawObject,UODrawObject] );
        Result := HandleEvent( 'onClick', Args );
        Py_XDECREF( Args );
end;

{
        \event onMove
        \param direction
        \condition Triggered when the player moves.
}

function TScriptManager.OnMove( Direction : Byte ) : Boolean;
var     Args : PPyObject;
begin
        Args := Py_BuildValue( '(b)', [Direction] );
        Result := HandleEvent( 'onMove', Args );
        Py_XDECREF( Args );
end;

{
        \event onMouseStayOver
        \param object
        \condition Triggered when the mouse stays on an object for 2 seconds.
}

function TScriptManager.OnMouseStayOver( UODrawObject : TUODrawObject ) : Boolean;
var     Args : PPyObject;
begin
        if UODrawObject <> nil then begin
                Args := Py_BuildValue( '(O&)', [@pyGetUODrawObject,UODrawObject] );
                Result := HandleEvent( 'onMouseStayOver', Args );
                Py_XDECREF( Args );
        end else begin
                Result := False;
        end;
end;

{
        \event onGameEnter
        \condition Triggered when the login procedure is finished and the player enters the game.
}

function TScriptManager.OnGameEnter : Boolean;
begin
        Result := HandleEvent( 'onGameEnter' );
end;

{
        \event onGameLeave
        \condition Triggered when the player leaves the game.
}

function TScriptManager.OnGameLeave : Boolean;
begin
        Result := HandleEvent( 'onGameLeave' );
end;

{
        \event onWarModeChange
        \param warmode
        \condition Triggered when the warmode of the player changes.
}

function TScriptManager.OnWarModeChange( WarMode : Boolean ) : Boolean;
var     Args : PPyObject;
        B : Byte;
begin
        if WarMode then begin
                B := 1;
        end else begin
                B := 0;
        end;
        Args := Py_BuildValue( '(b)', [B] );
        Result := HandleEvent( 'onWarModeChange', Args );
        Py_XDECREF( Args );
end;

constructor TPythonPacketHandler.Create( PacketId : Byte; PythonFunction : TPythonFunction );
begin
        _PacketId := PacketId;
        _PythonFunction := PythonFunction;
end;

destructor TPythonPacketHandler.Free;
begin
        PythonFunction.Free;
end;

end.
