unit uShard;

interface

uses    uConfig, Windows, Classes, SysUtils, uLog, Dialogs, uSendPackets, uItem,
        uLoaderAnimations;
        
const	MountListVersion = 1;
	MacroVersion = 1;

type    TMountEntry = Record
                MountItemId, AnimId : Word;
        end;
        PMountEntry = ^TMountEntry;

        TMapEntry = Record
                MapId : Word;
                MapName, StaIdxName, StaMulName : String;
                Breite, Hoehe : Word;
        end;
        PMapEntry = ^TMapEntry;

        PMacroAction = ^TMacroAction;
        TMacroAction = Record
        	Action : String;
        	NextMacro : PMacroAction;
        end;

        TMacroEntry = Record
                Key : String;
                Alt, Strg, Shift : Boolean;
                Action : PMacroAction;
        end;
        PMacroEntry = ^TMacroEntry;

        TLightTypEntry = Record
                Id : Word;
                Red : Array[ 0..31 ] of Byte;
                Green : Array[ 0..31 ] of Byte;
                Blue : Array[ 0..31 ] of Byte;
        end;
        PLightTypEntry = ^TLightTypEntry;

        TLightEntry = Record
                Id : Word;
                Color : Word;
        end;
        PLightEntry = ^TLightEntry;

        TAnimRecord = packed Record
                Id, Color : Word;
                FileNr : Byte;
                AnimTyp : Byte;
        end;

        AnimRecordList = Array[0..$FFFF] of TAnimRecord;

        TChairEntry = packed Record
                Chair : Boolean;
                Dir0, Dir2, Dir4, Dir6 : Boolean;
        end;

        TChairList = Array[0..$FFFF] of TChairEntry;

        TAnim2PaperdollEntry = packed Record
                AnimId : Word;
                GumpId : Word;
        end;

        PAnim2PaperdollEntry = ^TAnim2PaperdollEntry;

        TAmountEntry = packed Record
                BaseId : Word;
                Amount : Word;
                NewId : Word;
        end;

        PAmountEntry = ^TAmountEntry;

        TBodyEntry = packed Record
                BaseId : Word;
                NewId : Word;
                NewHue : Word;
        end;

        PBodyEntry = ^TBodyEntry;

        TShard = Class
                private
                        procedure CreateDefaultMountListe( FileName : String );
                        procedure LoadMountListe( FileName : String );
                        procedure CreateDefaultMapListe( FileName : String );
                        procedure LoadMapListe( FileName : String );
                        procedure CreateDefaultMacroListe( FileName : String );
                        procedure LoadMacroListe( FileName : String );
                        procedure CreateDefaultLichtTypListe( FileName : String );
                        procedure LoadLichtTypListe( FileName : String );
                        procedure CreateDefaultLichtListe( FileName : String );
                        procedure LoadLichtListe( FileName : String );
                        procedure CreateDefaultBodyConv( UOPath, ShardDir : String );
                        procedure LoadBodyConv( FileName : String );
                        procedure CreateDefaultBodyDef( UOPath, ShardDir : String );
                        procedure LoadBodyDef( FileName : String );
                        procedure CreateDefaultChairList( FileName : String );
                        procedure LoadChairList( FileName : String );
                        procedure CreateDefaultAnimation2PaperdollList( FileName : String );
                        procedure LoadAnimation2PaperdollList( FileName : String );
                        procedure CreateDefaultAmountList( FileName : String );
                        procedure LoadAmountList( FileName : String );
                        procedure CreateDefaultInvisList( FileName : String );
                        procedure LoadInvisList( FileName : String );
                        procedure SaveInvisList( FileName : String );
                public
                        ShardConfig : TConfig;
                        MountListe : TList;
                        MapListe : TList;
                        MacroListe : TList;
                        LichtTypListe : TList;
                        LichtListe : TList;
                        AnimDefs : AnimRecordList;
                        StuhlListe : TChairList;
                        Anim2PaperdollListe : TList;
                        AmountList : TList;
                        Dir : String;
                        constructor Create( UOPath, ConfigFileDir : String );
                        destructor Free;
                        function GetMountAnimId( ItemId : Word ) : Integer;
                        function CheckForKeyMacros( Key : Integer; Alt, Strg, Shift : Boolean ) : Boolean;
                        function GetLightColor( Id : Word ) : Word;
                        function GetLightColorEntry( Color : Word ) : PLightTypEntry;
                        function GetAnimRecord( Id : Word ) : TAnimRecord;
                        function GetChairDirection( Id : Word; CharDirection : Byte ) : Byte;
                        function GetAnimGumpId( AnimId : Word ) : Integer;
                        function GetAmountId( BaseId : Word; Amount : Word ) : Word;
        end;

implementation

uses    uPalanthir;

constructor TShard.Create( UOPath, ConfigFileDir : String );
begin
        Dir := ConfigFileDir;
        ShardConfig := TConfig.Create;
        ShardConfig.Load( ConfigFileDir + 'config.xml' );

        if not DirectoryExists( Dir + 'images' ) then
                ForceDirectories( Dir + 'images' );

        if not DirectoryExists( Dir + 'data' ) then
                ForceDirectories( Dir + 'data' );

        if not DirectoryExists( Dir + 'config' ) then
                ForceDirectories( Dir + 'config' );

        if not DirectoryExists( Dir + 'script' ) then
                ForceDirectories( Dir + 'script' );

        MountListe := TList.Create;
        if not FileExists( Dir + 'config\mountlist.txt' ) then
                CreateDefaultMountListe( Dir + 'config\mountlist.txt' );
        LoadMountListe( Dir + 'config\mountlist.txt' );

        MapListe := TList.Create;
        if not FileExists( Dir + 'config\maplist.txt' ) then
                CreateDefaultMapListe( Dir + 'config\maplist.txt' );
        LoadMapListe( Dir + 'config\maplist.txt' );

        MacroListe := TList.Create;
        if not FileExists( Dir + 'config\macros.txt' ) then
                CreateDefaultMacroListe( Dir + 'config\macros.txt' );
        LoadMacroListe( Dir + 'config\macros.txt' );

        LichtTypListe := TList.Create;
        if not FileExists( Dir + 'config\lighttypelist.txt' ) then
                CreateDefaultLichtTypListe( Dir + 'config\lighttypelist.txt' );
        LoadLichtTypListe( Dir + 'config\lighttypelist.txt' );

        LichtListe := TList.Create;
        if not FileExists( Dir + 'config\lightlist.txt' ) then
                CreateDefaultLichtListe( Dir + 'config\lightlist.txt' );
        LoadLichtListe( Dir + 'config\lightlist.txt' );

        if not FileExists( Dir + 'config\Bodyconv.def' ) then
                CreateDefaultBodyConv( UOPath, Dir + 'config\' );
        LoadBodyConv( Dir + 'config\Bodyconv.def' );

        if not FileExists( Dir + 'config\Body.def' ) then
                CreateDefaultBodyDef( UOPath, Dir + 'config\' );
        LoadBodyDef( Dir + 'config\Body.def' );

        if not FileExists( Dir + 'config\chairlist.txt' ) then
                CreateDefaultChairList( Dir + 'config\chairlist.txt' );
        LoadChairList( Dir + 'config\chairlist.txt' );

        Anim2PaperdollListe := TList.Create;
        if not FileExists( Dir + 'config\anim2paperdoll.txt' ) then
                CreateDefaultAnimation2PaperdollList( Dir + 'config\anim2paperdoll.txt' );
        LoadAnimation2PaperdollList( Dir + 'config\anim2paperdoll.txt' );

        AmountList := TList.Create;
        if not FileExists( Dir + 'config\amount.txt' ) then
                CreateDefaultAmountList( Dir + 'config\amount.txt' );
        LoadAmountList( Dir + 'config\amount.txt' );

        if not FileExists( Dir + 'config\invislist.txt' ) then
                CreateDefaultInvisList( Dir + 'config\invislist.txt' );
        LoadInvisList( Dir + 'config\invislist.txt' );
end;

destructor TShard.Free;
var     I : Integer;
	MacroAction, NextMacroAction : PMacroAction;
begin
        for I := 0 to MountListe.Count-1 do
                Dispose( PMountEntry( MountListe.Items[ I ] ) );
        for I := 0 to MapListe.Count-1 do
                Dispose( PMapEntry( MapListe.Items[ I ] ) );
        for I := 0 to MacroListe.Count-1 do begin
        	MacroAction := PMacroEntry( MacroListe.Items[ I ] ).Action;
        	while MacroAction <> nil do begin
        		NextMacroAction := MacroAction.NextMacro;
        		Dispose( MacroAction );
        		MacroAction := NextMacroAction;
        	end;
                Dispose( MacroListe.Items[ I ] );
        end;
        for I := 0 to LichtTypListe.Count-1 do
                Dispose( PLightTypEntry( LichtTypListe.Items[ I ] ) );
        for I := 0 to LichtListe.Count-1 do
                Dispose( PLightEntry( LichtListe.Items[ I ] ) );
        for I := 0 to Anim2PaperdollListe.Count-1 do
                Dispose( PAnim2PaperdollEntry( Anim2PaperdollListe.Items[ I ] ) );
        for I := 0 to AmountList.Count-1 do
                Dispose( PAmountEntry( AmountList.Items[ I ] ) );

        ShardConfig.Free;
        MountListe.Free;
        MapListe.Free;
        MacroListe.Free;
        LichtTypListe.Free;
        LichtListe.Free;
        Anim2PaperdollListe.Free;
        AmountList.Free;

        SaveInvisList( Dir + 'config\invislist.txt' );
end;

procedure TShard.LoadMountListe( FileName : String );
var     I, SPos : Integer;
        FStream, SubList : TStringList;
        Zeile : String;
        Entry : PMountEntry;
        Version : Integer;
begin
	Version := 0;

        FStream := TStringList.Create;
        FStream.LoadFromFile( FileName );

        for I := 0 to FStream.Count-1 do begin
                Zeile := FStream.Strings[ I ];

                if Zeile = '' then
                        continue;                                             

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                SubList := TStringList.Create;
                SubList.Delimiter := ' ';
                SubList.DelimitedText := Zeile;

                if SubList.Count < 2 then
                        continue;
		
		if SubList.Strings[ 0 ] = 'Version' then begin
			Version := StrToInt( SubList.Strings[ 1 ] );
                        continue;
		end;
		
                New( Entry );
                Entry.MountItemId := StrToInt( SubList.Strings[ 0 ] );
                Entry.AnimId := StrToInt( SubList.Strings[ 1 ] );
                MountListe.Add( Entry );

                SubList.Free;
        end;

        FStream.Free;
        
        if Version < MountListVersion then begin
        	for I := 0 to MountListe.Count-1 do
                	Dispose( PMountEntry( MountListe.Items[ I ] ) );
		MountListe.Clear;
        
        	DeleteFile( FileName );
        	CreateDefaultMountListe( FileName );
        	LoadMountListe( FileName );
        end;
end;

procedure TShard.CreateDefaultMountListe( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// List of all MountItems and their corresponding AnimationIDs' );
        FStream.Add( '// Format: <MountItemId> <AnimationId>' );
        FStream.Add( 'Version ' + IntToStr( MountListVersion ) );

        FStream.Add( '0x3FFE 0xD5' );
        FStream.Add( '0x3FFD 0xF1' );
        FStream.Add( '0x3FFC 0xF3' );

        FStream.Add( '0x3EA2 0xCC // dark brown horse' );
        FStream.Add( '0x3E9F 0xC8 // light brown horse' );
        FStream.Add( '0x3EA0 0xE2 // light grey horse' );
        FStream.Add( '0x3EA1 0xE4 // grey brown horse' );
        FStream.Add( '0x3EA6 0xDC // Llama' );
        FStream.Add( '0x3EA3 0xD2 // desert ostard' );
        FStream.Add( '0x3EA4 0xDA // frenzied ostard' );
        FStream.Add( '0x3EA5 0xDB // forest ostard' );
        FStream.Add( '0x3F14 0x3C // drake' );
        FStream.Add( '0x3FFB 0xD5 // unknown?' );
        FStream.Add( '0x3EA7 0xB1 // nightmare' );
        FStream.Add( '0x3EA8 0x75 // silver steed' );
        FStream.Add( '0x3EA9 0x72 // nightmare' );
        FStream.Add( '0x3EAA 0x73 // etheral' );
        FStream.Add( '0x3E9F 0xC8 // light brown horse' );
        FStream.Add( '0x3EAF 0x78 // war horse (blood red)' );
        FStream.Add( '0x3EB0 0x79 // war horse (light green)' );
        FStream.Add( '0x3EB1 0x77 // war horse (light blue)' );
        FStream.Add( '0x3EB3 0x90 // sea horse (medium blue)' );
        FStream.Add( '0x3EB5 0x74 // nightmare' );
        FStream.Add( '0x3EB6 0xB2 // nightmare 4' );
        FStream.Add( '0x3EAD 0x84 // kirin' );
        FStream.Add( '0x3EB2 0x76 // war horse (purple)' );
        FStream.Add( '0x3EB7 0xB3 // dark steed' );
        FStream.Add( '0x3EB8 0xBB // ridgeback' );
        FStream.Add( '0x3EBA 0xBC // savage ridgeback' );
        FStream.Add( '0x3EBB 0x319 // skeletal mount' );
        FStream.Add( '0x3EBC 0x317 // beetle' );
        FStream.Add( '0x3EBD 0x31A // swampdragon' );
        FStream.Add( '0x3EBE 0x31F // armored swamp dragon' );
        FStream.Add( '0x3E9E 0xBE' );
        FStream.Add( '0x3E92 0x11C // mondain steed' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

function TShard.GetMountAnimId( ItemId : Word ) : Integer;
var     I : Integer;
begin
        for I := 0 to MountListe.Count-1 do
                if ItemId = PMountEntry( MountListe.Items[ I ] ).MountItemId then begin
                        Result := PMountEntry( MountListe.Items[ I ] ).AnimId;
                        exit;
                end;

        Result := -1;
end;

procedure TShard.CreateDefaultMapListe( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// List of all Maps' );
        FStream.Add( '// Format: <MapId> <Mapfile> <StaIdxfile> <StaMulfile> <MapWidthInBlocks> <MapHeightInBlocks>' );
        FStream.Add( '' );

        FStream.Add( '0 map0.mul staidx0.mul statics0.mul 768 512' );
        FStream.Add( '1 map0.mul staidx0.mul statics0.mul 768 512' );
        FStream.Add( '2 map2.mul staidx2.mul statics2.mul 288 200' );
        FStream.Add( '3 map3.mul staidx3.mul statics3.mul 320 256' );
        FStream.Add( '4 map4.mul staidx4.mul statics4.mul 181 181' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadMapListe( FileName : String );
var     I, SPos : Integer;
        FStream, SubList : TStringList;
        Zeile : String;
        Entry : PMapEntry;
begin
        FStream := TStringList.Create;
        FStream.LoadFromFile( FileName );

        for I := 0 to FStream.Count-1 do begin
                Zeile := FStream.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                SubList := TStringList.Create;
                SubList.Delimiter := ' ';
                SubList.DelimitedText := Zeile;

                if SubList.Count < 6 then
                        continue;

                New( Entry );
                Entry.MapId := StrToInt( SubList.Strings[ 0 ] );
                Entry.MapName := SubList.Strings[ 1 ];
                Entry.StaIdxName := SubList.Strings[ 2 ];
                Entry.StaMulName := SubList.Strings[ 3 ];
                Entry.Breite := StrToInt( SubList.Strings[ 4 ] );
                Entry.Hoehe := StrToInt( SubList.Strings[ 5 ] );
                MapListe.Add( Entry );

                SubList.Free;
        end;

        FStream.Free;
end;

procedure TShard.CreateDefaultMacroListe( FileName : String );
var     Liste : TStringList;
begin
        Liste := TStringList.Create;
        Liste.Add( '//Macrofile, Format: <Key> <Alt> <Strg> <Shift>' );
        Liste.Add( '//+<Action1>' );
        Liste.Add( '//+<Action2>' );
        Liste.Add( '//+...' );
        Liste.Add( 'Version ' + IntToStr( MacroVersion ) );
        Liste.Add( 'J 1 0 0' );
        Liste.Add( '+Open Journal' );
        Liste.Add( 'R 1 0 0' );
        Liste.Add( '+Open Overview' );
        Liste.Add( 'P 1 0 0' );
        Liste.Add( '+Open Paperdoll' );
        Liste.Add( 'B 1 0 0' );
        Liste.Add( '+Say Blub' );
        Liste.Add( 'F1 0 0 0' );
        Liste.Add( '+Say Help' );
        Liste.SaveToFile( FileName );
        Liste.Free;
end;

procedure TShard.LoadMacroListe( FileName : String );
var     Liste, SubList : TStringList;
        I, SPos : Integer;
        Zeile : String;
        Entry : PMacroEntry;
        Action, NextAction : PMacroAction;
        Version : Integer;
begin
	Action := nil;
	Entry := nil;
	
	Version := 0;

        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Trim( Liste.Strings[ I ] );

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                SubList := TStringList.Create;
                SubList.Delimiter := ' ';
                SubList.DelimitedText := Zeile;
                if (SubList.Count > 0) and (SubList.Strings[ 0 ] = 'Version') then begin
                        try
        			Version := StrToInt( SubList.Strings[ 1 ] );
                        except
                        end;
                        SubList.Free;                        
                        continue;
		end;
                SubList.Free;
		
		if Zeile[1] = '+' then begin
			Zeile := Copy( Zeile, 2, Length( Zeile )-1 );
			
			New( NextAction );
			NextAction.Action := Zeile;
			NextAction.NextMacro := nil;
			
			if Entry = nil then
				continue;
				
			if Entry.Action = nil then begin
				Entry.Action := NextAction;
				Action := NextAction;
			end
			else begin
				Action.NextMacro := NextAction;
				Action := NextAction;
			end;
			
			continue;
		end;
		
                try
                        SPos := Pos( ' ', Zeile );
                        New( Entry );
                        Entry.Key := Copy( Zeile, 1, SPos-1 );
                        Entry.Alt := Zeile[ SPos+1 ] = '1';
                        Entry.Strg := Zeile[ SPos+3 ] = '1';
                        Entry.Shift := Zeile[ SPos+5 ] = '1';
                        Entry.Action := nil;

                        if Entry.Key = '' then begin
                                Log.Write( Format( 'Bug in macros.txt in line %d', [I] ) );
                                Log.Write( 'Skipping line...' );
                        end
                        else begin
                                MacroListe.Add( Entry );
                        end;
                except
                        Log.Write( Format( 'Bug in macros.txt in line %d', [I] ) );
                        Log.Write( 'Skipping line...' );
                end;
        end;
        Liste.Free;
        
        if Version < MacroVersion then begin
        	for I := 0 to MacroListe.Count-1 do begin
        		Action := PMacroEntry( MacroListe.Items[ I ] ).Action;
        		while Action <> nil do begin
        			NextAction := Action.NextMacro;
        			Dispose( Action );
        			Action := NextAction;
        		end;
        		Dispose( MacroListe.Items[ I ] );
        	end;
        	MacroListe.Clear;
        	
        	DeleteFile( FileName );
        	CreateDefaultMacroListe( FileName );
        	LoadMacroListe( FileName );
        end;
end;

function TShard.CheckForKeyMacros( Key : Integer; Alt, Strg, Shift : Boolean ) : Boolean;
var     I : Integer;
        Entry : PMacroEntry;
        Passt : Boolean;
        SpeechRequest : TSendSpeechRequest;
        SendDoubleClick : TSendDoubleClick;
        BackPack : TItem;
        ChooseTarget : TChooseTarget;
        Action : PMacroAction;
begin
        Passt := False;
        for I := 0 to MacroListe.Count-1 do begin
                Entry := PMacroEntry( MacroListe.Items[ I ] );
                if (Entry.Alt = Alt) and (Entry.Strg = Strg) and (Entry.Shift = Shift) then begin
                        case Key of
                                VK_F1..VK_F24 : begin
                                                        if (Entry.Key[1] = 'F') and (StrToInt( Copy( Entry.Key, 2, 2 ) ) = Key-111) then
                                                                Passt := True;
                                                end;
                                else begin
                                        if Chr( Key ) = Entry.Key then begin
                                                Passt := True;
                                        end;
                                end;
                        end;
                        if Passt then
                                break;
                end;
        end;
        
        Action := Entry.Action;

        if Passt then begin
        	while Action <> nil do begin 
                	if UpperCase( Copy( Action.Action, 1, 4 ) ) = 'SAY ' then begin
                        	if not Palanthir.CheckForMacros( Copy( Action.Action, 5, Length( Action.Action )-4 ) ) then begin
                                	SpeechRequest := TSendSpeechRequest.Create;
	                                SpeechRequest.SetType( KB_Speech_Regular );
        	                        SpeechRequest.SetColor( Palanthir.Keyboard.SpeechColor );
                	                SpeechRequest.SetMsg( Copy( Action.Action, 5, Length( Action.Action )-4 ) );
                        	        SpeechRequest.SetFont( 3 );
                                	SpeechRequest.SetLanguage( Palanthir.Keyboard.Language );
	                                Palanthir.NetClient.Send( SpeechRequest );
        	                end;
                	end
	                else if UpperCase( Copy( Action.Action, 1, 6 ) ) = 'EMOTE ' then begin
        	                SpeechRequest := TSendSpeechRequest.Create;
                	        SpeechRequest.SetType( KB_Speech_Emote );
                        	SpeechRequest.SetColor( Palanthir.Keyboard.EmoteColor );
	                        SpeechRequest.SetMsg( '*' + Copy( Action.Action, 6, Length( Action.Action )-4 ) + '*' );
        	                SpeechRequest.SetFont( 3 );
                	        SpeechRequest.SetLanguage( Palanthir.Keyboard.Language );
                        	Palanthir.NetClient.Send( SpeechRequest );
	                end
        	        else if UpperCase( Copy( Action.Action, 1, 8 ) ) = 'WHISPER ' then begin
                	        SpeechRequest := TSendSpeechRequest.Create;
                        	SpeechRequest.SetType( KB_Speech_Whisper );
	                        SpeechRequest.SetColor( Palanthir.Keyboard.SpeechColor );
        	                SpeechRequest.SetMsg( Copy( Action.Action, 5, Length( Action.Action )-8 ) );
                	        SpeechRequest.SetFont( 3 );
                        	SpeechRequest.SetLanguage( Palanthir.Keyboard.Language );
	                        Palanthir.NetClient.Send( SpeechRequest );
        	        end
                	else if UpperCase( Copy( Action.Action, 1, 5 ) ) = 'YELL ' then begin
                        	SpeechRequest := TSendSpeechRequest.Create;
	                        SpeechRequest.SetType( KB_Speech_Yell );
        	                SpeechRequest.SetColor( Palanthir.Keyboard.SpeechColor );
                	        SpeechRequest.SetMsg( Copy( Action.Action, 5, Length( Action.Action )-5 ) );
                        	SpeechRequest.SetFont( 3 );
	                        SpeechRequest.SetLanguage( Palanthir.Keyboard.Language );
        	                Palanthir.NetClient.Send( SpeechRequest );
                	end
	                else if UpperCase( Copy( Action.Action, 1, 12 ) ) = 'OPEN JOURNAL' then begin
        	                Palanthir.Player.Journal.Visible := True;
                	end
	                else if UpperCase( Copy( Action.Action, 1, 14 ) ) = 'OPEN PAPERDOLL' then begin
        	                Palanthir.Player.RequestPaperdoll;
                	end
	                else if UpperCase( Copy( Action.Action, 1, 13 ) ) = 'OPEN OVERVIEW' then begin
        	                if Palanthir.Player.Radar.Visible then
                	                Palanthir.Player.Radar.SetSize( not Palanthir.Player.Radar.Large )
                        	else
                                	Palanthir.Player.Radar.Visible := True;
	                end
        	        else if UpperCase( Copy( Action.Action, 1, 13 ) ) = 'OPEN BACKPACK' then begin
                	        BackPack := Palanthir.Player.GetItemOnLayer( Layer_Backpack );
                        	if BackPack <> nil then begin
                                	SendDoubleClick := TSendDoubleClick.Create;
	                                SendDoubleClick.SetSerial( BackPack.Serial );
        	                        Palanthir.NetClient.Send( SendDoubleClick );
                	        end;
	                end	
        	        else if UpperCase( Copy( Action.Action, 1, 10 ) ) = 'LASTTARGET' then begin
                	end
	                else if UpperCase( Copy( Action.Action, 1, 10 ) ) = 'TARGETSELF' then begin
        	                if Palanthir.Target.Activated then begin
                	                ChooseTarget := TChooseTarget.Create;
                        	        ChooseTarget.SetType( 0 );
                                	ChooseTarget.SetSerial( Palanthir.Target.Serial );
	                                ChooseTarget.SetCursorType( 0 );
        	                        ChooseTarget.SetTargetSerial( Palanthir.Player.Serial );
                	                ChooseTarget.SetX( Palanthir.Player.Pos.X );
                        	        ChooseTarget.SetY( Palanthir.Player.Pos.Y );
                                	ChooseTarget.SetZ( Palanthir.Player.Pos.Z );
	                                ChooseTarget.SetId( Palanthir.Player.ID );
        	                        Palanthir.SendPacket( ChooseTarget );
                	                Palanthir.Target.Activated := False;
                        	        Palanthir.CursorState := Cursor_Standard;
	                        end;
        	        end
                	else if UpperCase( Copy( Action.Action, 1, 182 ) ) = 'OPEN CONFIGURATION' then begin
	                end;
	                
	                Action := Action.NextMacro;
	        end;
		Result := True;
        end
        else begin
                Result := False;
        end;
end;

procedure TShard.CreateDefaultLichtTypListe( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// List of all LightTypes and their specific Values' );
        FStream.Add( '// A LightType consists of and id and an array of 32 values for each color (rgb)' );
        FStream.Add( '// Each pixel ig is caculated this way:' );
        FStream.Add( '// ColorComponent = ColorComponentWithoutLight * ( (32 - GlobalLight + PersonalLight)*8 + SumOfAllLightValuesAtThisPoint ) / 255' );
        FStream.Add( '// The value to use is given by light.mul, where each pixel of a light has a corresponding value between 0 and 31 (both included)' );
        FStream.Add( '// Format:' );
        FStream.Add( '// <ColorId>' );
        FStream.Add( '// <RedEntry0>,<RedEntry1>,<RedEntry2>,...,<RedEntry31>' );
        FStream.Add( '// <GreenEntry0>,<GreenEntry1>,<GreenEntry2>,...,<GreenEntry31>' );
        FStream.Add( '// <BlueEntry0>,<BlueEntry1>,<BlueEntry2>,...,<BlueEntry31>' );
        FStream.Add( '' );

        FStream.Add( '//Default Color' );
        FStream.Add( '0' );
        FStream.Add( '0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38, 0x40, 0x48, 0x50, 0x58, 0x60, 0x68, 0x70, 0x78, 0x80, 0x88, 0x90, 0x98, 0xA0, 0xA8, 0xB0, 0xB8, 0xC0, 0xC8, 0xD0, 0xD8, 0xE0, 0xE8, 0xF0, 0xFF' );
        FStream.Add( '0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38, 0x40, 0x48, 0x50, 0x58, 0x60, 0x68, 0x70, 0x78, 0x80, 0x88, 0x90, 0x98, 0xA0, 0xA8, 0xB0, 0xB8, 0xC0, 0xC8, 0xD0, 0xD8, 0xE0, 0xE8, 0xF0, 0xFF' );
        FStream.Add( '0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38, 0x40, 0x48, 0x50, 0x58, 0x60, 0x68, 0x70, 0x78, 0x80, 0x88, 0x90, 0x98, 0xA0, 0xA8, 0xB0, 0xB8, 0xC0, 0xC8, 0xD0, 0xD8, 0xE0, 0xE8, 0xF0, 0xFF' );
        FStream.Add( '' );

        FStream.Add( '//Torches' );
        FStream.Add( '1' );
        FStream.Add( '0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF' );
        FStream.Add( '0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38, 0x40, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0' );
        //FStream.Add( '0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF' );
        //FStream.Add( '0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0' );
        FStream.Add( '0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadLichtTypListe( FileName : String );
var     Liste, SubListe : TStringList;
        I, J, SPos, Typ : Integer;
        Zeile : String;
        Entry : PLightTypEntry;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        Typ := 0;

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                Case Typ of
                                        0 : begin
                                                New( Entry );
                                                Entry.Id := StrToInt( Zeile );
                                                Typ := 1;
                                                end;
                                        1 : begin
                                                SubListe := TStringList.Create;
                                                SubListe.Delimiter := ',';
                                                SubListe.DelimitedText := Zeile;

                                                if SubListe.Count < 32 then begin
                                                        Log.Write( Format( 'Bug in lighttypelist.txt in line %d: not enough entries', [I] ) );
                                                        for J := 0 to 31 do
                                                                Entry.Red[ J ] := 0;
                                                end
                                                else begin
                                                        for J := 0 to 31 do
                                                                Entry.Red[ J ] := StrToInt( SubListe.Strings[ J ] );
                                                end;
                                                SubListe.Free;
                                                Typ := 2;
                                                end;
                                        2 : begin
                                                SubListe := TStringList.Create;
                                                SubListe.Delimiter := ',';
                                                SubListe.DelimitedText := Zeile;

                                                if SubListe.Count < 32 then begin
                                                        Log.Write( Format( 'Bug in lighttypelist.txt in line %d: not enough entries', [I] ) );
                                                        for J := 0 to 31 do
                                                                Entry.Green[ J ] := 0;
                                                end
                                                else begin
                                                        for J := 0 to 31 do
                                                                Entry.Green[ J ] := StrToInt( SubListe.Strings[ J ] );
                                                end;
                                                SubListe.Free;
                                                Typ := 3;
                                                end;
                                        3 : begin
                                                SubListe := TStringList.Create;
                                                SubListe.Delimiter := ',';
                                                SubListe.DelimitedText := Zeile;

                                                if SubListe.Count < 32 then begin
                                                        Log.Write( Format( 'Bug in lighttypelist.txt in line %d: not enough entries', [I] ) );
                                                        for J := 0 to 31 do
                                                                Entry.Blue[ J ] := 0;
                                                end
                                                else begin
                                                        for J := 0 to 31 do
                                                                Entry.Blue[ J ] := StrToInt( SubListe.Strings[ J ] );
                                                end;
                                                SubListe.Free;
                                                Typ := 0;

                                                LichtTypListe.Add( Entry );
                                                end;
                                end;
                        except
                                Log.Write( Format( 'Bug in lighttypelist.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

procedure TShard.CreateDefaultLichtListe( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// List of all lights and their type' );
        FStream.Add( '// to change and add types, see lighttypelist.txt' );
        FStream.Add( '// Default type for all lights is 0' );
        FStream.Add( '// Format: <ArtId> <ColorType>' );
        FStream.Add( '' );

        FStream.Add( '// Torches & Candles' );
        FStream.Add( '0x9FD 1' );
        FStream.Add( '0x9FE 1' );
        FStream.Add( '0x9FF 1' );

        FStream.Add( '0xA02 1' );
        FStream.Add( '0xA03 1' );
        FStream.Add( '0xA04 1' );

        FStream.Add( '0xA07 1' );
        FStream.Add( '0xA08 1' );
        FStream.Add( '0xA09 1' );

        FStream.Add( '0xA0C 1' );
        FStream.Add( '0xA0D 1' );
        FStream.Add( '0xA0E 1' );

        FStream.Add( '0xA0F 1' );
        FStream.Add( '0xA10 1' );
        FStream.Add( '0xA11 1' );
        FStream.Add( '0xA12 1' );
        FStream.Add( '0xA13 1' );
        FStream.Add( '0xA14 1' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadLichtListe( FileName : String );
var     Liste, SubListe : TStringList;
        I, SPos : Integer;
        Zeile : String;
        Entry : PLightEntry;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                SubListe := TStringList.Create;
                                SubListe.Delimiter := ' ';
                                SubListe.DelimitedText := Zeile;

                                New( Entry );
                                Entry.Id := StrToInt( SubListe.Strings[ 0 ] );
                                Entry.Color := StrToInt( SubListe.Strings[ 1 ] );

                                SubListe.Free;

                                LichtListe.Add( Entry );
                        except
                                Log.Write( Format( 'Bug in lightlist.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

function TShard.GetLightColor( Id : Word ) : Word;
var     I : Integer;
begin
        Result := 0;
        for I := 0 to LichtListe.Count-1 do
                if PLightEntry( LichtListe.Items[ I ] ).Id = Id then begin
                        Result := PLightEntry( LichtListe.Items[ I ] ).Color;
                        break;
                end;
end;

function TShard.GetLightColorEntry( Color : Word ) : PLightTypEntry;
var     I : Integer;
begin
        for I := 0 to LichtTypListe.Count-1 do
                if PLightTypEntry( LichtTypListe.Items[ I ] ).Id = Color then begin
                        Result := PLightTypEntry( LichtTypListe.Items[ I ] );
                        exit;
                end;

        Log.Write( Format( 'Lighttype %d not found', [Color] ) );

        Result := nil;
end;

procedure TShard.CreateDefaultBodyConv( UOPath, ShardDir : String );
var     Liste : TStringList;
begin
        Liste := TStringList.Create;
        if FileExists( UOPath + 'Bodyconv.def' ) then begin
                Liste.LoadFromFile( UOPath + 'Bodyconv.def' );
        end
        else begin
                Liste.Add( '# This file overloads the data loaded from the CD/HD cache system from a very' );
                Liste.Add( '# low level. Given an id, it will return the index in' );
                Liste.Add( '# anim2.mul/anim3.mul/etc the game should use instead of animation from anim.mul' );
                Liste.Add( '#' );
                Liste.Add( '# The maximum value for an index is 2048.' );
                Liste.Add( '#' );
                Liste.Add( '# <id the animation is used as> <id of anim in anim2> <id of anim in anim3> <id of anim in anim4>' );
                Liste.Add( '#' );
        end;

        Liste.SaveToFile( ShardDir + 'Bodyconv.def' );
        Liste.Free;
end;

procedure TShard.LoadBodyConv( FileName : String );
var     Liste, SubListe : TStringList;
        I : Integer;
        Zeile : String;
        TargetId, Anim2, Anim3, Anim4, Anim5 : Integer;
begin
        for I := 0 to $FFFF do begin
                AnimDefs[ I ].Id := I;
                AnimDefs[ I ].FileNr := 0;
                AnimDefs[ I ].Color := 0;
                case I of
                        0..199: AnimDefs[ I ].AnimTyp := aHighDetailed;
                        200..399: AnimDefs[ I ].AnimTyp := aLowDetailed;
                        400..$FFFF: AnimDefs[ I ].AnimTyp := aPeople;
                end;
        end;

        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];
                if (Zeile = '') or (Zeile[1] = '#') or (Zeile[1] = '"') then
                        continue;

                SubListe := TStringList.Create;
                SubListe.Delimiter := Chr( VK_TAB );
                SubListe.DelimitedText := Zeile;

                try
                        TargetId := StrToInt( SubListe.Strings[ 0 ] );
                        if SubListe.Count > 1 then
                                Anim2 := StrToInt( SubListe.Strings[ 1 ] )
                        else
                                Anim2 := -1;
                        if SubListe.Count > 2 then
                                Anim3 := StrToInt( SubListe.Strings[ 2 ] )
                        else
                                Anim3 := -1;
                        if SubListe.Count > 3 then
                                Anim4 := StrToInt( SubListe.Strings[ 3 ] )
                        else
                                Anim4 := -1;
                        if SubListe.Count > 4 then
                                Anim5 := StrToInt( SubListe.Strings[ 4 ] )
                        else
                                Anim5 := -1;

                        if Anim2 > -1 then begin
                                AnimDefs[ TargetId ].FileNr := 2;
                                AnimDefs[ TargetId ].Id := Anim2;
                                case I of
                                        0..199: AnimDefs[ TargetId ].AnimTyp := aHighDetailed;
                                        200..399: AnimDefs[ TargetId ].AnimTyp := aLowDetailed;
                                        400..$FFFF: AnimDefs[ TargetId ].AnimTyp := aPeople;
                                end;
                        end
                        else if Anim3 > -1 then begin
                                AnimDefs[ TargetId ].FileNr := 3;
                                AnimDefs[ TargetId ].Id := Anim3;
                                case I of
                                        0..419: AnimDefs[ TargetId ].AnimTyp := aHighDetailed;
                                        420..489: AnimDefs[ TargetId ].AnimTyp := aLowDetailed;
                                        490..$FFFF: AnimDefs[ TargetId ].AnimTyp := aPeople;
                                end;
                        end
                        else if Anim4 > -1 then begin
                                AnimDefs[ TargetId ].FileNr := 4;
                                AnimDefs[ TargetId ].Id := Anim4;
                                case I of
                                        0..199: AnimDefs[ TargetId ].AnimTyp := aHighDetailed;
                                        200..399: AnimDefs[ TargetId ].AnimTyp := aLowDetailed;
                                        400..$FFFF: AnimDefs[ TargetId ].AnimTyp := aPeople;
                                end;
                        end
                        else if Anim5 > -1 then begin
                                AnimDefs[ TargetId ].FileNr := 4;
                                AnimDefs[ TargetId ].Id := Anim5;
                                case I of
                                        0..199: AnimDefs[ TargetId ].AnimTyp := aHighDetailed;
                                        200..399: AnimDefs[ TargetId ].AnimTyp := aLowDetailed;
                                        400..$FFFF: AnimDefs[ TargetId ].AnimTyp := aPeople;
                                end;
                        end;
                except
                        Log.Write( Format( 'Bug in bodyconv.def, line %d', [I] ), True );
                end;

                SubListe.Free;
        end;

        Liste.Free;
end;

procedure TShard.CreateDefaultBodyDef( UOPath, ShardDir : String );
var     Liste : TStringList;
begin
        Liste := TStringList.Create;
        if FileExists( UOPath + 'Body.def' ) then begin
                Liste.LoadFromFile( UOPath + 'Body.def' );
        end
        else begin
                Liste.Add( '# Format is: <ORIG BODY> {<NEW BODY>} <NEW HUE>' );
                Liste.Add( '' );
        end;

        Liste.SaveToFile( ShardDir + 'Body.def' );
        Liste.Free;
end;

procedure TShard.LoadBodyDef( FileName : String );
var     Liste : TStringList;
        I, Pos1, Pos2, Pos3 : Integer;
        Zeile, Tmp : String;
        OldId, NewId, NewHue : Word;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];
                if (Zeile = '') or (Zeile[1] = '#') or (Zeile[1] = '"') then
                        continue;

                try
                        Pos1 := Pos( '{', Zeile );
                        Pos2 := Pos( '}', Zeile );

                        OldId := StrToInt( Trim( Copy( Zeile, 1, Pos1-1 ) ) );

                        Tmp := Copy( Zeile, Pos1+1, Pos2-Pos1-1 );
                        Pos3 := Pos( ',', Tmp );
                        if Pos3 = 0 then begin
                                NewId := StrToInt( Tmp );
                        end else begin
                                NewId := StrToInt( Trim( Copy( Tmp, 1, Pos3-1 ) ) );
                        end;

                        NewHue := StrToInt( Trim( Copy( Zeile, Pos2+1, Length( Zeile )-Pos2 ) ) );

                        AnimDefs[ OldId ].Id := AnimDefs[ NewId ].Id;
                        AnimDefs[ OldId ].FileNr := AnimDefs[ NewId ].FileNr;
                        AnimDefs[ OldId ].AnimTyp := AnimDefs[ NewId ].AnimTyp;
                        AnimDefs[ OldId ].Color := NewHue;
                except
                        Log.Write( Format( 'Bug in Body.def, line %d', [I+1] ), True );
                end;
        end;

        Liste.Free;
end;

function TShard.GetAnimRecord( Id : Word ) : TAnimRecord;
begin
        Result := AnimDefs[ Id ];
end;

procedure TShard.CreateDefaultChairList( FileName : String );
var     Liste : TStringList;
begin
        Liste := TStringList.Create;
        Liste.Add( '//Chairfile, contains ArtIds of all chairs' );
        Liste.Add( '//Directions are:' );
        Liste.Add( '//Char looking to the upperleft corner: 6 ' );
        Liste.Add( '//Char looking to the upperright corner: 0 ' );
        Liste.Add( '//Char looking to the lowerright corner: 2 ' );
        Liste.Add( '//Char looking to the lowerleft corner: 4 ' );
        Liste.Add( '//Format: <ArtId> <PossibleDirection1> [<PossibleDirection2> <PossibleDirection3> <PossibleDirection4>]' );
        Liste.Add( '' );

        Liste.Add( '0x459 4 0' );
        Liste.Add( '0x45A 2 6' );
        Liste.Add( '0xA2A 6 0 2 4' );
        Liste.Add( '0xA2B 6 0 2 4' );
        Liste.Add( '0xB2C 6 2' );
        Liste.Add( '0xB2D 0 4' );
        Liste.Add( '0xB2E 4' );
        Liste.Add( '0xB2F 2' );
        Liste.Add( '0xB30 0' );
        Liste.Add( '0xB31 6' );
        Liste.Add( '0xB32 4' );
        Liste.Add( '0xB33 2' );
        Liste.Add( '0xB4E 2' );
        Liste.Add( '0xB4F 4' );
        Liste.Add( '0xB50 0' );
        Liste.Add( '0xB51 6' );
        Liste.Add( '0xB52 2' );
        Liste.Add( '0xB53 4' );
        Liste.Add( '0xB54 0' );
        Liste.Add( '0xB55 6' );
        Liste.Add( '0xB56 2' );
        Liste.Add( '0xB57 4' );
        Liste.Add( '0xB58 6' );
        Liste.Add( '0xB59 0' );
        Liste.Add( '0xB5A 2' );
        Liste.Add( '0xB5B 4' );
        Liste.Add( '0xB5C 0' );
        Liste.Add( '0xB5D 6' );
        Liste.Add( '0xB5E 6 0 4 2' );
        Liste.Add( '0xB5F 6 2' );
        Liste.Add( '0xB60 6 2' );
        Liste.Add( '0xB61 6 2' );
        Liste.Add( '0xB62 6 2' );
        Liste.Add( '0xB63 6 2' );
        Liste.Add( '0xB64 6 2' );
        Liste.Add( '0xB65 4 0' );
        Liste.Add( '0xB66 4 0' );
        Liste.Add( '0xB67 4 0' );
        Liste.Add( '0xB68 4 0' );
        Liste.Add( '0xB69 4 0' );
        Liste.Add( '0xB6A 4 0' );
        Liste.Add( '0xB91 4' );
        Liste.Add( '0xB92 4' );
        Liste.Add( '0xB93 2' );
        Liste.Add( '0xB94 2' );
        Liste.Add( '0x1049 6 2' );
        Liste.Add( '0x104A 4 0' );
        Liste.Add( '0x11FC  2 4 6 0' );
        Liste.Add( '0x1207 4 0' );
        Liste.Add( '0x1208 4 0 ' );
        Liste.Add( '0x1209 4 0' );
        Liste.Add( '0x120A 2 6' );
        Liste.Add( '0x120B 2 6' );
        Liste.Add( '0x120C 2 6' );
        Liste.Add( '0x1218 4' );
        Liste.Add( '0x1219 2' );
        Liste.Add( '0x121A 0' );
        Liste.Add( '0x121B 6' );
        Liste.Add( '0x1317 4' );
        Liste.Add( '0x1527 6 2' );
        Liste.Add( '0x1DC7 6 2' );
        Liste.Add( '0x1DC8 6 2' );
        Liste.Add( '0x1DC9 6 2' );
        Liste.Add( '0x1DCA 0 4' );
        Liste.Add( '0x1DCB 0 4' );
        Liste.Add( '0x1DCC 0 4' );
        Liste.Add( '0x1DCD 6 2' );
        Liste.Add( '0x1DCE 6 2' );
        Liste.Add( '0x1DCF 6 2' );
        Liste.Add( '0x1DD0 0 4' );
        Liste.Add( '0x1DD1 0 4' );
        Liste.Add( '0x1DD2 0 4' );
        Liste.Add( '0x290C 2 //Elben' );
        Liste.Add( '0x290D 4 //Elben' );
        Liste.Add( '0x290E 0 //Elben' );
        Liste.Add( '0x290F 6 //Elben' );
        Liste.Add( '0x2B9C 0 //Elben' );
        Liste.Add( '0x2B9D 6 //Elben' );
        Liste.Add( '0x2B9E 4 //Elben' );
        Liste.Add( '0x2B9F 2 //Elben' );
        Liste.Add( '0x38C5 2 4 6 0' );
        Liste.Add( '0x3DFF 4 0' );
        Liste.Add( '0x3E00 6 2' );

        Liste.SaveToFile( FileName );
        Liste.Free;
end;

procedure TShard.LoadChairList( FileName : String );
var     Id : Integer;
        Liste, SubListe : TStringList;
        I, J, SPos : Integer;
        Zeile : String;
        Dir0, Dir2, Dir4, Dir6 : Boolean; 
begin
        for I := 0 to $FFFF do
                StuhlListe[ I ].Chair := False;
        
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                SubListe := TStringList.Create;
                                SubListe.Delimiter := ' ';
                                SubListe.DelimitedText := Zeile;

                                Id := StrToInt( SubListe.Strings[ 0 ] );

                                Dir0 := False;
                                Dir2 := False;
                                Dir4 := False;
                                Dir6 := False;

                                for J := 1 to 4 do begin
                                        if SubListe.Count > J then begin
                                                Case StrToInt( SubListe.Strings[ J ] ) of
                                                        0 : Dir0 := True;
                                                        2 : Dir2 := True;
                                                        4 : Dir4 := True;
                                                        6 : Dir6 := True;
                                                end;
                                        end;
                                end;

                                StuhlListe[ Id ].Chair := True;
                                StuhlListe[ Id ].Dir0 := Dir0;
                                StuhlListe[ Id ].Dir2 := Dir2;
                                StuhlListe[ Id ].Dir4 := Dir4;
                                StuhlListe[ Id ].Dir6 := Dir6;
                        except
                                Log.Write( Format( 'Bug in chairlist.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

function TShard.GetChairDirection( Id : Word; CharDirection : Byte ) : Byte;
begin
        if (CharDirection = 0) and StuhlListe[ Id ].Dir0 then begin
                Result := 0;
                exit;
        end;
        if (CharDirection = 2) and StuhlListe[ Id ].Dir2 then begin
                Result := 2;
                exit;
        end;
        if (CharDirection = 4) and StuhlListe[ Id ].Dir4 then begin
                Result := 4;
                exit;
        end;
        if (CharDirection = 6) and StuhlListe[ Id ].Dir6 then begin
                Result := 6;
                exit;
        end;

        if StuhlListe[ Id ].Dir0 and ( CharDirection in [7,0,1] ) then begin
                Result := 0;
                exit;
        end else if StuhlListe[ Id ].Dir2 and ( CharDirection in [1,2,3] ) then begin
                Result := 2;
                exit;
        end else if StuhlListe[ Id ].Dir4 and ( CharDirection in [3,4,5] ) then begin
                Result := 4;
                exit;
        end else if StuhlListe[ Id ].Dir6 and ( CharDirection in [5,6,7] ) then begin
                Result := 6;
                exit;                
        end;

        if StuhlListe[ Id ].Dir0 then begin
                Result := 0;
        end else if StuhlListe[ Id ].Dir2 then begin
                Result := 2;
        end else if StuhlListe[ Id ].Dir4 then begin
                Result := 4;
        end else if StuhlListe[ Id ].Dir6 then begin
                Result := 6;
        end else begin
                Result := 0;
        end;
end;

procedure TShard.CreateDefaultAnimation2PaperdollList( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// File to define which gumps should be used in the paperdoll' );
        FStream.Add( '// for specific animations' );
        FStream.Add( '// Default gump for all animations is none (-1)' );
        FStream.Add( '// Format: <AnimationId> <GumpId>' );
        FStream.Add( '' );

        FStream.Add( '//Man' );
        FStream.Add( '0x190 0xC' );

        FStream.Add( '//Woman' );
        FStream.Add( '0x191 0xD' );

        FStream.Add( '//Blackthorne' );
        FStream.Add( '0x3DF 0xC' );

        FStream.Add( '//Counselor' );
        FStream.Add( '0x3DB 0xC' );

        FStream.Add( '//Dupre' );
        FStream.Add( '0x3E2 0xC' );

        FStream.Add( '//Ghost' );
        FStream.Add( '0x3CA 0xC' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadAnimation2PaperdollList( FileName : String );
var     Liste, SubListe : TStringList;
        I, SPos : Integer;
        Zeile : String;
        Entry : PAnim2PaperdollEntry;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                SubListe := TStringList.Create;
                                SubListe.Delimiter := ' ';
                                SubListe.DelimitedText := Zeile;

                                new( Entry );
                                Entry.AnimId := StrToInt( SubListe.Strings[ 0 ] );
                                Entry.GumpId := StrToInt( SubListe.Strings[ 1 ] );

                                Anim2PaperdollListe.Add( Entry );
                        except
                                Log.Write( Format( 'Bug in anim2paperdoll.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

function TShard.GetAnimGumpId( AnimId : Word ) : Integer;
var     I : Integer;
begin
        for I := 0 to Anim2PaperdollListe.Count-1 do
                if PAnim2PaperdollEntry( Anim2PaperdollListe.Items[ I ] ).AnimId = AnimId then begin
                        Result := PAnim2PaperdollEntry( Anim2PaperdollListe.Items[ I ] ).GumpId;
                        exit;
                end;

        Result := -1;
end;

procedure TShard.CreateDefaultAmountList( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// File to define which special Ids with Amount > 1 should have' );
        FStream.Add( '// if there is none, the client will just double the picture' );
        FStream.Add( '// Format: <BaseId> <MinimumAmountForThisId> <NewId>' );
        FStream.Add( '' );

        FStream.Add( '//Gold' );
        FStream.Add( '0xEED 2 0xEEE' );
        FStream.Add( '0xEED 6 0xEEF' );

        FStream.Add( '//Gold' );
        FStream.Add( '0xEEA 2 0xEEB' );
        FStream.Add( '0xEEA 6 0xEEC' );

        FStream.Add( '//Silver' );
        FStream.Add( '0xEF0 2 0xEF1' );
        FStream.Add( '0xEF0 6 0xEF2' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadAmountList( FileName : String );
var     Liste, SubListe : TStringList;
        I, SPos : Integer;
        Zeile : String;
        Entry : PAmountEntry;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                SubListe := TStringList.Create;
                                SubListe.Delimiter := ' ';
                                SubListe.DelimitedText := Zeile;

                                new( Entry );
                                Entry.BaseId := StrToInt( SubListe.Strings[ 0 ] );
                                Entry.Amount := StrToInt( SubListe.Strings[ 1 ] );
                                Entry.NewId := StrToInt( SubListe.Strings[ 2 ] );

                                AmountList.Add( Entry );
                        except
                                Log.Write( Format( 'Bug in amount.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

function TShard.GetAmountId( BaseId, Amount : Word ) : Word;
var     I : Integer;
        MaxAmount : Word;
begin
        MaxAmount := 0;
        Result := BaseId;
        for I := 0 to AmountList.Count-1 do begin
                if (PAmountEntry( AmountList.Items[ I ] ).BaseId = BaseId) and (PAmountEntry( AmountList.Items[ I ] ).Amount <= Amount) and (PAmountEntry( AmountList.Items[ I ] ).Amount > MaxAmount) then begin
                        if PAmountEntry( AmountList.Items[ I ] ).Amount = Amount then begin
                                Result := PAmountEntry( AmountList.Items[ I ] ).NewId;
                                exit;
                        end
                        else begin
                                MaxAmount := PAmountEntry( AmountList.Items[ I ] ).Amount;
                                Result := PAmountEntry( AmountList.Items[ I ] ).NewId;
                        end;
                end;
        end;
end;

procedure TShard.CreateDefaultInvisList( FileName : String );
var     FStream : TStringList;
begin
        FStream := TStringList.Create;
        FStream.Add( '// List of Static Art Ids that are not invisible' );
        FStream.Add( '' );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

procedure TShard.LoadInvisList( FileName : String );
var     Liste : TStringList;
        I, SPos : Integer;
        Zeile : String;
begin
        Liste := TStringList.Create;
        Liste.LoadFromFile( FileName );

        for I := 0 to Liste.Count-1 do begin
                Zeile := Liste.Strings[ I ];

                if Zeile = '' then
                        continue;

                SPos := Pos( '//', Zeile );
                if SPos = 1 then
                        continue;
                if SPos > 0 then
                        Zeile := Copy( Zeile, 1, SPos-1 );

                if Zeile <> '' then begin
                        try
                                Palanthir.AddToInvisList( StrToInt( Zeile ) );
                        except
                                Log.Write( Format( 'Bug in invislist.txt in line %d', [I] ) );
                        end;
                end;
        end;
        Liste.Free;
end;

procedure TShard.SaveInvisList( FileName : String );
var     I : Integer;
        FStream : TStringList;
begin
        if FileExists( FileName ) then
                DeleteFile( FileName );

        FStream := TStringList.Create;
        FStream.Add( '// List of Static Art Ids that are not invisible' );
        FStream.Add( '' );

        for I := 0 to Palanthir.GetInvisListCount-1 do
                FStream.Add( Format( '0x%x', [Palanthir.GetInvisListId( I )] ) );

        FStream.SaveToFile( FileName );
        FStream.Free;
end;

end.
