unit uSpeech;

interface

uses    Classes, SysUtils, uTexCache, uUtilities, uRenderer, uBasicDrawObject,
        Math;

const   SPTyp_None = 0;
        SPTyp_DrawObject = 1;
        SPTyp_GumpObject = 2;

type    TSpeechText = Class;
        TSpeech = Class;

        TSpeechQueueEntry = Class
                public
                        Font : Word;
                        Color : Word;
                        Text : String;
                        MaxLength : Word;
                        Typ : Byte;
                        NewTimer : Integer;
                        Unicode : Boolean;
                        constructor Create( vFont, vColor : Word; vText : String; vMaxLength : Word; vTyp : Byte; vNewTimer : Integer; vUnicode : Boolean = True );
                        destructor Free;
        end;
        
        TSpeechQueue = Class
                private
                        mSpeech : TSpeech;
                        mCache : TList;
                        mNextPoll : LongWord;
                public
                        constructor Create( vSpeech : TSpeech );
                        destructor Free;
                        procedure AddSpeech( Font, Color : Word; Text : String; MaxLength : Word; Typ : Byte; NewTimer : Integer; Unicode : Boolean = True );
                        procedure Poll;
        end;

        TSpeech = Class
                private
                        Typ : Byte;
                        Cache : TList;
                public
                        Queue : TSpeechQueue;                
                        Visible : Boolean;
                        UObject : Pointer;
                        X, Y : Integer;
                        constructor Create;
                        destructor Free;
                        procedure AddSpeech( Speech : TSpeech );
                        function Draw : Boolean;
                        procedure SetDrawObjectType;
                        procedure SetGumpObjectType;
                        function IsDrawObjectType : Boolean;
                        function IsGumpObjectType : Boolean;
                        procedure Add( SpeechText : TSpeechText );
                        function GetSpeechText( Index : Integer ) : TSpeechText;
                        function GetCacheCount : Integer;
                        procedure Delete( Index : Integer );
        end;

        TSpeechText = Class( TBasicDrawObject )
                private
                        _Text : String;
                        _Color, _Font : Word;
                        _SpeechTyp : Byte;
                        _MaxLength : Word;
                        _Count : Integer;
                        _Unicode : Boolean;
                        procedure SetCount( Count : Integer );
                public
                        Speech : TSpeech;
                        Timer : Integer;
                        Texture, SelTexture : TTexObject;
                        property Text : String read _Text;
                        property Count : Integer read _Count write SetCount;
                        constructor Create( Font, Color : Word; Text : String; MaxLength : Word; Typ : Byte; NewTimer : Integer; Unicode : Boolean = True ); reintroduce; overload;
                        constructor Create( Text : TSpeechText ); reintroduce; overload;
                        destructor Free; override;
                        function IsSpeech : Boolean; override;
                        procedure Draw( X, Y : Word );
                        procedure onClick;
                        procedure OnDblClick;
        end;

implementation

uses    uPalanthir, uBasicTypedefs, uGumps;

constructor TSpeechQueueEntry.Create( vFont, vColor : Word; vText : String; vMaxLength : Word; vTyp : Byte; vNewTimer : Integer; vUnicode : Boolean );
begin
        Font := vFont;
        Color := vColor;
        Text := vText;
        MaxLength := vMaxLength;
        Typ := vTyp;
        NewTimer := vNewTimer;
        Unicode := vUnicode;
end;

destructor TSpeechQueueEntry.Free;
begin
end;

constructor TSpeechQueue.Create( vSpeech : TSpeech );
begin
        mSpeech := vSpeech;
        mCache := TList.Create;
        mNextPoll := CustomGetTickCount;
end;

destructor TSpeechQueue.Free;
var     I : Integer;
begin
        for I := 0 to mCache.Count-1 do begin
                TSpeechQueueEntry( mCache.Items[ I ] ).Free;
        end;
        mCache.Free;
end;

procedure TSpeechQueue.AddSpeech( Font, Color : Word; Text : String; MaxLength : Word; Typ : Byte; NewTimer : Integer; Unicode : Boolean );
begin
        mCache.Add( TSpeechQueueEntry.Create( Font, Color, Text, MaxLength, Typ, NewTimer, Unicode ) );
end;

procedure TSpeechQueue.Poll;
var     I : Integer;
        NewText : TSpeechText;
        Entry : TSpeechQueueEntry;
begin
        if CustomGetTickCount < mNextPoll then begin
                Exit;
        end;

        I := 0;
        while( (I < 5) and (mCache.Count > 0) ) do begin
                Inc( I );
                Entry := TSpeechQueueEntry( mCache.Items[ 0 ] );
                mCache.Delete( 0 );

                NewText := TSpeechText.Create( Entry.Font, Entry.Color, Entry.Text, Entry.MaxLength, Entry.Typ, Entry.NewTimer, Entry.Unicode );
                NewText.Speech := mSpeech;
                mSpeech.Add( NewText );

                Entry.Free;
        end;

        mNextPoll := CustomGetTickCount + 1000;
end;

constructor TSpeech.Create;
begin
        Cache := TList.Create;
        UObject := nil;
        Typ := SPTyp_None;
        Visible := False;
        Queue := TSpeechQueue.Create( Self );
end;

destructor TSpeech.Free;
var     I : Integer;
begin
        Queue.Free;
        
        if UObject <> nil then begin
                case Typ of
                        SPTyp_DrawObject : TUODrawObject( UObject ).Speech := nil;
                        SPTyp_GumpObject : TGumpObject( UObject ).Speech := nil;
                end;
        end; 

        if Cache <> nil then begin
                for I := 0 to Cache.Count-1 do
                        TSpeechText( Cache.Items[ I ] ).Free;
                Cache.Free;
        end;
end;

procedure TSpeech.AddSpeech( Speech : TSpeech );
var     I : Integer;
        Text, NewText : TSpeechText;
begin
        for I := 0 to Speech.Cache.Count-1 do begin
                Text := TSpeechText( Speech.Cache.Items[ I ] );
                NewText := TSpeechText.Create( Text );
                NewText.Speech := Self;
                Cache.Add( NewText );
        end;
end;

function TSpeech.Draw : Boolean;
var     TmpY, J : Integer;
        Text : TSpeechText;
begin
        Queue.Poll;

        while Cache.Count > 6 do begin
                TSpeechText( Cache.Items[ 0 ] ).Free;
                Cache.Delete( 0 );
        end;

        if IsDrawObjectType then begin
                if Y > Palanthir.GameWindow.Y + Palanthir.GameWindow.Hoehe then
                        Y := Palanthir.GameWindow.Y + Palanthir.GameWindow.Hoehe - 2;
        end;

        TmpY := Y - 5;
        for J := Cache.Count-1 downto 0 do begin
                Text := TSpeechText( Cache.Items[ J ] );
                if CustomGetTickCount > Text.Timer then begin
                        Text.Free;
                        Cache.Delete( J );
                end
                else if Visible and IsDrawObjectType then begin
                        if X - Palanthir.GameWindow.X - (Text.Texture.Breite div 2) < 7 then
                                X := Palanthir.GameWindow.X + (Text.Texture.Breite div 2) + 7;
                        if X + (Text.Texture.Breite div 2) >= Palanthir.GameWindow.X + Palanthir.GameWindow.Breite - 7 then
                                X := Palanthir.GameWindow.X + Palanthir.GameWindow.Breite - (Text.Texture.Breite div 2) - 7;

                        TmpY := TmpY - Text.Texture.Hoehe - 5;

                        if TmpY - Palanthir.GameWindow.Y < Text.Texture.Hoehe + 5 then begin
                                Y := Y + ( Text.Texture.Hoehe + 5 - TmpY );
                                TmpY := TmpY + ( Text.Texture.Hoehe + 5 - TmpY );
                        end;
                end;
        end;

        TmpY := Y - 5;
        for J := Cache.Count-1 downto 0 do begin
                Text := TSpeechText( Cache.Items[ J ] );
                if Visible then begin
                        TmpY := TmpY - Text.Texture.Hoehe - 5;
                        Text.Draw( X - (Text.Texture.Breite div 2), TmpY );
                end;
        end;

        if (Cache.Count = 0) and (Queue.mCache.Count = 0) then begin
                Result := False;
                exit;
        end;

        Visible := False;

        Result := True;        
end;

procedure TSpeech.SetDrawObjectType;
begin
        Typ := SPTyp_DrawObject;
end;

procedure TSpeech.SetGumpObjectType;
begin
        Typ := SPTyp_GumpObject;
end;

function TSpeech.IsDrawObjectType : Boolean;
begin
        Result := Typ = SPTyp_DrawObject;
end;

function TSpeech.IsGumpObjectType : Boolean;
begin
        Result := Typ = SPTyp_GumpObject;
end;

procedure TSpeech.Add( SpeechText : TSpeechText );
var     I : Integer;
begin
        for I := 0 to Cache.Count-1 do begin
                if TSpeechText( Cache.Items[ I ] ).Text = SpeechText.Text then begin
                        TSpeechText( Cache.Items[ I ] ).Count := TSpeechText( Cache.Items[ I ] ).Count+1;
                        TSpeechText( Cache.Items[ I ] ).Timer := Max( TSpeechText( Cache.Items[ I ] ).Timer, SpeechText.Timer );
                        SpeechText.Free;
                        Exit;
                end;
        end;
        
        Cache.Add( SpeechText );
end;

function TSpeech.GetSpeechText( Index : Integer ) : TSpeechText;
begin
        if Index > Cache.Count-1 then
                Result := nil
        else
                Result := TSpeechText( Cache.Items[ Index ] );
end;

function TSpeech.GetCacheCount : Integer;
begin
        Result := Cache.Count;
end;

procedure TSpeech.Delete( Index : Integer );
begin
        if Index < Cache.Count-1 then begin
                TSpeechText( Cache.Items[ Index ] ).Free;
                Cache.Delete( Index );
        end;
end;

procedure TSpeechText.SetCount( Count : Integer );
begin
        _Count := Count;

        if (Count > 1) and Palanthir.ShowSpeechCount then begin
                Texture := Palanthir.Data.GetUnicodeTexture( _Font, Format( '%s (%d)', [_Text,_Count] ), _Color, _MaxLength );
                SelTexture := Palanthir.Data.GetUnicodeTexture( _Font, Format( '%s (%d)', [_Text,_Count] ), $9a, _MaxLength );
        end;
end;

constructor TSpeechText.Create( Font, Color : Word; Text : String; MaxLength : Word; Typ : Byte; NewTimer : Integer; Unicode : Boolean );
begin
        inherited Create;

        _Font := Font;
        _Color := Color;
        _Text := Text;
        _SpeechTyp := Typ;
        _MaxLength := MaxLength;
        _Unicode := Unicode;
        Speech := nil;
        Timer := CustomGetTickCount + NewTimer;

        if _Unicode then begin
                Texture := Palanthir.Data.GetUnicodeTexture( _Font, _Text, _Color, _MaxLength );
                SelTexture := Palanthir.Data.GetUnicodeTexture( _Font, _Text, $9a, _MaxLength );
        end
        else begin
                Texture := Palanthir.Data.GetAsciiTexture( _Font, _Text, _Color, _MaxLength );
                SelTexture := Palanthir.Data.GetAsciiTexture( _Font, _Text, $9a, _MaxLength );
        end;
end;

constructor TSpeechText.Create( Text : TSpeechText );
begin
        inherited Create;

        _Font := Text._Font;
        _Color := Text._Color;
        _Text := Text._Text;
        _SpeechTyp := Text._SpeechTyp;
        _MaxLength := Text._MaxLength;
        _Unicode := Text._Unicode;
        Timer := CustomGetTickCount + 3*1000;
        Count := 1;

        if _Unicode then begin
                Texture := Palanthir.Data.GetUnicodeTexture( _Font, _Text, _Color, _MaxLength );
                SelTexture := Palanthir.Data.GetUnicodeTexture( _Font, _Text, $9a, _MaxLength );
        end
        else begin
                Texture := Palanthir.Data.GetAsciiTexture( _Font, _Text, _Color, _MaxLength );
                SelTexture := Palanthir.Data.GetAsciiTexture( _Font, _Text, $9a, _MaxLength );
        end;
end;

destructor TSpeechText.Free;
begin
        if Texture <> nil then
                Texture.Free;
        if SelTexture <> nil then
                SelTexture.Free;

        inherited Free;                
end;

function TSpeechText.IsSpeech : Boolean;
begin
        Result := True;
end;

procedure TSpeechText.Draw( X, Y : Word );
begin
        if Palanthir.Maus.SelectedSpeech = Self then begin
                if SelTexture = nil then
                        exit;
                Renderer.DrawPixels( X, Y, SelTexture );
                Palanthir.Maus.CheckSpeech( X, Y, Self, SelTexture );

                Palanthir.BringToFront( Speech );
        end
        else begin
                if Texture = nil then
                        exit;
                Renderer.DrawPixels( X, Y, Texture );
                Palanthir.Maus.CheckSpeech( X, Y, Self, Texture );
        end;
end;

procedure TSpeechText.onClick;
begin
        if Speech <> nil then begin
                Palanthir.BringToFront( Speech );
        end;
end;

procedure TSpeechText.onDblClick;
begin
        if Speech <> nil then begin
                if Speech.IsDrawObjectType then begin
                        TUODrawObject( Speech.UObject ).OnDblClick;
                end;
        end;
end;

end.
