unit uWorldCache;

interface

uses    uObjectListe, uUObject, Dialogs, SysUtils, Classes, uPos, uLog, SyncObjs,
        uUtilities, uBasicTypedefs, uTexCache, dglOpenGL, uItem, uCounter;

type    TWorldCache = Class;

        TLoadBlockThread = Class( TThread )
                private
                        WorldCache : TWorldCache;
                protected
                        procedure Execute; override;
                        procedure Init( WorldCache : TWorldCache );
                public
                        constructor Create( CreateSuspended : Boolean );
                        destructor Free;
        end;


        TWorldBlock = Class
                public
                        MapListe : PMapListe;
                        StaticListe : PStaticListe;
                        StaticCount : LongWord;
                        ObjectListe : TObjectCache;
                        MultiCache : TList;
                        BlockID : LongWord;
                        Visible : Boolean;
                        constructor Create;
                        destructor Free;
                        procedure ClearItems;
                        procedure ClearStatics;
                        procedure ClearMap;
                        procedure ClearMapStatics;                        
                        function DistanceTo( Pos : TPos ) : Integer;
                        procedure AddStatic( StaticItem : TStaticItem );
                        procedure RemoveStatic( StaticItem : TStaticItem );
        end;

        TWorldListe = array[0..1] of TWorldBlock;

        PWorldListe = ^TWorldListe;

        TLoadBlock = Class
                public
                        BlockID : LongWord;
                        MapListe : PMapListe;
                        StaticListe : PStaticListe;
                        StaticCount : LongWord;
                        constructor Create;
                        destructor Free;
                        procedure FreeMemory;
        end;

        TWorldCache = Class
                private
                        Cache : PWorldListe;
                        Count : Integer;
                        LoadBlockList, LoadedBlockList : TList;
                        LoadBlockThread : TLoadBlockThread;                        
                        CriticalSection, CriticalSection2 : TCriticalSection;
                public
                        constructor Create;
                        destructor Free;
                        procedure AddObject( Block : TWorldBlock );
                        function GetObject( BlockID : LongWord; Immediately : Boolean = False ) : TWorldBlock;
                        function GetObjectWithoutLoad( BlockID : LongWord ) : TWorldBlock;
                        procedure DeleteObject( BlockID : LongWord ); overload;
                        procedure DeleteObject( Block : TWorldBlock ); overload;
                        procedure ShowBlock( BlockID : LongWord );
                        procedure HideBlock( BlockID : LongWord );
                        function BlockIsVisible( BlockID : LongWord ) : Boolean;
                        procedure SetAllInvis;
                        function GetMapHeight( X, Y : Word ) : ShortInt;
                        function GetStaticObjects( X, Y : Word ) : TList;
                        function GetItems( X, Y : Word ) : TList;
                        function GetChars( X, Y : Word ) : TList;                        
                        function GetMultiItems( X, Y : Word ) : TList;
                        function GetMapItem( X, Y : Word ) : TMapItem;
                        procedure CheckObjectDistances( BlockID : LongWord; Pos : TPos );
                        procedure GarbageCollection;
                        function GetCount : Integer;
                        procedure Reload;
                        procedure Poll;
                        procedure ClearMapStatics;
        end;

implementation

uses    uLoaderMap, uPalanthir;

procedure TLoadBlockThread.Execute;
var     Count : LongWord;
        Block : TLoadBlock;
begin
        while not Terminated do begin
                WorldCache.CriticalSection.Enter;
                Count := WorldCache.LoadBlockList.Count;
                WorldCache.CriticalSection.Leave;

                while (Count > 0) and (not Terminated) do begin
                        WorldCache.CriticalSection.Enter;
                        Block := TLoadBlock( WorldCache.LoadBlockList.Items[ 0 ] );
                        WorldCache.LoadBlockList.Delete( 0 );
                        WorldCache.CriticalSection.Leave;

                        Palanthir.Data.LoadMapBlock( Block );

                        WorldCache.CriticalSection.Enter;
                        WorldCache.LoadedBlockList.Add( Block );

                        Count := WorldCache.LoadBlockList.Count;
                        WorldCache.CriticalSection.Leave;
                end;

                Sleep( 10 );
        end;
end;

procedure TLoadBlockThread.Init( WorldCache : TWorldCache );
begin
        Self.WorldCache := WorldCache;
end;

constructor TLoadBlockThread.Create( CreateSuspended : Boolean );
begin
        inherited Create( CreateSuspended );
end;

destructor TLoadBlockThread.Free;
begin
end;

constructor TWorldBlock.Create;
begin
        MapListe := nil;
        StaticCount := 0;
        StaticListe := nil;
        ObjectListe := TObjectCache.Create;
        MultiCache := TList.Create;
        Visible := False;

        Counter.IncCount( Count_WorldBlock );
end;

destructor TWorldBlock.Free;
var     I : Integer;
begin
        Counter.DecCount( Count_WorldBlock );

        if MapListe <> nil then begin
                for I := 0 to 63 do begin
                        try
                                MapListe^[ I ].Free;
                        except
                                Log.Write( Format( 'Error freeing MapItem %d of Block %d', [I,BlockID] ) );
                        end;
                end;
                try
                        FreeMem( MapListe );
                except
                        Log.Write( Format( 'Error freeing MapList of Block %d', [BlockID] ) );
                end;
        end;

        if StaticCount > 0 then begin
                for I := 0 to StaticCount-1 do begin
                        try
                                StaticListe^[ I ].Free;
                        except
                                Log.Write( Format( 'Error freeing StaticItem %d of Block %d', [I,BlockID] ) );
                        end;
                end;
                try
                        FreeMem( StaticListe );
                except
                        Log.Write( Format( 'Error freeing StaticList of Block %d', [BlockID] ) );
                end;
        end;

        try
                MultiCache.Free;
        except
                Log.Write( Format( 'Error freeing MultiCache of Block %d', [BlockID] ) );
        end;

        try
                ObjectListe.Free;
        except
                Log.Write( Format( 'Error freeing ObjectCache of Block %d', [BlockID] ) );
        end;
end;

procedure TWorldBlock.ClearItems;
begin
        while ObjectListe.GetCount > 0 do
                ObjectListe.GetObjectByIndex( 0 ).Free;
end;

procedure TWorldBlock.ClearStatics;
var     I : Integer;
begin
        for I := 0 to StaticCount-1 do begin
                Palanthir.RemoveWorldItem( StaticListe^[ I ] );
                StaticListe^[ I ].Free;
        end;

        StaticCount := 0;
        FreeMem( StaticListe );
        StaticListe := nil;
end;

procedure TWorldBlock.ClearMap;
var     I : Integer;
begin
        if MapListe <> nil then begin
                for I := 0 to 63 do begin
                        Palanthir.RemoveWorldItem( MapListe^[ I ] );
                        MapListe^[ I ].Free;
                end;

                FreeMem( MapListe );
                MapListe := nil;
        end;
end;

procedure TWorldBlock.ClearMapStatics;
begin
        ClearMap;
        ClearStatics;
end;

function TWorldBlock.DistanceTo( Pos : TPos ) : Integer;
var     BlockPos : TPos;
begin
        BlockPos := TPos.Create( ( BlockID div GetMapHoehe )*8, ( BlockID mod GetMapHoehe )*8, 0 );

        Result := Round( BlockPos.DistanceTo( Pos ) );
        BlockPos.Free;
end;

procedure TWorldBlock.AddStatic( StaticItem : TStaticItem );
var     NewCount, I : Integer;
        NewStaticListe : PStaticListe;
begin
        NewCount := StaticCount+1;
        GetMem( NewStaticListe, NewCount*SizeOf( TStaticItem ) );

        for I := 0 to StaticCount-1 do begin
                NewStaticListe^[ I ] := StaticListe^[ I ];
        end;
        NewStaticListe[ NewCount-1 ] := StaticItem;

        if StaticListe <> nil then
                FreeMem( StaticListe );

        StaticListe := NewStaticListe;
        StaticCount := NewCount;
end;

procedure TWorldBlock.RemoveStatic( StaticItem : TStaticItem );
var     I, J : Integer;
        NewCount : Integer;
        NewStaticListe : PStaticListe;
begin
        NewCount := 0;
        for I := 0 to StaticCount-1 do begin
                if StaticListe^[ I ] = StaticItem then
                        Inc( NewCount );
        end;

        if NewCount > 0 then begin
                GetMem( NewStaticListe, NewCount*SizeOf( TStaticItem ) );
                J := 0;
                for I := 0 to StaticCount-1 do begin
                        if StaticListe^[ I ] <> StaticItem then begin
                                NewStaticListe^[ J ] := StaticListe^[ I ];
                                Inc( J );
                        end;
                end;

                FreeMem( StaticListe );
                StaticListe := NewStaticListe;
                StaticCount := NewCount;
        end
        else begin
                if StaticListe <> nil then
                        FreeMem( StaticListe );
                StaticListe := nil;
                StaticCount := 0;
        end;
end;

constructor TLoadBlock.Create;
begin
        MapListe := nil;
        StaticCount := 0;
        StaticListe := nil;
end;

destructor TLoadBlock.Free;
begin
end;

procedure TLoadBlock.FreeMemory;
var     I : Integer;
begin
        if MapListe <> nil then begin
                for I := 0 to 63 do
                        MapListe^[ I ].Free;
        end;
        FreeMem( MapListe );

        if StaticCount > 0 then begin
                for I := 0 to StaticCount-1 do begin
                        StaticListe^[ I ].Free;
                end;
                FreeMem( StaticListe );
        end;
end;

constructor TWorldCache.Create;
begin
        Count := 0;
        CriticalSection := TCriticalSection.Create;
        CriticalSection2 := TCriticalSection.Create;

        LoadBlockList := TList.Create;
        LoadedBlockList := TList.Create;

        LoadBlockThread := TLoadBlockThread.Create( True );
        LoadBlockThread.Init( Self );
        LoadBlockThread.Resume;
end;

destructor TWorldCache.Free;
var     I : Integer;
begin
        if Count > 0 then begin
                for I := 0 to Count-1 do begin
                        Cache^[ I ].Free;
                end;
                FreeMem( Cache );
        end;

        LoadBlockThread.Terminate;
        Sleep( 100 );
        LoadBlockThread.Free;        

        for I := 0 to LoadBlockList.Count-1 do begin
                TLoadBlock( LoadBlockList.Items[ I ] ).Free;
        end;
        for I := 0 to LoadedBlockList.Count-1 do begin
                TLoadBlock( LoadedBlockList.Items[ I ] ).Free;
        end;        

        LoadBlockList.Free;
        LoadedBlockList.Free;

        CriticalSection.Free;
        CriticalSection2.Free;
end;

function TWorldCache.GetMapHeight( X, Y : Word ) : ShortInt;
var     BlockID : LongWord;
        WorldBlock : TWorldBlock;
begin
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        WorldBlock := GetObject( BlockID );
        CriticalSection2.Leave;

        if WorldBlock.MapListe <> nil then
                Result := WorldBlock.MapListe^[ (Y mod 8)*8+(X mod 8) ].Pos.Z
        else if Palanthir.LoadMapStatics then begin
                CriticalSection2.Enter;
                Result := Palanthir.Data.GetMapHeight( X, Y );
                CriticalSection2.Leave;
        end else begin
                Result := 0;
        end;
end;

procedure TWorldCache.AddObject( Block : TWorldBlock );
var     Max, Min, Pos : Integer;
        NewCache : PWorldListe;
begin
        Min := 0;
        Max := Count;

        if Count <> 0 then begin
                while True do begin
                        Pos := ( Min + Max ) div 2;
                        if Cache^[ Pos ].BlockID > Block.BlockID then begin
                                if Max = Pos then
                                        break
                                else
                                        Max := Pos;
                        end
                        else if Cache^[ Pos ].BlockID < Block.BlockID then begin
                                if Min = Pos then
                                        break
                                else
                                        Min := Pos;
                        end
                        else if Cache^[ Pos ].BlockID = Block.BlockID then begin
                                break;
                        end;
                end;
        end
        else
                Max := 0;

        GetMem( NewCache, (Count+1)*sizeof( TWorldBlock ) );
        if Max > 0 then
                Move( Cache^[ 0 ], NewCache^[ 0 ], Max*sizeof( TWorldBlock ) );
        NewCache^[ Max ] := Block;
        if Count > Max then
                Move( Cache^[ Max ], NewCache^[ Max+1 ], (Count-Max)*sizeof( TWorldBlock ) );
        if Count > 0 then
                FreeMem( Cache );
        GetMem( Cache, (Count+1)*sizeof( TWorldBlock ) );
        Move( NewCache^[ 0 ], Cache^[ 0 ], (Count+1)*sizeof( TWorldBlock ) );
        FreeMem( NewCache );
        inc( Count );
end;

function TWorldCache.GetObject( BlockID : LongWord; Immediately : Boolean ) : TWorldBlock;
var     Max, Min, Pos : Integer;
        Block : TLoadBlock;
begin
        Min := 0;
        Max := Count;

        if Count <> 0 then begin
                while True do begin
                        Pos := ( Min + Max ) div 2;
                        if Cache^[ Pos ].BlockID > BlockID then begin
                                if Max = Pos then
                                        break
                                else
                                        Max := Pos;
                        end
                        else if Cache^[ Pos ].BlockID < BlockID then begin
                                if Min = Pos then
                                        break
                                else
                                        Min := Pos;
                        end
                        else if Cache^[ Pos ].BlockID = BlockID then begin
                                Result := Cache^[ Pos ];
                                exit;
                        end;
                end;
        end;


        Result := TWorldBlock.Create;
        Result.BlockID := BlockID;
        AddObject( Result );

        if Palanthir.LoadMapStatics then begin
                Block := TLoadBlock.Create;
                Block.BlockID := BlockID;

                if Immediately then begin
                        Palanthir.Data.LoadMapBlock( Block );
                        Result.MapListe := Block.MapListe;
                        Result.StaticCount := Block.StaticCount;
                        Result.StaticListe := Block.StaticListe;
                        Block.Free;
                end else begin
                        CriticalSection.Enter;
                        LoadBlockList.Add( Block );
                        CriticalSection.Leave;
                end;
        end;
end;

function TWorldCache.GetObjectWithoutLoad( BlockID : LongWord ) : TWorldBlock;
var     Max, Min, Pos : Integer;
begin
        Min := 0;
        Max := Count;

        if Count <> 0 then begin
                while True do begin
                        Pos := ( Min + Max ) div 2;
                        if Cache^[ Pos ].BlockID > BlockID then begin
                                if Max = Pos then
                                        break
                                else
                                        Max := Pos;
                        end
                        else if Cache^[ Pos ].BlockID < BlockID then begin
                                if Min = Pos then
                                        break
                                else
                                        Min := Pos;
                        end
                        else if Cache^[ Pos ].BlockID = BlockID then begin
                                Result := Cache^[ Pos ];
                                exit;
                        end;
                end;
        end;

        Result := nil;
end;

procedure TWorldCache.DeleteObject( BlockID : LongWord );
var     Max, Min, Pos : Integer;
        NewCache : PWorldListe;
        Block : TWorldBlock;
begin
        Min := 0;
        Max := Count;

        if Count <> 0 then begin
                while True do begin
                        Pos := ( Min + Max ) div 2;
                        if Cache^[ Pos ].BlockID > BlockID then begin
                                if Max = Pos then
                                        break
                                else
                                        Max := Pos;
                        end
                        else if Cache^[ Pos ].BlockID < BlockID then begin
                                if Min = Pos then
                                        break
                                else
                                        Min := Pos;
                        end
                        else if Cache^[ Pos ].BlockID = BlockID then begin
                                Block := Cache^[ Pos ];
                                Block.ClearItems;
                                if Count > 1 then begin
                                        GetMem( NewCache, (Count-1)*sizeof( TWorldBlock ) );
                                        if Pos > 0 then
                                                Move( Cache^[ 0 ], NewCache^[ 0 ], Pos*sizeof( TWorldBlock ) );
                                        if ( Count > Pos+1 ) then
                                                Move( Cache^[ Pos+1 ], NewCache^[ Pos ], (Count-Pos-1)*sizeof( TWorldBlock ) );
                                        FreeMem( Cache );
                                        GetMem( Cache, (Count-1)*sizeof( TWorldBlock ) );
                                        Move( NewCache^[ 0 ], Cache^[ 0 ], (Count-1)*sizeof( TWorldBlock ) );
                                        FreeMem( NewCache );
                                        Dec( Count );
                                end
                                else if Count > 0 then begin
                                        FreeMem( Cache );
                                        Count := 0;
                                end;
                                Block.Free;
                                exit;
                        end;
                end;
        end;
end;

procedure TWorldCache.DeleteObject( Block : TWorldBlock );
begin
        DeleteObject( Block.BlockID );
end;

procedure TWorldCache.ShowBlock( BlockID : LongWord );
var     Block : TWorldBlock;
        I : Integer;
begin
        CriticalSection2.Enter;
        Block := GetObject( BlockID );
        CriticalSection2.Leave;

        Block.Visible := True;

        if Block.MapListe <> nil then begin
                for I := 0 to 63 do
                        Palanthir.AddWorldItem( Block.MapListe^[ I ] );
        end;

        if Block.StaticCount > 0 then
                for I := 0 to Block.StaticCount-1 do
                        Palanthir.AddWorldItem( Block.StaticListe^[ I ] );

        if Block.ObjectListe.GetCount > 0 then
                for I := 0 to Block.ObjectListe.GetCount-1 do
                        Palanthir.AddWorldItem( Block.ObjectListe.GetObjectByIndex( I ) );

        for I := 0 to Block.MultiCache.Count-1 do
                Palanthir.AddWorldItem( Block.MultiCache.Items[ I ] );
end;

procedure TWorldCache.HideBlock( BlockID : LongWord );
var     Block : TWorldBlock;
begin
        CriticalSection2.Enter;
        Block := GetObjectWithoutLoad( BlockID );
        CriticalSection2.Leave;
        
        if ( Block <> nil ) and Block.Visible then begin
                Block.Visible := False;
                Palanthir.RemoveWorldBlock( BlockID );
        end;
end;

function TWorldCache.BlockIsVisible( BlockID : LongWord ) : Boolean;
var     Block : TWorldBlock;
begin
        CriticalSection2.Enter;
        Block := GetObjectWithoutLoad( BlockID );
        CriticalSection2.Leave;
        
        if ( Block <> nil ) and Block.Visible then begin
                Result := True;
        end else begin
                Result := False;
        end;
end;

procedure TWorldCache.SetAllInvis;
var     I : Integer;
        Block : TWorldBlock;
begin
        if Count > 0 then
                for I := 0 to Count-1 do begin
                        Block := Cache^[ I ];
                        Block.Visible := False;
                end;
end;

function TWorldCache.GetStaticObjects( X, Y : Word ) : TList;
var     I : Integer;
        Block : TWorldBlock;
        BlockID : Integer;
        Static : TStaticItem;
begin
        Result := TList.Create;
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        Block := GetObject( BlockID, True );
        CriticalSection2.Leave;
        
        for I := 0 to Block.StaticCount-1 do begin
                Static := Block.StaticListe^[ I ];
                if (Static.Pos.X = X) and (Static.Pos.Y = Y) then
                        Result.Add( Static );
        end;
end;

function TWorldCache.GetItems( X, Y : Word ) : TList;
var     BlockID : Integer;
        I : Integer;
        Block : TWorldBlock;
        UObject : TUObject;
begin
        Result := TList.Create;
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        Block := GetObject( BlockID, True );
        CriticalSection2.Leave;
        
        for I := 0 to Block.ObjectListe.GetCount-1 do begin
                UObject := Block.ObjectListe.GetObjectByIndex( I );
                if UObject.IsItem and (UObject.Pos.X = X) and (UObject.Pos.Y = Y) then
                        Result.Add( UObject );
        end;
end;

function TWorldCache.GetChars( X, Y : Word ) : TList;
var     BlockID : Integer;
        I : Integer;
        Block : TWorldBlock;
        UObject : TUObject;
begin
        Result := TList.Create;
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        Block := GetObject( BlockID, True );
        CriticalSection2.Leave;
        
        for I := 0 to Block.ObjectListe.GetCount-1 do begin
                UObject := Block.ObjectListe.GetObjectByIndex( I );
                if UObject.IsChar and (UObject.Pos.X = X) and (UObject.Pos.Y = Y) then
                        Result.Add( UObject );
        end;
end;

function TWorldCache.GetMultiItems( X, Y : Word ) : TList;
var     BlockID : Integer;
        I : Integer;
        Block : TWorldBlock;
        MultiItem : TMultiItem;
begin
        Result := TList.Create;
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        Block := GetObject( BlockID, True );
        CriticalSection2.Leave;
        
        for I := 0 to Block.MultiCache.Count-1 do begin
                MultiItem := TMultiItem( Block.MultiCache.Items[ I ] );
                if (MultiItem.Pos.X = X) and (MultiItem.Pos.Y = Y) then
                        Result.Add( MultiItem );
        end;
end;

function TWorldCache.GetMapItem( X, Y : Word ) : TMapItem;
var     BlockID : Integer;
        Block : TWorldBlock;
begin
        BlockID := ( X div 8 ) * GetMapHoehe + ( Y div 8 );

        CriticalSection2.Enter;
        Block := GetObject( BlockID, True );
        CriticalSection2.Leave;
        
        if Block.MapListe <> nil then
                Result := Block.MapListe^[ (Y mod 8)*8+(X mod 8) ]
        else
                Result := nil;
end;

procedure TWorldCache.CheckObjectDistances( BlockID : LongWord; Pos : TPos );
var     I : Integer;
        Block : TWorldBlock;
        UObject : TUObject;
        Liste : TList;
begin
        CriticalSection2.Enter;
        Block := GetObjectWithoutLoad( BlockID );
        CriticalSection2.Leave;

        if Block = nil then
                Exit;

        Liste := TList.Create;
        for I := 0 to Block.ObjectListe.GetCount-1 do
                Liste.Add( Block.ObjectListe.GetObjectByIndex( I ) );

        for I := 0 to Liste.Count-1 do begin
                UObject := TUObject( Liste.Items[ I ] );
                if not UObject.Pos.IsInRechteckRange( Pos, Palanthir.Player.VisRange ) then
                        UObject.Free;
        end;
        Liste.Free;
end;

procedure TWorldCache.GarbageCollection;
var     I : Integer;
        Block : TWorldBlock;
begin
        if Count > 0 then begin
                for I := Count-1 downto 0 do begin
                        Block := Cache^[ I ];
                        if (Block.ObjectListe.GetCount = 0) and (not Block.Visible) and (Block.MultiCache.Count = 0) and (Block.DistanceTo( Palanthir.Player.Pos ) > 24) then
                                DeleteObject( Block );
                end;
        end;
end;

function TWorldCache.GetCount : Integer;
begin
        Result := Count;
end;

procedure TWorldCache.Reload;
var     I : Integer;
        Block : TWorldBlock;
        LBlock : TLoadBlock;
begin
        if Count > 0 then
                for I := 0 to Count-1 do begin
                        Block := Cache^[ I ];

                        Block.ClearMapStatics;

                        if Palanthir.LoadMapStatics then begin
                                LBlock := TLoadBlock.Create;
                                LBlock.BlockID := Block.BlockID;
                                CriticalSection.Enter;
                                LoadBlockList.Add( LBlock );
                                CriticalSection.Leave;
                        end;
                end;
end;

procedure TWorldCache.Poll;
var     Count : LongWord;
        Block : TLoadBlock;
        WorldBlock : TWorldBlock;
begin
        CriticalSection.Enter;
        Count := LoadedBlockList.Count;
        CriticalSection.Leave;

        while Count > 0 do begin
                CriticalSection.Enter;
                Block := TLoadBlock( LoadedBlockList.Items[0] );
                LoadedBlockList.Delete( 0 );
                CriticalSection.Leave;

                WorldBlock := GetObjectWithoutLoad( Block.BlockID );
                if (WorldBlock <> nil) and Palanthir.LoadMapStatics then begin
                        WorldBlock.ClearMapStatics;
                        WorldBlock.MapListe := Block.MapListe;
                        WorldBlock.StaticCount := Block.StaticCount;
                        WorldBlock.StaticListe := Block.StaticListe;

                        if WorldBlock.Visible then
                                ShowBlock( WorldBlock.BlockID );

                        Block.Free;
                end
                else begin
                        Block.FreeMemory;
                        Block.Free;
                end;

                CriticalSection.Enter;
                Count := LoadedBlockList.Count;
                CriticalSection.Leave;
        end;
end;

procedure TWorldCache.ClearMapStatics;
var     I : Integer;
        Block : TWorldBlock;
begin
        if Count > 0 then
                for I := 0 to Count-1 do begin
                        Block := Cache^[ I ];
                        Block.ClearMapStatics;
                end;
end;

end.
