unit uPyConfig;

interface

uses    uConfig, Python, uPyUtilities, SysUtils;

{
        \object config
        \description Represents a configfile object.
}

type    TpyConfig = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                Config : TConfig;
        end;

        PpyConfig = ^TpyConfig;

function pyConfig_Str( Self : PPyObject ) : PPyObject; cdecl;
procedure pyConfig_Dealloc( Self : PPyObject ); cdecl;
function pyConfig_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyConfig_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyConfig_compare( Config1, Config2 : PPyObject ) : Integer; cdecl;
function pyConfig_getConfig( Self : PPyObject ) : TConfig; cdecl;
function pyConfig_check( Self : PPyObject ) : Boolean;
function pyGetConfig( Config : TConfig ) : PPyObject;

function pyConfig_getString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyConfig_getInteger( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyConfig_getBool( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

function pyConfig_setString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyConfig_setInteger( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyConfig_setBool( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;


const pyConfigType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyConfig';
		tp_basicsize : sizeof( pyConfigType );
                tp_itemsize : 0;

		tp_dealloc : pyConfig_Dealloc;
		tp_print : nil;
		tp_getattr : pyConfig_getAttr;
		tp_setattr : pyConfig_setAttr;
		tp_compare : pyConfig_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyConfig_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyConfigMethods : Array[0..6] of PyMethodDef =
                ( ( ml_name : 'getstring'; ml_meth : pyConfig_getString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getinteger'; ml_meth : pyConfig_getInteger; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getbool'; ml_meth : pyConfig_getBool; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setstring'; ml_meth : pyConfig_setString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setinteger'; ml_meth : pyConfig_setInteger; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setbool'; ml_meth : pyConfig_setBool; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

function pyConfig_Str( Self : PPyObject ) : PPyObject;
begin
        Result := PyString_FromString( 'Config' );
end;

procedure pyConfig_Dealloc( Self : PPyObject );
begin
        PyDealloc( Self );
end;

function pyConfig_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     Config : TConfig;
        TestName : String;
begin
        if not pyConfig_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        TestName := LowerCase( Name );

        Result := Py_FindMethod( @pyConfigMethods, Self, PChar( TestName ) );
end;

function pyConfig_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     Config : TConfig;
        TestName : String;
begin
        if not pyConfig_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := 1;
                Exit;
        end;

        TestName := LowerCase( Name );
        Result := 1;
end;

function pyConfig_compare( Config1, Config2 : PPyObject ) : Integer;
var     Config_1, Config_2 : TConfig;
begin
        if (not pyConfig_check( Config1 )) or (not pyConfig_check( Config2 )) then begin
                Result := -1;
                Exit;
        end;

        Config_1 := pyConfig_getConfig( Config1 );
        Config_2 := pyConfig_getConfig( Config2 );

        if Config_1 = Config_2 then begin
                Result := 0;
        end
        else begin
                Result := -1;
        end;
end;

function pyConfig_getConfig( Self : PPyObject ) : TConfig;
begin
        if not pyConfig_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        Result := PpyConfig( Self ).Config;
end;

function pyConfig_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyConfigType;
end;

function pyGetConfig( Config : TConfig ) : PPyObject;
var     PyConfig : PpyConfig;
begin
        PyConfig := PpyConfig( PyObject_new( @pyConfigType ) );
        PyConfig.Config := Config;
        Result := PPyObject( PyConfig );
end;

{
        \method config.getstring
        \param name
        \description Returns the string type entry of the configfile.
}
function pyConfig_getString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Config : TConfig;
        Name : PChar;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 's:config.getstring( name )', [@Name] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Result := PyString_FromString( PChar( Config.GetString( Name ) ) );
end;

{
        \method config.getinteger
        \param name
        \description Returns the integer type entry of the configfile.
} 
function pyConfig_getInteger( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Config : TConfig;
        Name : PChar;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 's:config.getinteger( name )', [@Name] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Result := PyInt_FromLong( Config.GetInteger( Name ) );
end;

{
        \method config.getbool
        \param name
        \description Returns the boolean type entry of the configfile.
}
function pyConfig_getBool( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Config : TConfig;
        Name : PChar;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 's:config.getbool( name )', [@Name] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if Config.GetBool( Name ) then begin
                Result := PyInt_FromLong( 1 );
        end
        else begin
                Result := PyInt_FromLong( 0 );
        end;
end;

{
        \method config.setstring
        \param name
        \param value
        \description Sets the string type entry of the configfile to the given value.
}
function pyConfig_setString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Config : TConfig;
        Name, Value : PChar;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'ss:config.setstring( name, value )', [@Name,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Config.SetString( Name, Value );
        Result := PyReturnTrue;
end;

{
        \method config.setinteger
        \param name
        \param value
        \description Sets the integer type entry of the configfile to the given value.
}
function pyConfig_setInteger( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Config : TConfig;
        Name : PChar;
        Value : Integer;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'si:config.setinteger( name, value )', [@Name,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Config.SetInteger( Name, Value );
        Result := PyReturnTrue;
end;

{
        \method config.setbool
        \param name
        \param value
        \description Sets the bool type entry of the configfile to the given value.
}
function pyConfig_setBool( Self : PPyObject; Args : PPyObject ) : PPyObject; 
var     Config : TConfig;
        Name : PChar;
        Value : Integer;
begin
        Config := pyConfig_getConfig( Self );

        if Config = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'si:config.setbool( name, value )', [@Name,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Config.SetBool( Name, Value = 1 );
        Result := PyReturnTrue;
end;

end.
