unit uPyGump;

interface

uses    Python, uPythonGump, SysUtils, uPyUtilities, uPyGumpObject, uGumps,
        Dialogs;

{
        \object gump
        \description Represents a gump object.
        To create a gump use the newgump function in the <module id="palanthir">palanthir</module> module.
}

type    TpyGump = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                GumpSerial : LongWord
        end;

        PpyGump = ^TpyGump;

function pyGump_Str( Self : PPyObject ) : PPyObject; cdecl;
procedure pyGump_Dealloc( Self : PPyObject ); cdecl;
function pyGump_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyGump_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyGump_compare( Gump1, Gump2 : PPyObject ) : Integer; cdecl;
function pyGump_getGump( Self : PPyObject ) : TPythonGump;
function pyGump_check( Self : PPyObject ) : Boolean;
function pyGetGump( Gump : TPythonGump ) : PPyObject; cdecl;
function pyConvertGump( PyObject : PPyObject; Gump : P_PythonGump ) : Integer; cdecl;

function pyGump_AddButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddTextButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddPageButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddImage( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddTiledImage( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddTilePic( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddLandTilePic( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddText( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddResizeGump( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddCroppedText( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddInputField( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddCheckBox( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddRadioButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddHtmlGump( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddXmfHtmlGump( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddCheckerTrans( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddGumpItem( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddTransparendGumpItem( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddStaticItem( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddScrollBar( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddHueButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddProgressBar( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddExtImage( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_AddTooltip( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

function pyGump_Show( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_Delete( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_getObjectCount( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_getObject( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_deleteObject( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_getButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_getInputFieldText( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_getRadioButton( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyGump_setBoundingBox( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

const pyGumpType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyGump';
		tp_basicsize : sizeof( pyGumpType );
                tp_itemsize : 0;

		tp_dealloc : pyGump_Dealloc;
		tp_print : nil;
		tp_getattr : pyGump_getAttr;
		tp_setattr : pyGump_setAttr;
		tp_compare : pyGump_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyGump_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyGumpMethods : Array[0..33] of PyMethodDef =
                ( ( ml_name : 'addbutton'; ml_meth : pyGump_AddButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtextbutton'; ml_meth : pyGump_AddTextButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addpagebutton'; ml_meth : pyGump_AddPageButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addimage'; ml_meth : pyGump_AddImage; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtiledimage'; ml_meth : pyGump_AddTiledImage; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtilepic'; ml_meth : pyGump_AddTilePic; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addlandtilepic'; ml_meth : pyGump_AddLandTilePic; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtext'; ml_meth : pyGump_AddText; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addresizegump'; ml_meth : pyGump_AddResizeGump; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addcroppedtext'; ml_meth : pyGump_AddCroppedText; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addinputfield'; ml_meth : pyGump_AddInputField; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addcheckbox'; ml_meth : pyGump_AddCheckBox; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addradiobutton'; ml_meth : pyGump_AddRadioButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addhtmlgump'; ml_meth : pyGump_AddHtmlGump; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addxmfhtmlgump'; ml_meth : pyGump_AddXmfHtmlGump; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addcheckertrans'; ml_meth : pyGump_AddCheckerTrans; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addgumpitem'; ml_meth : pyGump_AddGumpItem; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtransparendgumpitem'; ml_meth : pyGump_AddTransparendGumpItem; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addstaticitem'; ml_meth : pyGump_AddStaticItem; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addscrollbar'; ml_meth : pyGump_AddScrollBar; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addhuebutton'; ml_meth : pyGump_AddHueButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addprogressbar'; ml_meth : pyGump_AddProgressBar; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addextimage'; ml_meth : pyGump_AddExtImage; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'addtooltip'; ml_meth : pyGump_AddTooltip; ml_flags : METH_VARARGS; ml_doc : nil ),

                  ( ml_name : 'show'; ml_meth : pyGump_Show; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'delete'; ml_meth : pyGump_Delete; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getobjectcount'; ml_meth : pyGump_getObjectCount; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getobject'; ml_meth : pyGump_getObject; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'deleteobject'; ml_meth : pyGump_deleteObject; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getbutton'; ml_meth : pyGump_getButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getinputfieldtext'; ml_meth : pyGump_getInputFieldText; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getradiobutton'; ml_meth : pyGump_getRadioButton; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setboundingbox'; ml_meth : pyGump_setBoundingBox; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

uses    uPalanthir;

function pyGump_Str( Self : PPyObject ) : PPyObject;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Result := PyString_FromString( PChar( 'pythongump' ) );
end;

procedure pyGump_Dealloc( Self : PPyObject );
begin
        PyDealloc( Self );
end;

function pyGump_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     Gump : TPythonGump;
        TestName : String;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;
        TestName := LowerCase( Name );

        {
                \property gump.internalid Internal id of the gump. (readonly)
        }
        if TestName = 'internalid' then begin
                Result := PyInt_FromLong( Gump.InternalID );
        end
        {
                \property gump.x X position of the gump.
        }
        else if TestName = 'x' then begin
                Result := PyInt_FromLong( Gump.X );
        end
        {
                \property gump.y Y position of the gump.
        }
        else if TestName = 'y' then begin
                Result := PyInt_FromLong( Gump.Y );
        end
        {
                \property gump.activepage The page of the gump that is drawn at the moment.
        }
        else if TestName = 'activepage' then begin
                Result := PyInt_FromLong( Gump.ActivePage );
        end
        {
                \property gump.nomove Indicates whether the gump is movable or not.
        }
        else if TestName = 'nomove' then begin
                if Gump.noMove then
                        Result := PyInt_FromLong( 1 )
                else
                        Result := PyInt_FromLong( 0 );
        end
        {
                \property gump.noclose Indicates whether the client is allowed to close the gump or not.
        }
        else if TestName = 'noclose' then begin
                if Gump.noClose then
                        Result := PyInt_FromLong( 1 )
                else
                        Result := PyInt_FromLong( 0 );
        end
        {
                \property gump.nodispose Indicates whether the gump is disposable or not.
        }
        else if TestName = 'nodispose' then begin
                if Gump.noDispose then
                        Result := PyBool_FromLong( 1 )
                else
                        Result := PyBool_FromLong( 0 );
        end
        {
                \property gump.noautoclose Indicates whether the gump will be closed after a click on any button.
        }
        else if TestName = 'noautoclose' then begin
                if Gump.noAutoClose then
                        Result := PyBool_FromLong( 1 )
                else
                        Result := PyBool_FromLong( 0 );
        end
        {
                \property gump.customvars A dictionary to save variables on the gump. (see cv)
        }
        {
                \property gump.cv A dictionary to save variables on the gump. (see customvars)
        }
        else if (TestName = 'customvars') or (TestName = 'cv') then begin
                if Gump.CustomVars = nil then begin
                        Gump.CustomVars := PyDict_New;
                end;

                Py_INCREF( Gump.CustomVars );
                Result := Gump.CustomVars;
        end
        {
                \property gump.enableboundingbox Indicates if the boundingbox is used.
        }
        else if TestName = 'enableboundingbox' then begin
                if Gump.BBEnabled then
                        Result := PyBool_FromLong( 1 )
                else
                        Result := PyBool_FromLong( 0 );
        end
        {
                \property gump.globalalpha Alpha value that is used for the whole gump.
        }
        else if TestName = 'globalalpha' then begin
                Result := PyBool_FromLong( Gump.GlobalAlpha );
        end
        else begin
                Result := Py_FindMethod( @pyGumpMethods, Self, PChar( TestName ) );
        end;
end;

function pyGump_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     Gump : TPythonGump;
        TestName : String;
begin
        if not pyGump_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := 1;
                Exit;
        end;
        TestName := LowerCase( Name );

        if TestName = 'x' then begin
                if PyInt_Check( Value ) then begin
                        Gump.X := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'y' then begin
                if PyInt_Check( Value ) then begin
                        Gump.Y := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'activepage' then begin
                if PyInt_Check( Value ) then begin
                        Gump.ActivePage := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'nomove' then begin
                if PyInt_Check( Value ) then begin
                        Gump.noMove := PyInt_AsLong( Value ) = 1;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'noclose' then begin
                if PyInt_Check( Value ) then begin
                        Gump.noClose := PyInt_AsLong( Value ) = 1;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'nodispose' then begin
                if PyInt_Check( Value ) then begin
                        Gump.noDispose := PyInt_AsLong( Value ) = 1;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'noautoclose' then begin
                if PyInt_Check( Value ) then begin
                        Gump.noAutoClose := PyInt_AsLong( Value ) = 1;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'enableboundingbox' then begin
                if PyInt_Check( Value ) then begin
                        Gump.BBEnabled := PyInt_AsLong( Value ) = 1;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'globalalpha' then begin
                if PyInt_Check( Value ) then begin
                        Gump.GlobalAlpha := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onbuttonclick The function that is triggered when a button is clicked.
                
                 The function should have the following prototype:
                 <code>def onbuttonclick( gump, returnnumber ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is the <object id="gump">gump</object> object you clicked on and returnnumber the returnnumber of the button used.

                 Return 1 to stop the core to handle this click.
        }
        else if (TestName = 'onbuttonclick') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnButtonClickEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onclickxy The function that is triggered when a button is clicked.

                 The function should have the following prototype:
                 <code>def onclickxy( gump, x, y ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 X, Y are the coordinates of the clickpoint relativ to the origin of the gump.

                 Return 1 to stop the core to handle this click.
        }
        else if (TestName = 'onclickxy') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnClickEventXY( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.ondblclick The function that is triggered when a gump is dblclicked.

                 The function should have the following prototype:
                 <code>def ondblclick( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the <object id="gumpobject">gumpobject</object> object you dblclicked on.

                 Return 1 to stop the core to handle this dblclick.
        }
        else if (TestName = 'ondblclick') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnDblClickEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.ondblclickxy The function that is triggered when a gump is dblclicked.

                 The function should have the following prototype:
                 <code>def onbuttonclickxy( gump, x, y ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 X, Y are the coordinates of the dblclickpoint relativ to the origin of the gump.

                 Return 1 to stop the core to handle this dblclick.
        }
        else if (TestName = 'ondblclickxy') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnDblClickEventXY( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.oncheckboxchange The function that is triggered when a checkbox is changed.

                 The function should have the following prototype:
                 <code>def oncheckboxchange( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the <object id="gumpobject">checkbox</object> object that changed.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'oncheckboxchange') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnCheckBoxChangeEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onscrollbarchange The function that is triggered when a scrollbar is changed.

                 The function should have the following prototype:
                 <code>def onscrollbarchange( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the <object id="gumpobject">scrollbar</object> object that changed.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onscrollbarchange') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnScrollBarChangeEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onmoveoverbutton The function that is triggered when the cursor moves over a button.

                 The function should have the following prototype:
                 <code>def onmoveoverbutton( gump, returnnumber ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Returnnumber is the returnnumber of the button. 

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onmoveoverbutton') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnMoveOverButtonEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onmovegump The function that is triggered when the gump is moved around.

                 The function should have the following prototype:
                 <code>def onmovegump( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onmovegump') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnMoveEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.ondrawgump The function that is triggered each time directly before the gump is drawn.

                 The function should have the following prototype:
                 <code>def ondrawgump( gump ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.

                 Return 1 if you dont want the gump to be drawn.
        }
        else if (TestName = 'ondrawgump') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnDrawEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onkeyreturn The function that is triggered when return is pressed and a gump is selected.

                 The function should have the following prototype:
                 <code>def onkeyreturn( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onkeyreturn') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnKeyReturnEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onkeytab The function that is triggered when tab is pressed and a gump is selected.

                 The function should have the following prototype:
                 <code>def onkeytab( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onkeytab') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnKeyTabEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.oninputfieldchange The function that is triggered when an inputfield changes.

                 The function should have the following prototype:
                 <code>def oninputfieldchange( gump, inputfield ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Inputfield is the <object id="gumpobject">inputfield</object> object that changes.
        }
        else if (TestName = 'oninputfieldchange') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnInputFieldChangeEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onscrollbarrelease The function that is triggered when a scrollbar is released.

                 The function should have the following prototype:
                 <code>def onscrollbarrelease( gump, inputfield ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Scrollbar is the <object id="gumpobject">scrollbar</object> object that is released.
        }
        else if (TestName = 'onscrollbarrelease') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnScrollBarReleaseEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onbeforedrawgump The function that is triggered for all gumps each time before all gumps are drawn.

                 The function should have the following prototype:
                 <code>def onbeforedrawgump( gump ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 if you dont want the gump to be drawn.
        }
        else if (TestName = 'onbeforedrawgump') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnBeforeDrawEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onafterdrawgump The function that is triggered for all gumps each time after all gumps are drawn.

                 The function should have the following prototype:
                 <code>def onafterdrawgump( gump ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onafterdrawgump') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnAfterDrawEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.ondelete The function that is triggered before the gump is deleted.

                 The function should have the following prototype:
                 <code>def ondelete( gump ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'ondelete') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnDeleteEvent( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onmousedown The function that is triggered when the left mousebutton goes down.

                 The function should have the following prototype:
                 <code>def onmousedown( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onmousedown') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnMouseDown( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        {
                \property gump.onmouseup The function that is triggered when the left mousebutton goes up.

                 The function should have the following prototype:
                 <code>def onmouseup( gump, gumpobject ):
                 &nbsp;&nbsp;pass</code>

                 Although you can choose any name you want.
                 Gump is this <object id="gump">gump</object> object.
                 Gumpobject is the selected <object id="gumpobject">gumpobject</object> object.

                 Return 1 to stop the core to handle this event.
        }
        else if (TestName = 'onmouseup') then begin
                if PyCallable_Check( Value ) > 0 then begin
                        Gump.SetOnMouseUp( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else begin
                Result := 1;
        end;
end;

function pyGump_compare( Gump1, Gump2 : PPyObject ) : Integer;
var     P1, P2 : TPythonGump;
begin
        if (not pyGump_check( Gump1 )) and (not pyGump_check( Gump2 )) then begin
                Result := -1;
                Exit;
        end;

        P1 := pyGump_getGump( Gump1 );
        P2 := pyGump_getGump( Gump2 );
        
        if P1 = P2 then begin
                Result := 0;
        end
        else begin
                Result := -1;
        end;
end;

function pyGump_getGump( Self : PPyObject ) : TPythonGump;
var     I : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        Result := nil;
        for I := 0 to Palanthir.GlobalGumpList.Count-1 do begin
                if TGump( Palanthir.GlobalGumpList.Items[ I ] ).InternalID = PpyGump( Self ).GumpSerial then begin
                        if TGump( Palanthir.GlobalGumpList.Items[ I ] ).Typ = GumpTyp_PythonGump then begin
                                Result := TPythonGump( Palanthir.GlobalGumpList.Items[ I ] );
                                Break;
                        end;
                end;
        end;
end;

function pyGump_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyGumpType;
end;

function pyGetGump( Gump : TPythonGump ) : PPyObject;
var     PyGump : PpyGump;
begin
        PyGump := PpyGump( PyObject_new( @pyGumpType ) );
        PyGump.GumpSerial := Gump.InternalID;
        Result := PPyObject( PyGump );
end;

function pyConvertGump( PyObject : PPyObject; Gump : P_PythonGump ) : Integer;
begin
        if not pyGump_check( PyObject ) then begin
                PyErr_BadArgument;
                Result := 0;
                Exit;
        end;

        Gump^ := pyGump_getGump( pyObject );
        Result := 1;
end;

{
        \method gump.addbutton
        \param x X position of the button.
        \param y Y position of the button.
        \param upid
        \param downid
        \param returnnumber
        \param page
        \param moveoverid (optional)
        \param checkcompleterectangle (optional) Set to 1 to get a mouseclick everywhere in the rectangle of the texture of the button.
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a button, adds it to the gump and returns it.
}

function pyGump_AddButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        UpId, DownId, MoveOverId : Word;
        ReturnNumber : Integer;
        Page : Word;
        CheckCompleteRectangle : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        MoveOverId := 0;
        CheckCompleteRectangle := 0;
        if PyArg_ParseTuple( Args, 'iihhih|hb:gump.addbutton( x, y, upid, downid, returnnumber, page, [moveoverid], [checkcompleterectangle] )', [@X,@Y,@UpId,@DownId,@ReturnNumber,@Page,@MoveOverId,@CheckCompleteRectangle] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        if MoveOverId = 0 then begin
                MoveOverId := UpId;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;
        Result := pyGetGumpObject( Gump.AddButton( X, Y, UpId, DownId, MoveOverId, ReturnNumber, Page, CheckCompleteRectangle = 1 ) );
end;

{
        \method gump.addtextbutton
        \param x X position of the button.
        \param y Y position of the button.
        \param returnnumber
        \param text
        \param font Font id of the text.
        \param page
        \param width
        \param uphue (optional)
        \param downhue (optional)
        \param moveoverhue (optional)
        \param backgroundupid (optional)
        \param backgrounddownid (optional)
        \param backgroundmoveoverid (optional)
        \param backgrounduphue (optional)
        \param backgrounddownhue (optional)
        \param backgroundmoveoverhue (optional)
        \param height (optional)
        \param ispagebutton Set to 1 to use this textbutton as pagebutton. (optional)
        \param pageid If this button is used as pagebutton, then its the target page. Otherwise its useless. (optional)
        \param border Determines whether the characters have a black one-pixel border or not. (optional)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a textbutton, adds it to the gump and returns it.
}

function pyGump_AddTextButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        ReturnNumber : Integer;
        Text : PChar;
        Font : Byte;
        Page, Breite, Hoehe : Word;
        UpHue, DownHue, MoveOverHue, BackgroundUpId, BackgroundDownId, BackgroundMoveOverId : Word;
        BackgroundUpHue, BackgroundDownHue, BackgroundMoveOverHue : Word;
        IsPageButton : Byte;
        PageId : Word;
        Border : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Hoehe := 0;
        UpHue := 0;
        DownHue := 0;
        MoveOverHue := 0;
        BackGroundUpId := 0;
        BackGroundDownId := 0;
        BackGroundMoveOverId := 0;
        BackGroundUpHue := 0;
        BackGroundDownHue := 0;
        BackGroundMoveOverHue := 0;
        IsPageButton := 0;
        PageId := 0;
        Border := 1;

        if PyArg_ParseTuple( Args, 'iiisbhh|hhhhhhhhhhbhb:gump.addtextbutton( x, y, returnnumber, text, font, page, width, [uhue], [dhue], [mohue], [bupid], [bdid], [bmoid], [buhue], [bdhue], [bmohue], [height], [ispagebutton], [pageid] )', [@X, @Y, @ReturnNumber, @Text, @Font, @Page, @Breite, @UpHue, @DownHue, @MoveOverHue, @BackgroundUpId, @BackgroundDownId, @BackgroundMoveOverId, @BackgroundUpHue, @BackgroundDownHue, @BackgroundMoveOverHue, @Hoehe, @IsPageButton, @PageId, @Border] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddTextButton( X, Y, ReturnNumber, Text, Font, Page, Breite, UpHue, DownHue, MoveOverHue, BackGroundUpId, BackGroundDownId, BackGroundMoveOverId, BackGroundUpHue, BackGroundDownHue, BackGroundMoveOverHue, Hoehe, IsPageButton = 1, PageId, Border = 1 ) );
end;

{
        \method gump.addpagebutton
        \param x X position of the button.
        \param y Y position of the button.
        \param upid
        \param downid
        \param pageid Id of the target page.
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a pagebutton, adds it to the gump and returns it.
}

function pyGump_AddPageButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        UpId, DownId : Word;
        Page, PageId : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhhh:gump.addpagebutton( x, y, upid, downid, pageid, page )', [@X,@Y,@UpId,@DownId,@PageId,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;


        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddPageButton( X, Y, UpId, DownId, PageId, Page ) );
end;

{
        \method gump.addimage
        \param x X position of the button.
        \param y Y position of the button.
        \param id
        \param hue
        \param page
        \param width (optional)
        \param height (optional)
        \param keepproportions (optional)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an image, adds it to the gump and returns it.
}

function pyGump_AddImage( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Id, Hue : Integer;
        Page : Word;
        Width, Height : Word;
        Image : TGImage;
        KeepProportions : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Width := $FFFF;
        Height := $FFFF;
        KeepProportions := 0;
        if PyArg_ParseTuple( Args, 'iiiih|hhb:gump.addimage( x, y, id, hue, page, [width], [height], [keepproportions] )', [@X,@Y,@Id,@Hue,@Page,@Width,@Height,@KeepProportions] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;


        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;
        Image := Gump.AddImage( X, Y, Id, Hue, Page );
        Image.Breite := Width;
        Image.Hoehe := Height;
        Image.KeepProportions := KeepProportions = 1;
        Result := pyGetGumpObject( Image );
end;

{
        \method gump.addtiledimage
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param id
        \param hue
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an tiled image, adds it to the gump and returns it.
}

function pyGump_AddTiledImage( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Hoehe, Breite : Integer;
        Id, Hue, Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iiiihhh:gump.addtiledimage( x, y, width, height, id, hue, page )', [@X,@Y,@Breite,@Hoehe,@Id,@Hue,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddTiledImage( X, Y, Hoehe, Breite, Id, Hue, Page ) );
end;

{
        \method gump.addtilepic
        \param x X position of the button.
        \param y Y position of the button.
        \param id Id of static art tile.
        \param hue
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an tile image (static art), adds it to the gump and returns it.
}

function pyGump_AddTilePic( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Id, Hue, Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhh:gump.addtilepic( x, y, id, hue, page )', [@X,@Y,@Id,@Hue,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddTilePic( X, Y, Id, Hue, Page ) );
end;

{
        \method gump.addlandtilepic
        \param x X position of the button.
        \param y Y position of the button.
        \param id Id of map art tile.
        \param hue
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an land tile image (map art), adds it to the gump and returns it.
}

function pyGump_AddLandTilePic( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Id, Hue, Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhh:gump.addlandtilepic( x, y, id, hue, page )', [@X,@Y,@Id,@Hue,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddLandTilePic( X, Y, Id, Hue, Page ) );
end;

{
        \method gump.addtext
        \param x X position of the button.
        \param y Y position of the button.
        \param text
        \param font Font id of the text.
        \param hue
        \param page
        \param border Determines whether the characters have a black one-pixel border or not. (optional, default=True)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates some text, adds it to the gump and returns it.
}

function pyGump_AddText( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Page : Word;
        Text : PChar;
        Font : Byte;
        Hue : Word;
        Border : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Border := 1;
        if PyArg_ParseTuple( Args, 'iisbhh|b:gump.addtext( x, y, text, font, hue, page, border )', [@X,@Y,@Text,@Font,@Hue,@Page,@Border] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddText( X, Y, Text, Font, Hue, Page, Border = 1 ) );
end;

{
        \method gump.addresizegump
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param id
        \param hue
        \param page
        \param onlyborder If 1, then only the borders are drawn. (optional)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a resizegump (background), adds it to the gump and returns it.
}

function pyGump_AddResizeGump( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Hoehe, Breite : Integer;
        Id, Hue, Page : Word;
        OnlyBorder : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        OnlyBorder := 0;
        if PyArg_ParseTuple( Args, 'iiiihhh|b:gump.addtiledimage( x, y, width, height, id, hue, page, [onlyborder] )', [@X,@Y,@Breite,@Hoehe,@Id,@Hue,@Page,@OnlyBorder] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddResizeGump( X, Y, Id, Breite, Hoehe, Hue, Page, OnlyBorder = 1 ) );
end;

{
        \method gump.addcroppedtext
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param text
        \param font Font id of the text.
        \param hue
        \param page
        \param maxlines Maximum lines that are shown. (optional, default = 1)
        \param border Determines whether the characters have a black one-pixel border or not. (optional, default = 1)
        \param divheightmod Changes the differenz between 2 lines of text by the given number of pixels. (optional, default = 0)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates some cropped text, adds it to the gump and returns it.
}

function pyGump_AddCroppedText( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Page : Word;
        Text : PChar;
        Font : Byte;
        Hue : Word;
        MaxLines : Word;
        Hoehe, Breite : Integer;
        Border : Byte;
        DivHeight : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        MaxLines := 1;
        Border := 1;
        DivHeight := 0;
        if PyArg_ParseTuple( Args, 'iiiisbhh|hbi:gump.addcroppedtext( x, y, width, height, text, font, hue, page, [maxlines] )', [@X,@Y,@Breite,@Hoehe,@Text,@Font,@Hue,@Page,@MaxLines,@Border,@DivHeight] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddCroppedText( X, Y, Hoehe, Breite, Text, Font, Hue, Page, MaxLines, Border = 1, DivHeight ) );
end;

{
        \method gump.addinputfield
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param text
        \param font Font id of the text.
        \param textid Id to identify this inputfield.
        \param hue
        \param page
        \param allowonlynumbers (optional, default = 0)
        \param stars Show stars instead of the real characters. (optional, default = 0)
        \param allowmultilines (optional, default = 0)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an inputfield, adds it to the gump and returns it.
}

function pyGump_AddInputField( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Page : Word;
        Hoehe, Breite : Integer;
        Text : PChar;
        Font : Byte;
        TextId : Integer;
        Hue : Word;
        AllowOnlyNumbers : Byte;
        Stars : Byte;
        AllowMultiLines : Byte;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        AllowOnlyNumbers := 0;
        Stars := 0;
        AllowMultiLines := 0;
        if PyArg_ParseTuple( Args, 'iiiisbihh|bbb:gump.addinputfield( x, y, width, height, text, font, textid, hue, page, [allowonlynumbers], [stars], [allowmultilines] )', [@X,@Y,@Breite,@Hoehe,@Text,@Font,@TextId,@Hue,@Page,@AllowOnlyNumbers,@Stars,@AllowMultiLines] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddInputField( X, Y, Hoehe, Breite, Text, Font, TextId, Hue, Page, AllowOnlyNumbers = 1, Stars = 1, AllowMultiLines = 1 ) );
end;

{
        \method gump.addcheckbox
        \param x X position of the button.
        \param y Y position of the button.
        \param trueid
        \param falseid
        \param checked
        \param returnnumber
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a checkbox, adds it to the gump and returns it.
}

function pyGump_AddCheckBox( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        TrueId, FalseId : Word;
        Checked : Byte;
        ReturnNumber : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhbih:gump.addcheckbox( x, y, trueid, falseid, checked, returnnumber, page )', [@X,@Y,@TrueId,@FalseId,@Checked,@ReturnNumber,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddCheckBox( X, Y, TrueId, FalseId, Checked = 1, ReturnNumber, Page ) );
end;

{
        \method gump.addradiobutton
        \param x X position of the button.
        \param y Y position of the button.
        \param trueid
        \param falseid
        \param checked
        \param returnnumber
        \param group
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a radiobutton, adds it to the gump and returns it.
}

function pyGump_AddRadioButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        TrueId, FalseId : Word;
        Checked : Byte;
        ReturnNumber : Integer;
        Page, Group : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhbihh:gump.addradiobutton( x, y, trueid, falseid, checked, returnnumber, group, page )', [@X,@Y,@TrueId,@FalseId,@Checked,@ReturnNumber,@Group,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddRadioButton( X, Y, TrueId, FalseId, Checked = 1, ReturnNumber, Group, Page ) );
end;

{
        \method gump.addhtmlgump
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param text
        \param hasback
        \param canscroll
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a htmlgump, adds it to the gump and returns it.
}

function pyGump_AddHtmlGump( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Text : PChar;
        Breite, Hoehe : Integer;
        hasBack, canScroll : Byte;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iiiisbbh:gump.addhtmlgump( x, y, width, height, text, hasback, canscroll, page )', [@X,@Y,@Breite,@Hoehe,@Text,@hasBack,@canScroll,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddHtmlGump( X, Y, Text, Breite, Hoehe, hasBack = 1, canScroll = 1, Page ) );
end;

{
        \method gump.addxmfhtmlgump
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param clilocid Id of the text in the clilocfile.
        \param hasback
        \param canscroll
        \param page
        \param hue (optional)
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates an xmfhtmlgump, adds it to the gump and returns it.
}

function pyGump_AddXmfHtmlGump( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Breite, Hoehe : Integer;
        hasBack, canScroll : Byte;
        Page, Color : Word;
        ClilocId : LongWord;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Color := 0;
        if PyArg_ParseTuple( Args, 'iiiiibbh|h:gump.addxmfhtmlgump( x, y, width, height, clilocid, hasback, canscroll, page, [hue] )', [@X,@Y,@Breite,@Hoehe,@ClilocId,@hasBack,@canScroll,@Page,@Color] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddXmfHtmlGump( X, Y, ClilocID, Breite, Hoehe, hasBack = 1, canScroll = 1, Color, Page ) );
end;

{
        \method gump.addcheckertrans
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a checkertrans, adds it to the gump and returns it.
}

function pyGump_AddCheckerTrans( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Hoehe, Breite : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iiiih:gump.addcheckertrans( x, y, width, height, page )', [@X,@Y,@Breite,@Hoehe,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddCheckerTrans( X, Y, Hoehe, Breite, Page ) );
end;

{
        \method gump.addgumpitem
        \param x X position of the image.
        \param y Y position of the image.
        \param serial
        \param pos Position in the drawlist of the gump, where this objects is inserted.
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a gumpitem, adds it to the gump and returns it.
        This is mainly used for the paperdoll.
}

function pyGump_AddGumpItem( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Serial : LongWord;
        Pos : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Pos := -1;
        Page := 0;
        if PyArg_ParseTuple( Args, 'iiiih:gump.addgumpitem( x, y, serial, pos, page )', [@X,@Y,@Serial,@Pos,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;
        Result := pyGetGumpObject( Gump.AddGumpItem( X, Y, Serial, Pos, Page ) );
end;

{
        \method gump.addtransparendgumpitem
        \param x X position of the image.
        \param y Y position of the image.
        \param id
        \param hue
        \param layer
        \param pos Position in the drawlist of the gump, where this objects is inserted.
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a semitransparend gumpitem, adds it to the gump and returns it.
        This is mainly used for the paperdoll.
}

function pyGump_AddTransparendGumpItem( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Id, Hue, Layer : Word;
        Pos : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhhih:gump.addgumpitem( x, y, id, hue, layer, pos, page )', [@X,@Y,@Id,@Hue,@Layer,@Pos,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;
        Result := pyGetGumpObject( Gump.AddGumpItem( X, Y, Id, Hue, Layer, Pos, Page ) );
end;

{
        \method gump.addstaticitem
        \param x X position of the button.
        \param y Y position of the button.
        \param serial
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a staticitem gumpobject, adds it to the gump and returns it.
        This is used for example for containers and secure trading gumps.
}

function pyGump_AddStaticItem( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Serial : LongWord;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Page := 0;
        if PyArg_ParseTuple( Args, 'iiih:gump.addstaticitem( x, y, serial, page )', [@X,@Y,@Serial,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddStaticItem( X, Y, Serial, Page ) );
end;

{
        \method gump.addscrollbar
        \param x X position of the button.
        \param y Y position of the button.
        \param horizontal
        \param length
        \param upid
        \param moveoverid
        \param number
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a scrollbar gumpobject, adds it to the gump and returns it.
}

function pyGump_AddScrollBar( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Horizontal : Byte;
        Length, UpId, MoveOverId : Word;
        Number : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iibhhhih:gump.addscrollbar( x, y, horizontal, length, upid, moveoverid, number, page )', [@X,@Y,@Horizontal,@Length,@UpId,@MoveOverId,@Number,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddScrollBar( X, Y, Horizontal = 1, Length, UpId, MoveOverId, Number, Page ) );
end;

{
        \method gump.addhuebutton
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param red
        \param green
        \param blue
        \param returnnumber
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a huebutton gumpobject, adds it to the gump and returns it.
}

function pyGump_AddHueButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Breite, Hoehe : Word;
        Red, Green, Blue : Byte;
        ReturnNumber : Integer;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhbbbih:gump.addhuebutton( x, y, width, height, red, green, blue, returnnumber, page )', [@X,@Y,@Breite,@Hoehe,@Red,@Green,@Blue,@ReturnNumber,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddHueButton( X, Y, Breite, Hoehe, Red, Green, Blue, ReturnNumber, Page ) );
end;

{
        \method gump.addprogressbar
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param current
        \param max
        \param backlefthue
        \param backrighthue
        \param lefthue
        \param righthue
        \param text
        \param align Either "RIGHT", "LEFT" or "CENTER" as string.
        \param textcolor
        \param horizontal
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a addprogressbar gumpobject, adds it to the gump and returns it.
}

function pyGump_AddProgressBar( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Breite, Hoehe : Word;
        Current, Max, BackLeftColor, BackRightColor, LeftColor, RightColor : LongWord;
        Text, Align : PChar;
        TextColor : LongWord;
        Waagerecht : Byte;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhiiiiiissibh:gump.addprogressbar( x, y, width, height, current, max, backlefthue, backrighthue, lefthue, righthue, text, align, textcolor, horizontal, page )', [@X,@Y,@Breite,@Hoehe,@Current,@Max,@BackLeftColor,@BackRightColor,@LeftColor,@RightColor,@Text,@Align,@TextColor,@Waagerecht,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddProgressBar( X, Y, Breite, Hoehe, Current, Max, BackLeftColor, BackRightColor, LeftColor, RightColor, Text, Align, TextColor, Waagerecht = 1, Page ) );
end;

{
        \method gump.addextimage
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param url
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a addextimage gumpobject, adds it to the gump and returns it.
}

function pyGump_AddExtImage( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Breite, Hoehe : Word;
        URL : PChar;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhsh:gump.addextimage( x, y, width, height, url, page )', [@X,@Y,@Breite,@Hoehe,@URL,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddExtImage( X, Y, Breite, Hoehe, URL, Page ) );
end;

{
        \method gump.addtooltip
        \param x X position of the button.
        \param y Y position of the button.
        \param width
        \param height
        \param text
        \param page
        \return A <object id="gumpobject">gumpobject</object> object.
        \description This method creates a tooltip gumpobject, adds it to the gump and returns it.
        This is just an rectangle, where a tooltip is shown when the mouse stays over it for more than 1 second.
        It always overrides all existing tooltips of non tooltip gumpobjects at its position.
}

function pyGump_AddTooltip( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        X, Y : Integer;
        Breite, Hoehe : Word;
        Text : PChar;
        Page : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iihhsh:gump.addtooltip( x, y, width, height, text, page )', [@X,@Y,@Breite,@Hoehe,@Text,@Page] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.AddTooltip( X, Y, Breite, Hoehe, Text, Page ) );
end;

{
        \method gump.show
        \description Adds the gump to the internal drawlist.
}

function pyGump_Show( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;
                
        Palanthir.AddGump( Gump );

        Result := PyReturnNone;
end;

{
        \method gump.delete
        \description Deletes the gump.
}

function pyGump_Delete( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Palanthir.DeleteGump( Gump );
        Result := PyReturnNone;
end;

{
        \method gump.getobjectcount
        \return int
        \description Returns the number of gumpobject this gump contains.
}

function pyGump_getObjectCount( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := PyInt_FromLong( Gump.GetObjectCount );
end;

{
        \method gump.getobject
        \param index Index of the gumpobject.
        \return Gumpobject A <object id="gumpobject">gumpobject</object> object.
        \description Returns the gumpobject of this gump with the given index.
}

function pyGump_getObject( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        Index : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'i:gump.getobject( index )', [@Index] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Result := pyGetGumpObject( Gump.GetObject( Index ) );
end;

{
        \method gump.deleteobject
        \param gumpobject A <object id="gumpobject">gumpobject</object> object.
        \description Deletes the gumpobject. Only works if the gumpobject is assosiated with this gump.
}

function pyGump_deleteObject( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        GumpObject : TGumpObject;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'O&:gump.deleteobject( gumpobject )', [@PyConvertGumpObject,@GumpObject] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;
        
        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;        
        Gump.DeleteObject( GumpObject );
        Result := PyReturnNone;
end;

{
        \method gump.getbutton
        \param returnnumber
        \return A <object id="gumpobject">button</object> object.
        \description Returns the button with the given returnnumber.
}

function pyGump_getButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        ReturnNumber : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'i:gump.getbutton( returnnumber )', [@ReturnNumber] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;

        Result := PyGetGumpObject( Gump.GetButton( ReturnNumber ) );
end;

{
        \method gump.getinputfieldtext
        \param returnnumber
        \description Returns the text of the inputfield with the given returnnumber.
}

function pyGump_getInputFieldText( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        ReturnNumber : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'i:gump.getinputfieldtext( returnnumber )', [@ReturnNumber] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;

        Result := PyString_FromString( PChar( Gump.GetInputText( ReturnNumber ) ) );
end;

{
        \method gump.getradiobutton
        \param groupid
        \description Returns the text of the inputfield with the given returnnumber.
}

function pyGump_getRadioButton( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        GroupId : Word;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'h:gump.getradiobutton( groupid )', [@GroupId] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnNone;
                Exit;
        end;

        Result := PyInt_FromLong( Gump.GetRadioButton( GroupId ) );
end;

{
        \method gump.setboundingbox
        \param x
        \param y
        \param width
        \param height
        \description Sets the bounding box of the gump. The bounding box has to be enabled by Gump.enableboundingbox.
}

function pyGump_setBoundingBox( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Gump : TPythonGump;
        BBX, BBY, BBWidth, BBHeight : Integer;
begin
        if not pyGump_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'iiii:gump.setboundingbox( x, y, width, height )', [@BBX,@BBY,@BBWidth,@BBHeight] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump := pyGump_getGump( Self );
        if Gump = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Gump.SetBoundingBox( BBX, BBY, BBWidth, BBHeight );
        Result := PyReturnTrue;
end;

end.
