unit uPyItem;

interface

uses    Python, uItem, SysUtils, uPyUtilities, uUObject;

{
        \object item
        \description Represents an item object.
}

type    TpyItem = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                ItemSerial : LongWord;
        end;

        PpyItem = ^TpyItem;

function pyItem_Str( Self : PPyObject ) : PPyObject; cdecl;
procedure pyItem_Dealloc( Self : PPyObject ); cdecl;
function pyItem_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyItem_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyItem_compare( Item1, Item2 : PPyObject ) : Integer; cdecl;
function pyItem_getItem( Self : PPyObject ) : TItem;
function pyItem_check( Self : PPyObject ) : Boolean;
function pyGetItem( Item : TItem ) : PPyObject; cdecl;
function pyConvertItem( PyObject : PPyObject; Item : P_Item ) : Integer; cdecl;

function pyItem_delete( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyItem_moveto( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyItem_additem( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyItem_removeFromCont( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyItem_grabItem( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

const pyItemType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyItem';
		tp_basicsize : sizeof( pyItemType );
                tp_itemsize : 0;

		tp_dealloc : pyItem_Dealloc;
		tp_print : nil;
		tp_getattr : pyItem_getAttr;
		tp_setattr : pyItem_setAttr;
		tp_compare : pyItem_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyItem_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyItemMethods : Array[0..5] of PyMethodDef =
                ( ( ml_name : 'delete'; ml_meth : pyItem_delete; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'moveto'; ml_meth : pyItem_moveto; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'additem'; ml_meth : pyItem_additem; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'removefromcont'; ml_meth : pyItem_removeFromCont; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'grabitem'; ml_meth : pyItem_grabItem; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

uses    uPalanthir, uPyPos, uPyChar, uChar;

function pyItem_Str( Self : PPyObject ) : PPyObject;
var     Item : TItem;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Item := pyItem_getItem( Self );

        if Item <> nil then
                Result := PyString_FromString( PChar( Format( '0x%x', [Item.Serial] ) ) )
        else
                Result := PyReturnNone;
end;

procedure pyItem_Dealloc( Self : PPyObject );
begin
        PyDealloc( Self );
end;

function pyItem_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     PyItem : PpyItem;
        Item : TItem;
        TestName : String;
        I : Integer;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        PyItem := PpyItem( Self );
        Item := pyItem_getItem( Self );

        if Item = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        TestName := LowerCase( Name );

        {
                \property item.id Id
        }
        if TestName = 'id' then begin
                Result := PyInt_FromLong( Item.ID );
        end
        {
                \property item.color Color
        }
        else if TestName = 'color' then begin
                Result := PyInt_FromLong( Item.Color );
        end
        {
                \property item.x X
        }
        else if TestName = 'x' then begin
                Result := PyInt_FromLong( Item.Pos.X );
        end
        {
                \property item.y Y
        }
        else if TestName = 'y' then begin
                Result := PyInt_FromLong( Item.Pos.Y );
        end
        {
                \property item.z Z
        }
        else if TestName = 'z' then begin
                Result := PyInt_FromLong( Item.Pos.Z );
        end
        {
                \property item.map Map
        }
        else if TestName = 'map' then begin
                Result := PyInt_FromLong( Item.Pos.Map );
        end
        {
                \property item.isitem IsItem
        }
        else if TestName = 'isitem' then begin
                Result := pyInt_FromLong( 1 );
        end
        {
                \property item.ismap IsMap
        }
        {
                \property item.isstatic IsStatic
        }
        {
                \property item.ischar IsChar
        }
        {
                \property item.isplayer IsPlayer
        }
        {
                \property item.iseffect IsEffect
        }
        {
                \property item.ismultiitem IsMultiItem
        }
        else if (TestName = 'ismap') or (TestName = 'isstatic') or (TestName = 'ischar') or
                (TestName = 'isplayer') or (TestName = 'iseffect') or (TestName = 'ismultiitem') then begin
                Result := pyInt_FromLong( 0 );
        end
        {
                \property item.amount Amount
        }
        else if TestName = 'amount' then begin
                Result := PyInt_FromLong( Item.Amount );
        end
        {
                \property item.layer Layer
        }
        else if TestName = 'layer' then begin
                Result := PyInt_FromLong( Item.Layer );
        end
        {
                \property item.serial Serial
        }
        else if TestName = 'serial' then begin
                Result := PyInt_FromLong( Item.Serial );
        end
        {
                \property item.container The <object id="item">item</object> or <object id="char">char</object> object that contains this item.
        }
        else if TestName = 'container' then begin
                if Item.Container.IsItem then begin
                        Result := PyGetItem( TItem( Item.Container ) );
                end
                else if Item.Container.IsChar then begin
                        Result := PyGetChar( TChar( Item.Container ) );
                end
                else begin
                        Result := PyReturnNone;
                end;
        end
        {
                \property item.content A list of <object id="item">item</object> objects of the items that are in this container.
        }
        else if Testname = 'content' then begin
                Result := PyTuple_New( Item.Content.Count );
                for I := 0 to Item.Content.Count-1 do begin
                        PyTuple_SetItem( Result, I, PyGetItem( TItem( Item.Content.Items[ I ] ) ) );
                end;
        end
        {
                \property item.customvars A dictionary to save variables on the item. (see cv)
        }
        {
                \property item.cv A dictionary to save variables on the item. (see customvars)
        }
        else if (TestName = 'customvars') or (TestName = 'cv') then begin
                if Item.CustomVars = nil then begin
                        Item.CustomVars := PyDict_New;
                end;

                Py_INCREF( Item.CustomVars );
                Result := Item.CustomVars;
        end
        else begin
                Result := Py_FindMethod( @pyItemMethods, Self, PChar( TestName ) );
        end;
end;

function pyItem_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     PyItem : PpyItem;
        Item : TItem;
        TestName : String;
begin
        if not pyItem_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        PyItem := PpyItem( Self );
        Item := pyItem_getItem( Self );

        if Item = nil then begin
                Result := 1;
                Exit;
        end;

        TestName := LowerCase( Name );

        if TestName = 'id' then begin
                if PyInt_Check( Value ) then begin
                        Item.ID := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'color' then begin
                if PyInt_Check( Value ) then begin
                        Item.Color := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'amount' then begin
                if PyInt_Check( Value ) then begin
                        Item.Amount := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'layer' then begin
                if PyInt_Check( Value ) then begin
                        Item.Layer := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'serial' then begin
                if PyInt_Check( Value ) then begin
                        Item.UnRegisterObject;
                        Item.Serial := PyInt_AsLong( Value );
                        Item.RegisterObject;
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'container' then begin
                if PyItem_Check( Value ) then begin
                        Item.Container := PyItem_getItem( Value );
                        Result := 0;
                end
                else if PyChar_Check( Value ) then begin
                        Item.Container := PyChar_getChar( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else begin
                Result := 1;
        end;
end;


function pyItem_compare( Item1, Item2 : PPyObject ) : Integer;
var     Item_1, Item_2 : TItem;
begin
        if (not pyItem_check( Item1 )) or (not pyItem_check( Item2 )) then begin
                Result := -1;
                Exit;
        end;

        Item_1 := pyItem_getItem( Item1 );
        Item_2 := pyItem_getItem( Item2 );

        if (Item_1 = nil) or (Item_2 = nil) then begin
                Result := -1;
                Exit;
        end;

        if Item_1.Serial = Item_2.Serial then
                Result := 0
        else
                Result := -1;
end;

function pyItem_getItem( Self : PPyObject ) : TItem;
var     UObject : TUObject;
begin
        if not pyItem_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        UObject := Palanthir.Data.GlobalObjectList.GetObject( PpyItem( Self ).ItemSerial );
        if (UObject <> nil) and UObject.IsItem then
                Result := TItem( UObject )
        else
                Result := nil;
end;

function pyItem_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyItemType;
end;

function pyGetItem( Item : TItem ) : PPyObject;
var     PyItem : PpyItem;
begin
        PyItem := PpyItem( PyObject_new( @pyItemType ) );
        PyItem.ItemSerial := Item.Serial;
        Result := PPyObject( PyItem );
end;

function pyConvertItem( PyObject : PPyObject; Item : P_Item ) : Integer;
begin
        if not pyItem_check( PyObject ) then begin
                PyErr_BadArgument;
                Result := 0;
                Exit;
        end;

        Item^ := pyItem_getItem( pyObject );
        Result := 1;
end;

{
        \method item.delete
        \description Deletes the item.
}

function pyItem_delete( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Item : TItem;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item := TItem( Palanthir.Data.GlobalObjectList.GetObject( PPyItem( Self ).ItemSerial ) );
        if Item = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item.Free;
        Result := PyReturnTrue;
end;

{
        \method item.moveto
        \param x
        \param y
        \param z
        \param map (optional, default = currentmap)
        \description Moves the item to (x, y, z, map)
}
function pyItem_moveto( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Item : TItem;
        X, Y, Z : Integer;
        Map : Byte;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item := pyItem_getItem( Self );
        if Item = nil then begin
                Result := PyReturnNone;
                Exit;
        end;

        Map := Palanthir.CurrentMap;
        if PyArg_ParseTuple( Args, 'iii|b:item.moveto( x, y, z, [map] )', [@X,@Y,@Z,@Map] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        Item.MoveTo( X, Y, Z, Map );

        Result := PyReturnNone;
end;

{
        \method item.additem
        \param item An <object id="item">item</object> object
        \description Adds an item to a container.
}
function pyItem_additem( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Cont, Item : TItem;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Cont := pyItem_getItem( Self );
        if Cont = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'O&|b:item.additem( item )', [@pyConvertItem,@Item] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        if Item = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if Item.Container <> nil then
                Item.RemoveFromCont;
        
        Item.UnRegisterObject;

        Item.Container := Cont;
        Item.RegisterObject;
        
        Cont.AddItem( Item );

        Result := PyReturnNone;
end;

{
        \method item.removefromcont
        \description Removes the item from its container.
}
function pyItem_removeFromCont( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Item : TItem;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item := pyItem_getItem( Self );
        if Item = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item.RemoveFromCont;

        Result := PyReturnTrue;
end;

{
        \method item.grabitem
        \description Tries to grab the item.
}
function pyItem_grabItem( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Item : TItem;
begin
        if not pyItem_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item := pyItem_getItem( Self );
        if Item = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Item.GrabItem;

        Result := PyReturnTrue;
end;

end.
