unit uPyPacket;

interface

uses    Python, uPacket, SysUtils, uPyUtilities;

{
        \object packet
        \description Represents a packet object.
        To create a packet use the newpacket function in the <module id="palanthir">palanthir</module> module.
}

type    pyPacket = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                Packet : UOPacket;
        end;

        PpyPacket = ^pyPacket;

function pyPacket_Str( Self : PPyObject ) : PPyObject; cdecl;
function pyPacket_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyPacket_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyPacket_compare( Packet1, Packet2 : PPyObject ) : Integer; cdecl;
function pyPacket_getPacket( Self : PPyObject ) : UOPacket;
function pyPacket_check( Self : PPyObject ) : Boolean;
function pyGetPacketObject( Packet : UOPacket ) : PPyObject; cdecl;
function pyConvertPacketObject( PyObject : PPyObject; Packet : P_UOPacket ) : Integer; cdecl;

function pyPacket_getByte( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_getWord( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_getLongWord( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_getAsciiString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_getUnicodeString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_setByte( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_setWord( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_setLongWord( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_setAsciiString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_setUnicodeString( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPacket_send( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

const pyPacketType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyPacket';
		tp_basicsize : sizeof( pyPacketType );
                tp_itemsize : 0;

		tp_dealloc : PyDealloc;
		tp_print : nil;
		tp_getattr : pyPacket_getAttr;
		tp_setattr : pyPacket_setAttr;
		tp_compare : pyPacket_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyPacket_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyPacketMethods : Array[0..11] of PyMethodDef =
                ( ( ml_name : 'getbyte'; ml_meth : pyPacket_getByte; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getword'; ml_meth : pyPacket_getWord; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getlongword'; ml_meth : pyPacket_getLongWord; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getasciistring'; ml_meth : pyPacket_getAsciiString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'getunicodestring'; ml_meth : pyPacket_getUnicodeString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setbyte'; ml_meth : pyPacket_setByte; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setword'; ml_meth : pyPacket_setWord; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setlongword'; ml_meth : pyPacket_setLongWord; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setasciistring'; ml_meth : pyPacket_setAsciiString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'setunicodestring'; ml_meth : pyPacket_setUnicodeString; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : 'send'; ml_meth : pyPacket_Send; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

uses    uPalanthir;

function pyPacket_Str( Self : PPyObject ) : PPyObject;
var     Packet : PPyPacket;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Packet := PPyPacket( Self );
        Result := PyString_FromString( PChar( Packet.Packet.GetInternalName ) );
end;

function pyPacket_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     Packet : UOPacket;
        TestName : String;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := pyPacket_getPacket( Self );
        TestName := LowerCase( Name );

        {
                \property packet.id Id of the packet.
        }
        if TestName = 'id' then begin
                Result := PyInt_FromLong( Packet.GetByte( 0 ) );
        end
        {
                \property packet.length Length of the packet.
        }
        else if TestName = 'length' then begin
                Result := PyInt_FromLong( Packet.getLength );
        end
        else begin
                Result := Py_FindMethod( @pyPacketMethods, Self, PChar( TestName ) );
        end;
end;

function pyPacket_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     Packet : UOPacket;
        TestName : String;
begin
        if not pyPacket_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        Packet := pyPacket_getPacket( Self );
        TestName := LowerCase( Name );

        if TestName = 'id' then begin
                if PyInt_Check( Value ) then begin
                        Packet.SetByte( 0, PyInt_AsLong( Value ) );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'length' then begin
                if PyInt_Check( Value ) then begin
                        Packet.SetLength( PyInt_AsLong( Value ) );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else begin
                Result := 1;
        end;
end;

function pyPacket_compare( Packet1, Packet2 : PPyObject ) : Integer;
var     P1, P2 : UOPacket;
begin
        if (not pyPacket_check( Packet1 )) and (not pyPacket_check( Packet2 )) then begin
                Result := -1;
                Exit;
        end;

        P1 := pyPacket_getPacket( Packet1 );
        P2 := pyPacket_getPacket( Packet2 );

        if P1 = P2 then begin
                Result := 0;
        end
        else begin
                Result := -1;
        end;
end;

function pyPacket_getPacket( Self : PPyObject ) : UOPacket;
begin
        if not pyPacket_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        Result := PpyPacket( Self ).Packet;
end;

function pyPacket_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyPacketType;
end;

function pyGetPacketObject( Packet : UOPacket ) : PPyObject;
var     PP : PpyPacket;
begin
        PP := PpyPacket( PyObject_new( @pyPacketType ) );
        PP.Packet := Packet;
        Result := PPyObject( PP );
end;

function pyConvertPacketObject( PyObject : PPyObject; Packet : P_UOPacket ) : Integer;
begin
        if not pyPacket_check( PyObject ) then begin
                PyErr_BadArgument;
                Result := 0;
                Exit;
        end;

        Packet^ := pyPacket_getPacket( pyObject );
        Result := 1;
end;

{
        \method packet.getbyte
        \param pos Position within the packet to read
        \return uint8
}

function pyPacket_getByte( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'h:packet.getbyte( pos )', [@Pos] ) = 0 then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Result := PyInt_FromLong( Packet.GetByte( Pos ) );
end;

{
        \method packet.getword
        \param pos Position within the packet to read
        \return uint16
}

function pyPacket_getWord( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'h:packet.getword( pos )', [@Pos] ) = 0 then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Result := PyInt_FromLong( Packet.GetWord( Pos ) );
end;

{
        \method packet.getlongword
        \param pos Position within the packet to read
        \return uint32
}

function pyPacket_getLongWord( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'h:packet.getlongword( pos )', [@Pos] ) = 0 then begin
                Result := PyInt_FromLong( 0 );
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Result := PyInt_FromLong( Packet.GetLongWord( Pos ) );
end;

{
        \method packet.getasciistring
        \param pos Position within the packet to read
        \param length Maximum length of the string
        \return string
}

function pyPacket_getAsciiString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos, Length : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyString_FromString( '' );
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hh:packet.getasciistring( pos, length )', [@Pos,@Length] ) = 0 then begin
                Result := PyString_FromString( '' );
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Result := PyString_FromString( PChar( Packet.GetAsciiString( Pos, Length ) ) );
end;

{
        \method packet.getunicodestring
        \param pos Position within the packet to read
        \param length Maximum length of the string
        \return string
}

function pyPacket_getUnicodeString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos, Length : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyString_FromString( '' );
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hh:packet.getunicodestring( pos, length )', [@Pos,@Length] ) = 0 then begin
                Result := PyString_FromString( '' );
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Result := PyString_FromString( PChar( Packet.GetUnicodeString( Pos, Length ) ) );
end;

{
        \method packet.setbyte
        \param pos Position within the packet to write.
        \param value Value to set.
        \description Sets an uint8 Integer within the packet.
}

function pyPacket_setByte( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
        Value : Byte;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hB:packet.setbyte( pos, value )', [@Pos,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Packet.SetByte( Pos, Value );

        Result := PyReturnTrue;
end;

{
        \method packet.setword
        \param pos Position within the packet to write.
        \param value Value to set.
        \description Sets an uint16 Integer within the packet.
}

function pyPacket_setWord( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
        Value : Word;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hH:packet.setword( pos, value )', [@Pos,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Packet.SetWord( Pos, Value );

        Result := PyReturnTrue;
end;

{
        \method packet.setlongword
        \param pos Position within the packet to write.
        \param value Value to set.
        \description Sets an uint32 Integer within the packet.
}

function pyPacket_setLongWord( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos : Word;
        Value : LongWord;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hI:packet.setlongword( pos, value )', [@Pos,@Value] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Packet.SetLongWord( Pos, Value );

        Result := PyReturnTrue;
end;

{
        \method packet.setbyte
        \param pos Position within the packet to write.
        \param value Value to set.
        \description Sets an uint8 Integer within the packet.
}

function pyPacket_setAsciiString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos, Length : Word;
        Value : PChar;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hsh:packet.setasciistring( startpos, text, length )', [@Pos,@Value,@Length] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Packet.SetAsciiString( Value, Pos, Length );

        Result := PyReturnTrue;
end;

{
        \method packet.setasciistring
        \param pos Position within the packet to write.
        \param value Value to be set.
        \param length Maximum length of the string.
        \description Sets an ascii string within the packet.
}

function pyPacket_setUnicodeString( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
        Pos, Length : Word;
        Value : PChar;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'hsh:packet.setunicodestring( startpos, text, length )', [@Pos,@Value,@Length] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Packet.SetUnicodeString( Value, Pos, Length );

        Result := PyReturnTrue;
end;

{
        \method packet.setunicodestring
        \param pos Position within the packet to write.
        \param value Value to be set.
        \param length Maximum length of the string.
        \description Sets an unicode string within the packet.
}

function pyPacket_send( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Packet : UOPacket;
begin
        if not pyPacket_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Packet := PPyPacket( Self ).Packet;
        Palanthir.SendPacket( Packet );

        Result := PyReturnTrue;
end;

{
        \method packet.send
        \description Sends the packet to the server.
}

end.
