unit uPyPos;

interface

uses    Python, uPos, SysUtils, uPyUtilities, uUObject;

{
        \object Pos
        \description Represents a Pos object.
}

type    TpyPos = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                Pos : TPos;
        end;

        PpyPos = ^TpyPos;

function pyPos_Str( Self : PPyObject ) : PPyObject; cdecl;
procedure pyPos_Dealloc( Self : PPyObject ); cdecl;
function pyPos_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyPos_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyPos_compare( Pos1, Pos2 : PPyObject ) : Integer; cdecl;
function pyPos_getPos( Self : PPyObject ) : TPos;
function pyPos_check( Self : PPyObject ) : Boolean;
function pyGetPos( Pos : TPos ) : PPyObject; cdecl;
function pyConvertPosObject( PyObject : PPyObject; Pos : P_Pos ) : Integer; cdecl;

function pyPos_getBlockID( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPos_getDistanceTo( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;
function pyPos_isInRechteckRange( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

const pyPosType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyPos';
		tp_basicsize : sizeof( pyPosType );
                tp_itemsize : 0;

		tp_dealloc : pyPos_Dealloc;
		tp_print : nil;
		tp_getattr : pyPos_getAttr;
		tp_setattr : pyPos_setAttr;
		tp_compare : pyPos_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyPos_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyPosMethods : Array[0..3] of PyMethodDef =
                ( ( ml_name :'getblockid'; ml_meth : pyPos_getBlockID; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name :'getdistanceto'; ml_meth : pyPos_getDistanceTo; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name :'isinrectrange'; ml_meth : pyPos_isInRechteckRange; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

uses    uPalanthir;

function pyPos_Str( Self : PPyObject ) : PPyObject;
var     Pos : TPos;
begin
        if not pyPos_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        Pos := pyPos_getPos( Self );

        if Pos <> nil then
                Result := PyString_FromString( PChar( Format( '%d-%d-%d-%d', [Pos.X,Pos.Y,Pos.Z,Pos.Map] ) ) )
        else
                Result := PyReturnNone;
end;

procedure pyPos_Dealloc( Self : PPyObject );
begin
        PyDealloc( Self );
end;

function pyPos_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     PyPos : PpyPos;
        Pos : TPos;
        TestName : String;
begin
        if not pyPos_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        PyPos := PpyPos( Self );
        Pos := pyPos_getPos( Self );

        if Pos = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        TestName := LowerCase( Name );

        {
                \property pos.x (readonly)
        }
        if TestName = 'x' then begin
                Result := PyInt_FromLong( Pos.X );
        end
        {
                \property pos.y (readonly)
        }
        else if TestName = 'y' then begin
                Result := PyInt_FromLong( Pos.Y );
        end
        {
                \property pos.z (readonly)
        }
        else if TestName = 'z' then begin
                Result := PyInt_FromLong( Pos.Z );
        end
        {
                \property pos.map (readonly)
        }
        else if TestName = 'map' then begin
                Result := PyInt_FromLong( Pos.Map );
        end
        else begin
                Result := Py_FindMethod( @pyPosMethods, Self, PChar( TestName ) );
        end;
end;

function pyPos_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     PyPos : PpyPos;
        Pos : TPos;
        TestName : String;
begin
        if not pyPos_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        PyPos := PpyPos( Self );
        Pos := pyPos_getPos( Self );

        if Pos = nil then begin
                Result := 1;
                Exit;
        end;

        TestName := LowerCase( Name );
        {if TestName = 'x' then begin
                if PyInt_Check( Value ) then begin
                        Pos.X := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'y' then begin
                if PyInt_Check( Value ) then begin
                        Pos.Y := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'z' then begin
                if PyInt_Check( Value ) then begin
                        Pos.Z := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'map' then begin
                if PyInt_Check( Value ) then begin
                        Pos.Map := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else begin
                Result := 1;
        end;  }
        Result := 1;
end;


function pyPos_compare( Pos1, Pos2 : PPyObject ) : Integer;
var     Pos_1, Pos_2 : TPos;
begin
        if (not pyPos_check( Pos1 )) or (not pyPos_check( Pos2 )) then begin
                Result := -1;
                Exit;
        end;

        Pos_1 := pyPos_getPos( Pos1 );
        Pos_2 := pyPos_getPos( Pos2 );

        if (Pos_1 = nil) or (Pos_2 = nil) then begin
                Result := -1;
                Exit;
        end;

        if Pos_1.Equals( Pos_2 ) then
                Result := 0
        else
                Result := -1;
end;

function pyPos_getPos( Self : PPyObject ) : TPos;
var     UObject : TUObject;
begin
        if not pyPos_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        Result := PpyPos( Self ).Pos;
end;

function pyPos_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyPosType;
end;

function pyGetPos( Pos : TPos ) : PPyObject;
var     PyPos : PpyPos;
begin
        PyPos := PpyPos( PyObject_new( @pyPosType ) );
        PyPos.Pos := Pos;
        Result := PPyObject( PyPos );
end;

function pyConvertPosObject( PyObject : PPyObject; Pos : P_Pos ) : Integer;
begin
        if not pyPos_check( PyObject ) then begin
                PyErr_BadArgument;
                Result := 0;
                Exit;
        end;

        Pos^ := pyPos_getPos( pyObject );
        Result := 1;
end;

{
        \method pos.getblockid
        \return Integer
        \description Returns the id of the worldblock where the position is. 
}

function pyPos_getBlockID( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Pos : TPos;
begin
        if not pyPos_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Pos := PPyPos( Self ).Pos;
        Result := PyInt_FromLong( Pos.GetBlockID );
end;

{
        \method pos.getdistanceto
        \param pos Target position.
        \return Double
        \description Returns distance between the to positions. 
}

function pyPos_getDistanceTo( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Pos, Pos2 : TPos;
begin
        if not pyPos_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'O&:pos.distanceto( pos )', [@pyConvertPosObject,@Pos2] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Pos := PPyPos( Self ).Pos;

        Result := PyFloat_FromDouble( Pos.DistanceTo( Pos2 ) );
end;

{
        \method pos.isinrectrange
        \param pos Target position.
        \param range Range to check.
        \return bool
        \description Returns 1 if the position is within a square around the given position with the given range.  
}

function pyPos_isInRechteckRange( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     Pos, Pos2 : TPos;
        Range : Integer;
begin
        if not pyPos_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        if PyArg_ParseTuple( Args, 'O&i:pos.distanceto( pos )', [@pyConvertPosObject,@Pos2,@Range] ) = 0 then begin
                Result := PyReturnFalse;
                Exit;
        end;

        Pos := PPyPos( Self ).Pos;

        if Pos.IsInRechteckRange( Pos2, Range ) then
                Result := PyInt_FromLong( 1 )
        else
                Result := PyInt_FromLong( 0 );
end;

end.
