unit uPyStatic;

interface

uses    Python, uBasicTypedefs, SysUtils, uPyUtilities, uUObject;

{
        \object static
        \description Represents a static object.
}

type    TpyStatic = Record
                ob_refcnt : Integer;
		ob_type : PPyTypeObject;
                StaticItem : TStaticItem;
        end;

        PpyStatic = ^TpyStatic;

function pyStatic_Str( Self : PPyObject ) : PPyObject; cdecl;
procedure pyStatic_Dealloc( Self : PPyObject ); cdecl;
function pyStatic_getAttr( Self : PPyObject; Name : PChar ) : PPyObject; cdecl;
function pyStatic_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer; cdecl;
function pyStatic_compare( Static1, Static2 : PPyObject ) : Integer; cdecl;
function pyStatic_getStatic( Self : PPyObject ) : TStaticItem;
function pyStatic_check( Self : PPyObject ) : Boolean;
function pyGetStatic( StaticItem : TStaticItem ) : PPyObject; cdecl;
function pyConvertStatic( PyObject : PPyObject; StaticItem : P_StaticItem ) : Integer; cdecl;

function pyStatic_moveto( Self : PPyObject; Args : PPyObject ) : PPyObject; cdecl;

const pyStaticType : PyTypeObject = (
		ob_refcnt : 0;
		ob_type : nil;
		ob_size : 0;
		tp_name : 'pyStatic';
		tp_basicsize : sizeof( pyStaticType );
                tp_itemsize : 0;

		tp_dealloc : pyStatic_Dealloc;
		tp_print : nil;
		tp_getattr : pyStatic_getAttr;
		tp_setattr : pyStatic_setAttr;
		tp_compare : pyStatic_compare;
		tp_repr : nil;

		tp_as_number : nil;
		tp_as_sequence : nil;
		tp_as_mapping : nil;

		tp_hash : nil;
		tp_call : nil;
		tp_str : pyStatic_Str;
		tp_getattro : nil;
		tp_setattro : nil;

		tp_as_buffer : nil;
		tp_flags : 0;
		tp_doc : nil;

		tp_traverse : nil;

		tp_clear : nil;
		tp_richcompare : nil;

		tp_weaklistoffset : 0;

		tp_iter : nil;
		tp_iternext : nil;

		tp_methods : nil;
		tp_members : nil;
		tp_getset : nil;
		tp_base : nil;
		tp_dict : nil;
		tp_descr_get : nil;
		tp_descr_set : nil;
		tp_dictoffset : 0;
		tp_init : nil;
		tp_alloc : nil;
		tp_new : nil;
		tp_free : nil;
		tp_is_gc : nil;
		tp_bases : nil;
		tp_mro : nil;
		tp_cache : nil;
		tp_subclasses : nil;
		tp_weaklist : nil;
        );

        pyStaticMethods : Array[0..1] of PyMethodDef =
                ( ( ml_name : 'moveto'; ml_meth : pyStatic_moveto; ml_flags : METH_VARARGS; ml_doc : nil ),
                  ( ml_name : nil; ml_meth : nil; ml_flags : 0; ml_doc : nil ) );

implementation

uses    uPalanthir, uPyPos;

function pyStatic_Str( Self : PPyObject ) : PPyObject;
var     StaticItem : TStaticItem;
begin
        if not pyStatic_check( Self ) then begin
                Result := PyReturnNone;
                Exit;
        end;

        StaticItem := pyStatic_getStatic( Self );

        if StaticItem <> nil then
                Result := PyString_FromString( PChar( Format( '%d %d %d 0x%x', [StaticItem.Pos.X,StaticItem.Pos.Y,StaticItem.Pos.Z,StaticItem.Id] ) ) )
        else
                Result := PyReturnNone;
end;

procedure pyStatic_Dealloc( Self : PPyObject );
begin
        PyDealloc( Self );
end;

function pyStatic_getAttr( Self : PPyObject; Name : PChar ) : PPyObject;
var     pyStatic : PpyStatic;
        StaticItem : TStaticItem;
        TestName : String;
begin
        if not pyStatic_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        pyStatic := PpyStatic( Self );
        StaticItem := pyStatic_getStatic( Self );

        if StaticItem = nil then begin
                Result := PyReturnFalse;
                Exit;
        end;

        TestName := LowerCase( Name );

        {
                \property static.id Id
        }
        if TestName = 'id' then begin
                Result := PyInt_FromLong( StaticItem.ID );
        end
        {
                \property static.color Color
        }
        else if TestName = 'color' then begin
                Result := PyInt_FromLong( StaticItem.Color );
        end
        {
                \property static.x X
        }
        else if TestName = 'x' then begin
                Result := PyInt_FromLong( StaticItem.Pos.X );
        end
        {
                \property static.y Y
        }
        else if TestName = 'y' then begin
                Result := PyInt_FromLong( StaticItem.Pos.Y );
        end
        {
                \property static.z Z
        }
        else if TestName = 'z' then begin
                Result := PyInt_FromLong( StaticItem.Pos.Z );
        end
        {
                \property static.map Map
        }
        else if TestName = 'map' then begin
                Result := PyInt_FromLong( StaticItem.Pos.Map );
        end
        {
                \property static.isstatic IsStatic
        }
        else if TestName = 'isstatic' then begin
                Result := pyInt_FromLong( 1 );
        end
        {
                \property static.isitem IsItem
        }
        {
                \property static.ischar IsChar
        }
        {
                \property static.isplayer IsPlayer
        }
        {
                \property static.iseffect IsEffect
        }
        {
                \property static.ismultiitem IsMultiItem
        }
        else if (TestName = 'isitem') or (TestName = 'isstatic') or (TestName = 'ischar') or
                (TestName = 'isplayer') or (TestName = 'iseffect') or (TestName = 'ismultiitem') then begin
                Result := pyInt_FromLong( 0 );
        end
        {
                \property static.customvars A dictionary to save variables on the item. (see cv)
        }
        {
                \property static.cv A dictionary to save variables on the item. (see customvars)
        }
        else if (TestName = 'customvars') or (TestName = 'cv') then begin
                if StaticItem.CustomVars = nil then begin
                        StaticItem.CustomVars := PyDict_New;
                end;

                Py_INCREF( StaticItem.CustomVars );
                Result := StaticItem.CustomVars;
        end
        else begin
                Result := Py_FindMethod( @pyStaticMethods, Self, PChar( TestName ) );
        end;
end;

function pyStatic_setAttr( Self : PPyObject; Name : PChar; Value : PPyObject ) : Integer;
var     pyStatic : PpyStatic;
        StaticItem : TStaticItem;
        TestName : String;
begin
        if not pyStatic_check( Self ) then begin
                Result := 1;
                Exit;
        end;

        pyStatic := PpyStatic( Self );
        StaticItem := pyStatic_getStatic( Self );

        if StaticItem = nil then begin
                Result := 1;
                Exit;
        end;

        TestName := LowerCase( Name );

        if TestName = 'id' then begin
                if PyInt_Check( Value ) then begin
                        StaticItem.ID := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else if TestName = 'color' then begin
                if PyInt_Check( Value ) then begin
                        StaticItem.Color := PyInt_AsLong( Value );
                        Result := 0;
                end
                else begin
                        Result := 1;
                end;
        end
        else begin
                Result := 1;
        end;
end;


function pyStatic_compare( Static1, Static2 : PPyObject ) : Integer;
var     Static_1, Static_2 : TStaticItem;
begin
        if (not pyStatic_check( Static1 )) or (not pyStatic_check( Static2 )) then begin
                Result := -1;
                Exit;
        end;

        Static_1 := pyStatic_getStatic( Static1 );
        Static_2 := pyStatic_getStatic( Static2 );

        if (Static_1 = nil) or (Static_2 = nil) then begin
                Result := -1;
                Exit;
        end;

        if Static_1 = Static_2 then
                Result := 0
        else
                Result := -1;
end;

function pyStatic_getStatic( Self : PPyObject ) : TStaticItem;
begin
        if not pyStatic_check( Self ) then begin
                Result := nil;
                Exit;
        end;

        Result := PpyStatic( Self ).StaticItem;
end;

function pyStatic_check( Self : PPyObject ) : Boolean;
begin
        Result := Self.ob_type = @pyStaticType;
end;

function pyGetStatic( StaticItem : TStaticItem ) : PPyObject;
var     pyStatic : PpyStatic;
begin
        pyStatic := PpyStatic( PyObject_new( @pyStaticType ) );
        pyStatic.StaticItem := StaticItem;
        Result := PPyObject( pyStatic );
end;

function pyConvertStatic( PyObject : PPyObject; StaticItem : P_StaticItem ) : Integer;
begin
        if not pyStatic_check( PyObject ) then begin
                PyErr_BadArgument;
                Result := 0;
                Exit;
        end;

        StaticItem^ := pyStatic_getStatic( pyObject );
        Result := 1;
end;

{
        \method static.moveto
        \param x
        \param y
        \param z
        \param map (optional, default = currentmap)
        \description Moves the static item to (x, y, z, map) 
}
function pyStatic_moveto( Self : PPyObject; Args : PPyObject ) : PPyObject;
var     StaticItem : TStaticItem;
        X, Y, Z : Integer;
        Map : Byte;
begin
        if not pyStatic_check( Self ) then begin
                Result := PyReturnFalse;
                Exit;
        end;

        StaticItem := pyStatic_getStatic( Self );

        Map := Palanthir.CurrentMap;
        if PyArg_ParseTuple( Args, 'iii|b:static.moveto( x, y, z, [map] )', [@X,@Y,@Z,@Map] ) = 0 then begin
                Result := PyReturnNone;
                Exit;
        end;

        StaticItem.MoveTo( X, Y, Z, Map );
end;

end.
