Unit StrAdapter;

Interface

Uses
  StrAdapterBase,
  Classes,         // for TStrings
  SysUtils;        // for Exception

Type
    // base exception class for all string adapter exceptions
  EStringAdapterException = Class(Exception);

    // String Adapter exception thrown on unknown string encoding types.
    //
    // Most common cause for this exception is the missing of the
    // string coding units (e.g. StrAdapterISO8859_1.pas) in the project
    // file. All string adapter units support a special coding will register
    // under help of the initialization part of a unit and due this fact,
    // the units must be included in the project file to get compiled and
    // registered.
  ESAWrongEncodingException = Class(EStringAdapterException);

  
  TStringAdapter = Class
  Private
    Procedure    SetEncoding(Const Value: String);

  Protected
    fEncoding    : String;
    fUseEntities : Boolean;
    fEncoder     : TBaseStringAdapter;

    Function    SelectStringAdapter(Const AEncoding : String; Const UseEntities : Boolean): TBaseStringAdapter;

  Public
    Constructor Create(Const AEncoding : String; Const AUseEntities : Boolean = True);
    Destructor  Destroy; Override;

    Class Procedure RegisterAdapter(AdapterClass : TStringAdapterMetaClass);
    Class Function  GetRegisteredAdapters : TStrings;
    Class Function  IsCodingAvailable(Const ACodingID : String): Boolean;

      // decodes a string (replacing e.g. the &amp; etc with the chars &)
    Function    StringDecode(Const ACodedString : String): String;
      // encodes a string (replacing e.g. the chars & etc with the &amp;)
    Function    StringEncode(Const AEncodedString : String): String;

  Published
    Property    EncodingID : String        Read  fEncoding
                                           Write SetEncoding;
    Property    UseEntities : Boolean      Read  fUseEntities;
  End;

Implementation

Uses
  StrAdapterDefault;     // damit sich wenigstens der default registriert...

Var
  StrAdapters : Array Of TStringAdapterMetaClass;

{ TStringAdapter }

Constructor TStringAdapter.Create(Const AEncoding: String; Const AUseEntities : Boolean = True);
Begin
  Inherited Create;

  fEncoding    := AEncoding;
  fUseEntities := AUseEntities;
  fEncoder     := SelectStringAdapter(fEncoding, fUseEntities);
  If Not Assigned(fEncoder) Then
    Raise ESAWrongEncodingException.Create('unknown encoding type!');
End;

Destructor TStringAdapter.Destroy;
Begin
  If Assigned(fEncoder) Then
    fEncoder.Free;

  Inherited Destroy;
End;

Class Function TStringAdapter.GetRegisteredAdapters: TStrings;
Var
  i : Integer;
Begin
  Result := TStringList.Create;

  For i := Low(StrAdapters) To High(StrAdapters) Do
    Result.Add(StrAdapters[i].EncodingID);
End;

Class Function TStringAdapter.IsCodingAvailable(Const ACodingID: String): Boolean;
Var
  i : Integer;
Begin
  Result := False;

  For i := Low(StrAdapters) To High(StrAdapters) Do
    If ( UpperCase(StrAdapters[i].EncodingID) = UpperCase(ACodingID) ) Then
    Begin
      Result := True;
      Break;
    End;
End;

Class Procedure TStringAdapter.RegisterAdapter(AdapterClass: TStringAdapterMetaClass);
Var
  i : Integer;
Begin
    // already registered?
  For i := Low(StrAdapters) To High(StrAdapters) Do
    If ( StrAdapters[i] = AdapterClass ) Then
      Exit;

  SetLength(StrAdapters, Length(StrAdapters)+1);
  StrAdapters[Length(StrAdapters)-1] := AdapterClass;
End;

Function TStringAdapter.SelectStringAdapter(Const AEncoding: String; Const UseEntities : Boolean): TBaseStringAdapter;
Var
  l_Encoding : String;
  i          : Integer;
Begin
  l_Encoding := UpperCase(Trim(AEncoding));

  For i := Low(StrAdapters) To High(StrAdapters) Do
    If ( UpperCase(StrAdapters[i].EncodingID) = l_Encoding ) Then
    Begin
      Result := StrAdapters[i].Create(UseEntities);
      Exit;
    End;

  Raise ESAWrongEncodingException.Create('unknown encoding type!');
End;

Procedure TStringAdapter.SetEncoding(const Value: String);
Begin
  If Assigned(fEncoder) Then
    fEncoder.Free;
  fEncoder  := SelectStringAdapter(Value, True);
  fEncoding := Value;
End;

Function TStringAdapter.StringDecode(Const ACodedString: String): String;
Begin
  If Assigned(fEncoder) Then
    Result := fEncoder.Decode(ACodedString)
  Else
    Result := ACodedString;
End;

Function TStringAdapter.StringEncode(Const AEncodedString: String): String;
Begin
  If Assigned(fEncoder) Then
    Result := fEncoder.Encode(AEncodedString)
  Else
    Result := AEncodedString;
End;

End.
