// Copyright
//
//   This unit is released as freeware for all non-commercial projects. All
//   commercial projects using this unit have to inform the author - the usage
//   in that projects is allowed afterwards. If this won't be done - the usage
//   is illegal! 
//
// Features
//   reading / writing of keys:
//
//   all used paths have the format of
//      entity.subentity.subsubentity.name
//   this will be the element
//      <entity>
//        <subentity>
//          <subsubentity>
//            <name>value</name>
//          </subsubentity>
//        </subentity>
//      </entity>
//
//    to read / write sub entity on the same level with equal names use
//    the SubIndex parameter.
//
//    Another possibility to access sub entities on the same level with
//    equal names in paths is to define the sub-index with brackets inside
//    of the path:
//       example: entity.subentity[1].subentity.name
//    The index in the brackets is zero-based (like the sub-index parameter)
//    and can be passed inside of the path.
//
//    Due the fact of using the point "." in path parameters, this library
//    has up to now no possibility to handle valid node/attribute/entity names
//    containing a point. This fact is marked as todo and I will find a solution
//    by time. Use another library without this feature to work with xml files
//    using a point in names.
//
// Author
//
//   Muetze1@gmx.de
//
// Thanks
//
//   UTF-8 coding unit with friendly support created by Daniel Berghold.
//   contact: berdac02@htl-kaindorf.ac.at
//
//   Bug report by Jeri Lynn about sub indexing in node paths and
//   CreatePathAndNode()
//
// Disclaimer
//
//   The source code is distributed as it is, no warrenty implied or
//   otherwise is given.  The author is not liable for any loss or damage
//   of data and other things caused by use of the source.
//
//   If there is any error / mistake / malfunction in the source, the author
//   will remove it on notification - but there will be no liability from the
//   notification to the author to remove it immediatly.
//
//   Accept this rules or don't use it!
//
// History
//
//   Started: Rostock, the 28. April 2003
//
// $Id: XMLLib.pas,v 1.29 2005-11-16 23:46:50+01 muetze1 Exp muetze1 $
//

// if the following compiler define exists - the AttributeByName[] function
// of the TXMLAttributeList will always return a TXMLAttribute - if the name
// doesn't exist, it will be created.
{$define CREATEUNKOWNXMLATTR}

// if the following compiler define exists - the root node will return attributes.
// if the root node has attributes on reading a xml structure and the compiler
// define is not set, no exception or error will be created, but accessing the
// attribute property of the root node will raise an exception.
{$define ALLOWATTRIBUTEONROOT}

// if the following compiler define exists - all functions will check the validity
// of data. It enables the name checks for the nodes and attributes, values, paths,
// etc.
{$define DOVALIDITYCHECKS}

// if the following compiler define exists - all node and attribute names will be
// case-insensitive. The XML standard requires differences in case to be handled
// as different nodes.
{$define NAMESCASEINSENSITIVE}


Unit XMLLib;

Interface

{%ToDo 'XMLLib.todo'}

Uses
  Classes,      // for TStrings
  SysUtils,     // for Exception
  StrAdapter,
  ContNrs;      // for TObjectList

  // Exceptions
Type
    // base exception class for all XML Lib exceptions
  EXMLLibException     = Class(Exception);
    // an invalid node path was passed to the library
    // mostly created when using a wrong root key or adding an second
    // root key
  EXMLInvalidNodePath  = Class(EXMLLibException);
    // an invalid operation occured
  EXMLInvalidOperation = Class(EXMLLibException);
    // This exception will be raised on renaming a node name's to a existing
    // node name.
  EXMLNodeNameExists = Class(EXMLLibException);
    // node not found
  EXMLNodeNotFound = Class(EXMLLibException);
    // node name invalid
  EXMLNodeNameInvalid = Class(EXMLLibException);
    // attribute name already exist
  EXMLNodeAttrNameExists = Class(EXMLLibException);
    // attribute name invalid
  EXMLNodeAttrNameInvalid = Class(EXMLLibException);

Type
  TXMLNode = Class;            //## forward declaration
  TXMLAttributeList = Class;   //## forward declaration

    // The TFilterMode enumeration defines the search behaviour on finding
    // string elements.
    //
    // Parameters
    //   fmCaseSensitive   - The string comparsion should be case sensitive.
    //   fmCaseInsensitve  - The string comparsion should be case insensitive.
    //
    // See also
    //   TXMLNodeList.GetFilteredList(), TXMLNode.GetChilds(), TXMLCommentList.IndexOf()
  TFilterMode = ( fmCaseSensitive, fmCaseInsensitive );

    // The TAttrSetMode defines the behaviour when setting attributes
    // to a node.
    //
    // Parameters
    //   asmAddAttributes       - The new attributes should be added to list
    //                            instead overwriting the existing attributes.
    //   asmOverwriteAttributes - The attributes should overwrite the existing
    //                            attributes. All existing attributes will be
    //                            deleted before adding the new attributes.
  TAttrSetMode = ( asmAddAttributes, asmOverwriteAttributes );

    // handling on non existent paths/nodes
  TXMLNodeHandling = ( nhReturnNil, nhRaiseException, nhCreateNode );

    // Description
    //   Class for handling and holding comments of the XML file. The comments
    //   could be added, removed and modified with this class. One node can have
    //   more than one comment, the class holds a list of TStrings instances -
    //   one instance for one comment.
    //
    // See also
    //   TXMLNode.Comments
  TXMLCommentList = Class
  Protected
      // object for holding the TStrings objects (the comments) for the node.
    fComments : TObjectList;

  Private
    Function  GetComment(Index: Integer): TStrings;
    Function  GetCount: Integer;
    Procedure SetComment(Index: Integer; const Value: TStrings);
    function  GetCommentOnText(Index: String): TStrings;

  Public
    Constructor Create; Overload; Virtual;
    Constructor Create(AComments : TXMLCommentList); Overload; Virtual;

    Destructor  Destroy; Override;

    Function  Add(Comment : TStrings): Integer;
    Procedure Delete(Const AIndex : Integer);
    Function  IndexOf(Comment : TStrings): Integer; Overload;
    Function  IndexOf(Const ACommentText : String; Const ACaseMode: TFilterMode = fmCaseSensitive): Integer; Overload;
    Procedure Clear;

      // All the comments are accessible with the Comments[] property. All
      // comments could be iterated with this property.
      //
      // Parameters
      //   The Index is the index of the comment to retrieve in the list. The
      //   can be in the range between 0 and Count-1, all other indexes will
      //   a EStringListError.
      //
      // Returns
      //   The comment at the index Index.
      //
      // Example
      //   Iteration through all items.
      //   <CODE>
      //     Procedure ShowAllItems;
      //     Var
      //       i : Integer;
      //     Begin
      //       For i := 0 To Pred(CommentList.Count) Do
      //         ShowMessage(CommentList.Comments[i].Text);
      //     End;
      //   </CODE>
      //
      // Remarks
      //   The Comments is the default property and can be left out. The access
      //   to the comments with CommentList.Comments[2] is the same as
      //   CommentList[2].
    Property  Comments[Index : Integer] : TStrings   Read  GetComment
                                                     Write SetComment; Default;

      // The Texts[] property has the same functionality as the Comments[]
      // property, but Texts[] property takes the comment itself to find the
      // according TXMLComment instance. If the comment is not in the list, the
      // property returns nil.
      //
      // Parameters
      //   Index - The comment to find
      //
      // Returns
      //   The TXMLComment instance or nil otherwise.
    Property  Texts[Index : String] : TStrings       Read  GetCommentOnText;

      // The Count read only property contains the amount of elements in the
      // comment list class.
      //
      // Returns
      //   Returns the comment count in the list.
      //
      // Example
      //   An example is given at the Comments[] property.
    Property  Count : Integer                        Read  GetCount;
  End;

    // Description
    //   Class for handling and holding XML sub nodes of nodes. The nodes could
    //   be added, removed, found and modified with this class.
    //
    // See also
    //   TXMLNode.Nodes
  TXMLNodeList = Class
  Protected
      // object for holding the TXMLNode objects of the list.
    fNodes : TObjectList;

    Function  GetXMLNodeStringSubIndex(ANodeName: String; Const ASubIndex : Integer = 0): TXMLNode; Virtual;

  Private
    Function  GetCount: Integer;
    Function  GetXMLNode(Index: Integer): TXMLNode;
    Function  GetXMLNodeString(Index: String): TXMLNode;

  Public
    Constructor Create(Const ACanDestroy : Boolean = True); Overload; Virtual;
    Constructor Create(ANewParent : TXMLNode; ANodeList : TXMLNodeList; Const ACanDestroy : Boolean = True); Overload; Virtual;
    Destructor  Destroy; Override;

    Function  Add(ANode : TXMLNode): Integer; Virtual;
    Procedure Delete(Const AIndex : Integer); Overload; Virtual;
    Procedure Delete(Var ANode: TXMLNode); Overload; Virtual;
    Function  IndexOf(ANode : TXMLNode): Integer; Overload; Virtual;
    Function  IndexOf(Const ANodeName : String; Const ASubIndex : Integer = -1): Integer; Overload; Virtual;

    Function  Exists(Const ANodeName : String): Boolean;

    Function  GetFilteredList(Const AFilter : String;
                              Const AMode : TFilterMode = fmCaseInsensitive): TXMLNodeList; Virtual;

    Function  GetNodeNames : TStrings; Virtual;

    Function  SubNodeCount(Const ANodeName : String): Integer; Virtual;
    Function  GetNode(Const ANodeName : String; Const ASubIndex : Integer = -1): TXMLNode; Virtual;

    Procedure Clear; Virtual;

      // All the nodes are accessible with the Nodes[] property. This function
      // is useful to iterate through all nodes.
      //
      // Parameters
      //   The Index is the index of the node to retrieve in the list. The
      //   can be in the range between 0 and Count-1, all other indexes will
      //   raise a exception.
      //
      // Returns
      //   The node at the specified index or nil otherwise.
      //
      // Example
      //   Iteration through all items.
      //   <CODE>
      //     Procedure ShowAllNodes;
      //     Var
      //       i : Integer;
      //     Begin
      //       For i := 0 To Pred(NodeList.Count) Do
      //         ShowMessage(NodeList.Nodes[i].Name);
      //     End;
      //   </CODE>
      //
      // Remarks
      //   The Nodes is the default property and can be left out. The access
      //   to the nodes with NodesList.Nodes[2] is the same as Nodes[2].
    Property  Nodes[Index : Integer] : TXMLNode        Read  GetXMLNode; Default;

      // Use this function if you want to retrieve the nodes by name and
      // without taking care of nodes with the same name.
      //
      // Parameters
      //   The Index is the name of the node to retrieve in the list.
      //
      // Returns
      //   The node at the specified index or nil otherwise.
    Property  NodesByName[Index : String] : TXMLNode   Read  GetXMLNodeString;

      // Returns the number of nodes in the list.
    Property  Count : Integer                          Read  GetCount;
  End;

    // The TXMLValue class is used for representing any data values in any
    // kind in the xml file. This can be the value of a node or the value
    // of an attribute. Data conversion routines are implemented for easy
    // access to the data.
  TXMLValue = Class
  Private
    fValue     : String;
    fUseCDATA  : Boolean;

    Function   GetValueAsBoolean: Boolean;
    Function   GetValueAsDouble: Double;
    Function   GetValueAsInteger: Integer;
    Function   GetValueAsString: String;
    Procedure  SetValueAsBoolean(Const Value: Boolean);
    Procedure  SetValueAsDouble(Const Value: Double);
    Procedure  SetValueAsInteger(Const Value: Integer);
    Procedure  SetValueAsString(Const Value: String);
    Function   GetUseCDATA: Boolean;
    Procedure  SetUseCDATA(Const Value: Boolean);

  Protected
    Function   CheckCDATANeed: Boolean; Virtual;

  Public
    Constructor Create(Const AValue : String; Const AUseCDATA : Boolean = False); Overload; Virtual;
    Constructor Create; Overload; Virtual;
    Destructor  Destroy; Override;

    Procedure  Assign(AValue : TXMLValue); Virtual;

    Function   IsNull : Boolean; Virtual;
    Function   AsIntegerDefault(Const ADefault : Integer): Integer; Virtual;

  Published
      // This property can be used to set or retrieve the value as a integer
      // type. If the value can not be converted to a integer it returns 0.
      //
      // See also
      //   AsIntegerDefault
    Property   AsInteger : Integer   Read  GetValueAsInteger
                                     Write SetValueAsInteger;
      // This property can be used to set or retrieve the value as a string
      // type.
    Property   AsString  : String    Read  GetValueAsString
                                     Write SetValueAsString;
      // This property can be used to set or retrieve the value as a boolean
      // type. The value in the xml file can be a string as "true" or "false"
      // or a number. Any number not equal to zero will be interpreted as true
      // and any string not equal to "true" as false.
    Property   AsBoolean : Boolean   Read  GetValueAsBoolean
                                     Write SetValueAsBoolean;
      // This property can be used to set or retrieve the value as a double
      // type. If the value can not be converted, the function returns 0.0. Any
      // decimal seperators (. or ,) can be used.
    Property   AsDouble  : Double    Read  GetValueAsDouble
                                     Write SetValueAsDouble;
      // This property can be used to set or retrieve the flag controlling the
      // usage of the CDATA construct around the value.
    Property   UseCDATA  : Boolean   Read  GetUseCDATA
                                     Write SetUseCDATA;
  End;

  TXMLAbstractAttribute = Class

  End;

    // This class represents one attribute of a node. The attribute always
    // belongs to a attribute list.
  TXMLAttribute = Class(TXMLAbstractAttribute)
  Protected
    fAttrList  : TXMLAttributeList;
    fValue     : TXMLValue;
    fName      : String;

    Procedure   SplitData(Const AData: String); Virtual;
    Function    CheckName(Const AName: String): Boolean; Virtual;     

  Private
    Procedure   SetName(Const Value: String);
    Procedure   SetValue(Const Value: String);
    Function    GetValue: String;

  Public
    Constructor Create(AttrList : TXMLAttributeList; Const Data : String = ''); Overload; Virtual;
    Constructor Create(AttrList : TXMLAttributeList; Const AName, AValue : String); Overload; Virtual;
    Destructor  Destroy; Override;

  Published
      // This property represents the name of the attribute. The name can be
      // changed through assigning a new name to this property.
      //
      // Exceptions
      //   If the newly assigned name is already existing in the attribute
      //   list, it will raise a EXMLNodeAttrNameExists exception.
    Property   Name      : String    Read  fName
                                     Write SetName;

      // This property represents the value of the attribute.
    Property   Value     : TXMLValue Read  fValue;
  End;

    // TXMLAttributeList is the base class for managing TXMLAttributes for a
    // node.
  TXMLAttributeList = Class
  Private
    Function   GetAttributeByIndex(Index: Integer): TXMLAttribute;
    Function   GetAttributeByName(Index: String): TXMLAttribute;
    Function   GetCount: Integer;

  Protected
    fNode     : TXMLNode;
    fAttrList : TObjectList;

    Procedure  SetNode(const Value: TXMLNode); Virtual;

  Public
    Constructor Create(ANode : TXMLNode = Nil); Overload; Virtual;
    Constructor Create(ANode : TXMLNode; Const AttrString : String; AStringAdapter : TStringAdapter); Overload; Virtual;
    Destructor  Destroy; Override;

    Function   Exists(Const AName : String): Boolean; Overload; VirtuaL;
    Function   Exists(Const AAttr : TXMLAttribute): Boolean; Overload; VirtuaL;
    Function   IndexOf(Const AName : String): Integer; VirtuaL;
    Procedure  Delete(Const AName : String); Overload; VirtuaL;
    Procedure  Delete(Const AIndex : Integer); Overload; VirtuaL;
    Function   Add : TXMLAttribute; Overload; VirtuaL;
    Function   Add(XMLAttribute : TXMLAttribute) : TXMLAttribute; Overload; VirtuaL;
    Procedure  Clear; Virtual;

    Function   AsStrings : TStrings; Virtual;
    Function   AsValueList : TStrings; Virtual;

    Function   GetAttributesAsString(AStringAdapter : TStringAdapter) : String; Virtual;
    Procedure  SetAttributesAsString(Const AttributesString : WideString; AStringAdapter : TStringAdapter); Virtual;

      // This is the default array property to access all attribute in the list
      // by a index. The index has to be in range between 0 and Count - 1, any
      // other index will raise a exception.
      //
      // Parameters
      //   Index   - The index of the attribute to retrieve.
      //
      // Returns
      //   Returns the attribute instance or nil and a raised exception if
      //   the index is out of range.
      //
      // Example
      //   This property can be used to iterate through all attributes.
      //   <code>
      //    Procedure ShowAllAttributes;
      //    Var
      //      i : Integer;
      //    Begin
      //      For i := 0 To Pred(AttributeList.Count) Do
      //        ShowMessage(AttributeList.Attributes[i].Name);
      //    End;
      //   </code>
      //   Through the default settings to this property you can access the
      //   elements also with AttributeList[i] instead of
      //   AttributeList.Attributes[i]
    Property   Attributes[Index : Integer]: TXMLAttribute      Read GetAttributeByIndex;
                                                               Default;

      // With the AttributeByName[] property you can access attributes by
      // their name instead of the index (like the Attributes[] property).
      //
      // Parameters
      //   Index  - The name of the of the attribute to retrieve.
      //
      // Returns
      //   The attribute instance or nil if the attribute is not in the list.
      //
      // Example
      //  <code>
      //    Value := AttributeList.AttributeByName['attrname'].Value.AsInteger;
      //  </code>
    Property   AttributeByName[Index : String]: TXMLAttribute  Read GetAttributeByName;

  Published
      // The property Count returns the amount of entries currently in the list.
    Property   Count  : Integer     Read  GetCount;

      // The property Node represents the node where the attributes belong to.
    Property   Node   : TXMLNode    Read  fNode
                                    Write SetNode;
  End;

    // This is the main class for a XML node. It holds references to the
    // attributes, comments and the child nodes.
  TXMLNode = Class
  Private
    Procedure  SetName(const Value: String);
    Function   GetEncodingID: String;
    Procedure  SetEncodingID(const Value: String);
    Function   GetAttributes: TXMLAttributeList;
    Procedure  Set_WriteHeader(const Value: Boolean);

  Protected
    fPath           : String;
    fName           : String;
    fValue          : TXMLValue;
    fParent         : TXMLNode;
    fLevel          : Integer;
    fAttributes     : TXMLAttributeList;
    fStringAdapter  : TStringAdapter;
    fNodes          : TXMLNodeList;
    fWriteHeader    : Boolean;

    Function   CheckName(Const ANewName: String): Boolean; Virtual;

    Function   GetChar(AStream : TStream; Const AAllowSpecialChars: Boolean = False; Const AStopOnSpace : Boolean = False) : Char; Virtual;
    Procedure  SetNameAndSplitAttributes(Const ANameAttr : String); Virtual;
    Procedure  UpdatePath; Virtual;
    Function   GetHasChildren: Boolean; Virtual;

    Function   GetTag(AStream : TStream; Const AddComment : Boolean = True) : String; Virtual;
    Function   GetData(AStream : TStream) : String; Virtual;

    Function   ExtractQuotedString(Const AString : String): String; Virtual;

    Procedure  UpdateChildsEncoding(Const AEncodingID : String);

  Public
      // All the comments are accessible with the Comments[] property/instance. All
      // comments could be iterated with this class.
      //
      // See also
      //   TXMLCommentList
    Comments : TXMLCommentList;

      // creates a new node with the passed TStringAdapter
    Constructor Create(AParent : TXMLNode; AStringAdapter : TStringAdapter); Overload; Virtual;
      // creates a new node which uses the TStringAdapter of the parent node
    Constructor Create(AParent : TXMLNode); Overload; Virtual;
      // copy constructor
    Constructor Create(ANewParent : TXMLNode; AOriginalNode : TXMLNode); Overload; Virtual;
    Destructor  Destroy; Override;

    Function   Read(AStream : TStream): Boolean; Virtual;
    Procedure  Write(AStream : TStream); Virtual;

    Procedure  ClearChildren; Virtual;

      // the caller has to care about freeing the returned list...
    Function   GetChilds(Const AFilter : String; Const AFilterMode : TFilterMode = fmCaseInsensitive): TXMLNodeList; Virtual;
    Function   AddChilds(AChilds : TXMLNodeList): Boolean;

    Procedure  SetAttributeList(AttributeList : TXMLAttributeList; Const AMode : TAttrSetMode); Virtual;

    Function   GetNode(Const ANodeName : String; Const AMode : TXMLNodeHandling = nhReturnNil; Const ASubIndex : Integer = -1): TXMLNode; Virtual;

  Published
    Property   Name            : String              Read  fName
                                                     Write SetName;
    Property   Path            : String              Read  fPath;
    Property   Value           : TXMLValue           Read  fValue;
    Property   Level           : Integer             Read  fLevel;
    Property   Parent          : TXMLNode            Read  fParent;
    Property   HasChildren     : Boolean             Read  GetHasChildren;
    Property   Attributes      : TXMLAttributeList   Read  GetAttributes;
    Property   Nodes           : TXMLNodeList        Read  fNodes;
    Property   EncodingID      : String              Read  GetEncodingID
                                                     Write SetEncodingID;
    Property   WriteHeader     : Boolean             Read  fWriteHeader
                                                     Write Set_WriteHeader;
  End;

  TXMLLib = Class
  Private
    fRoot          : TXMLNode;
    fWriteHeader   : Boolean;

    Procedure Read(AStream : TStream);
    Procedure Write(AStream : TStream);

    Function  CutNext(Var AString : String; Const Divi : Char): String;

    Procedure SetRoot(const Value: TXMLNode);
    function  GetEncoding: String;
    procedure SetEncoding(const Value: String);
    procedure Set_WriteHeader(const Value: Boolean);

  Public
    Constructor Create;
    Destructor  Destroy; Override;

    Function  LoadFromFile(Const AFileName : String): Boolean;
    Procedure LoadFromStream(AStream : TStream);
    Function  SaveToFile(Const AFileName : String): Boolean;
    Procedure SaveToStream(AStream : TStream);

    Function  CreatePathAndNode(Const APath : String; Out ADestNode : TXMLNode; Const ASubIndex : Integer = -1): Boolean;
    Function  GetNodeFromPath(Const APath : String; Const ASubIndex : Integer = -1): TXMLNode;

    Procedure Clear;

    Class Procedure RetrieveSupportedEncodings(AStrings : TStrings);

    Class Function  GetUnitID: String;

    Property  Root : TXMLNode              Read  fRoot
                                           Write SetRoot;
    Property  EncodingID : String          Read  GetEncoding
                                           Write SetEncoding;
    Property  WriteHeader : Boolean        Read  fWriteHeader
                                           Write Set_WriteHeader;
  End;

Implementation

{ TXMLLib }

Uses
  Masks,
  Dialogs;

Const
    // RCS Library version identification
  XMLLIBID : String = '$Revision: 1.29 $ $Date: 2005-11-16 23:46:50+01 $';

Var
    // count of white spaces for indent a node level
  Indent : Integer = 2;

  gLineNumber : Integer = 1;

  // helper functions
Function  StripSubIndexFromName(Const ANodeName : String): String;
Var
  lPosIdx1,
  lPosIdx2 : Integer;
Begin
  Result := Trim(ANodeName);

  lPosIdx1 := Pos('[', Result);
  lPosIdx2 := Pos(']', Result);

  If ( lPosIdx1 > 0 ) And ( lPosIdx1 < lPosIdx2 ) And
     ( lPosIdx2 = Length(Result) ) Then
    Delete(Result, lPosIdx1, MaxInt);
End;

Function  GetSubIndexFromName(Const ANodeName : String; Var ASubIndex: Integer): Boolean;
Var
  lPosIdx1,
  lPosIdx2 : Integer;
Begin
  Result := False;

  lPosIdx1 := Pos('[', ANodeName);
  lPosIdx2 := Pos(']', ANodeName);

  If ( lPosIdx1 > 0 ) And ( lPosIdx1 < lPosIdx2 ) And
     ( lPosIdx2 = Length(ANodeName) ) Then
  Begin
    Result    := True;
    ASubIndex := StrToIntDef(Copy(ANodeName, lPosIdx1+1, lPosIdx2-lPosIdx1-1), -1);
  End;
End;

Constructor TXMLLib.Create;
Begin
  Inherited Create;

  //fStringAdapter := TStringAdapter.Create('', True);
End;

Destructor TXMLLib.Destroy;
Begin
  fRoot.Free;

  Inherited;
End;

Procedure TXMLLib.Clear;
Begin
  FreeAndNil(fRoot);
End;

Function TXMLLib.CreatePathAndNode(Const APath : String; Out ADestNode : TXMLNode;
                                   Const ASubIndex : Integer): Boolean;
Var
  Pth, s : String;
//  Index  : Integer;
  Node   : TXMLNode;
Begin
  Result := False;
  ADestNode := Nil;

  Try
    Pth := APath;
    s   := CutNext(Pth, '.');

    If Not Assigned(fRoot) Then
    Begin
      fRoot := TXMLNode.Create(Nil{, fStringAdapter});
      fRoot.Name := s;
    End
    Else
    Begin
{$IFDEF NAMESCASEINSENSITIVE}
      If ( Not AnsiSameText(fRoot.Name, s) ) Then
{$ELSE}
      If ( fRoot.Name <> s ) Then
{$ENDIF}
        Raise EXMLInvalidNodePath.Create('More than one root node is not allowed');
    End;

    Node := fRoot;
    If ( Trim(Pth) <> '' ) Then
    Begin
{$IFDEF NAMESCASEINSENSITIVE}
      If AnsiSameText(Node.Name, s) Then
{$ELSE}
      If ( Node.Name = s ) Then
{$ENDIF}
      Begin
        If ( Length(Pth) > 0 ) Then
          Node := Node.GetNode(Pth, nhCreateNode, ASubIndex);

        ADestNode := Node;
        Result := True;
      End;
    End
    Else
    Begin
{$IFDEF NAMESCASEINSENSITIVE}
      If ( AnsiSameText(fRoot.Name, s) ) Then
{$ELSE}
      If ( fRoot.Name = s ) Then
{$ENDIF}
      Begin
        ADestNode := fRoot;
        Result := True; // ok, Node created
      End
      Else
        Raise EXMLInvalidNodePath.Create('More than one root node is not allowed');
    End;
  Except
    On EXMLLibException Do ;
  Else
    Raise;
  End;
End;

Function TXMLLib.CutNext(Var AString: String; Const Divi: Char): String;
Begin
  If ( Pos(Divi, AString) > 0 ) Then
  Begin
    Result := Copy(AString, 1, Pos(Divi, AString)-1);
    Delete(AString, 1, Pos(Divi, AString));
  End
  Else
  Begin
    Result := AString;
    AString := '';
  End;
End;

Function TXMLLib.GetEncoding: String;
Begin
  If Assigned(fRoot) And Assigned(fRoot.fStringAdapter) Then
    Result := fRoot.fStringAdapter.EncodingID
  Else
    Result := 'UTF-8';
End;

Function TXMLLib.GetNodeFromPath(const APath: String; Const ASubIndex : Integer): TXMLNode;
Var
  Pth, s : String;
Begin
  Pth := APath;
  Result := fRoot;
  If Assigned(Result) Then
  Begin
    s := CutNext(Pth, '.');
{$IFDEF NAMESCASEINSENSITIVE}
    If AnsiSameText(Result.Name, s) Then
{$ELSE}
    If ( Result.Name = s ) Then
{$ENDIF}
    Begin
      If ( Length(Pth) > 0 ) Then
        Result := Result.GetNode(Pth);
    End
    Else
      Result := Nil;
  End;
End;

Class Function TXMLLib.GetUnitID: String;
Begin
  Result := Trim(StringReplace(XMLLIBID, '$', '', [rfReplaceAll, rfIgnoreCase]));
End;

Function TXMLLib.LoadFromFile(const AFileName: String): Boolean;
Var
  fStream : TStream;
Begin
  Result := False;

  If ( FileExists(AFileName) ) Then
  Begin
    fStream := TFileStream.Create(AFileName, fmOpenRead Or fmShareDenyWrite);
    Try
      Read(fStream);
    Finally
      fStream.Free;
    End;
    Result := True;
  End;
End;

Procedure TXMLLib.LoadFromStream(AStream: TStream);
Begin
  Read(AStream);
End;

Procedure TXMLLib.Read(AStream: TStream);
Begin
  If ( Assigned(fRoot) ) Then
    fRoot.Free;

  gLineNumber := 1;

  fRoot := TXMLNode.Create(Nil{, fStringAdapter});
  fRoot.Read(AStream);
End;

Class Procedure TXMLLib.RetrieveSupportedEncodings(AStrings: TStrings);
Var
  l_Encodings : TStrings;
Begin
  If Assigned(AStrings) Then
  Begin
    l_Encodings := TStringAdapter.GetRegisteredAdapters;
    Try
      AStrings.Clear;
      AStrings.AddStrings(l_Encodings);
    Finally
      l_Encodings.Free;
    End;
  End;
End;

Function TXMLLib.SaveToFile(const AFileName: String): Boolean;
Var
  lStream : TStream;
Begin
  Result  := False;
  lStream := Nil;

  Try
    lStream := TFileStream.Create(AFileName, fmCreate Or fmShareExclusive);
  Except
    ;
  End;

  If Assigned(lStream) Then
  Begin
    Try
      Try
        Write(lStream);
        Result := True;
      Except
        ;
      End;
    Finally
      lStream.Free;
    End;
  End;
End;

Procedure TXMLLib.SaveToStream(AStream: TStream);
Begin
  Write(AStream);
End;

Procedure TXMLLib.SetEncoding(Const Value: String);
Begin
  If Assigned(fRoot) Then
    fRoot.EncodingID := Value;
End;

Procedure TXMLLib.SetRoot(const Value: TXMLNode);
Begin
  If ( fRoot <> Value ) Then
  Begin
    FreeAndNil(fRoot);
    fRoot := Value;
    If Assigned(fRoot) Then
      fRoot.WriteHeader := fWriteHeader;
  End;
End;

Procedure TXMLLib.Set_WriteHeader(const Value: Boolean);
Begin
  fWriteHeader := Value;
  
  If Assigned(fRoot) Then
    fRoot.WriteHeader := fWriteHeader;
End;

Procedure TXMLLib.Write(AStream: TStream);
Begin
  If Assigned(fRoot) Then
    fRoot.Write(AStream);
End;

{ TXMLNodeList }

  // This is the default constructor of the TXMLNodeList class.
  //
  // Parameters
  //   ACanDestroy  - This parameter defines the destruction of the contained
  //                  nodes in the xml node list on destruction of the node list
  //                  class destruction. With a value of true, the nodes are
  //                  maintained by the node list instance and will be destroied
  //                  on destruction of the list class. If the parameter will
  //                  be false, the nodes won't be freed on destruction of the
  //                  list and the user has to care about destruction of the
  //                  nodes.
  //                  The parameter is voluntary and has a default value of true.
Constructor TXMLNodeList.Create(Const ACanDestroy : Boolean);
Begin
  Inherited Create;

  fNodes := TObjectList.Create(ACanDestroy);
End;

  // This is the copy constructor of the TXMLNodeList class.
  //
  // Parameters
  //   ANewParent   - The new parent (owner) of the copied nodes of the
  //                  original node list.
  //   ANodeList    - The original node list to copy.
  //   ACanDestroy  - This parameter defines the destruction of the contained
  //                  nodes in the new node list on destruction of the node list
  //                  class destruction. With a value of true, the nodes are
  //                  maintained by the node list instance and will be destroied
  //                  on destruction of the list class. If the parameter will
  //                  be false, the nodes won't be freed on destruction of the
  //                  list and the user has to care about destruction of the
  //                  nodes.
Constructor TXMLNodeList.Create(ANewParent : TXMLNode; ANodeList: TXMLNodeList; Const ACanDestroy: Boolean);
Var
  i : Integer;
Begin
  Inherited Create;

  fNodes := TObjectList.Create(ACanDestroy);
  For i := 0 To ANodeList.Count -1 Do
    fNodes.Add( TXMLNode.Create(ANewParent, ANodeList.Nodes[i]) );
End;

  // Adds a new node to the xml node list.
  //
  // Parameters
  //   ANode   - The node instance to add.
  //
  // Returns
  //   Add returns the index where the node was added in the node list.
  //
  // Remarks
  //   The passed node instance is now maintained by the node list and should
  //   not be freed from outside, when the node list was created with the
  //   ACanDestroy parameter in the Create function!
Function TXMLNodeList.Add(ANode: TXMLNode): Integer;
Begin
  Result := fNodes.Add(ANode);
End;

  // Deletes the node at the position AIndex in the xml node list.
  //
  // Parameters
  //   AIndex   - The index of the node to delete.
  //
  // Remarks
  //   The deleted node will be automatically be freed, if the xml node list
  //   owns the nodes (see constructor)
Procedure TXMLNodeList.Delete(const AIndex: Integer);
Begin
  fNodes.Delete(AIndex);
End;

Procedure TXMLNodeList.Delete(Var ANode: TXMLNode);
Var
  lIndex : Integer;
Begin
  lIndex := IndexOf(ANode);

  If ( lIndex > -1 ) Then
  Begin
    fNodes.Delete(lIndex);
    If ( Not fNodes.OwnsObjects ) Then
      ANode.Free;

    ANode := Nil;
  End;
End;

  // The destructor for freeing the node list.
Destructor TXMLNodeList.Destroy;
Begin
  fNodes.Free;

  Inherited;
End;

Function TXMLNodeList.GetCount: Integer;
Begin
  Result := fNodes.Count;
End;

  // Use IndexOf() to retrieve the index of a definied node in the node list.
  //
  // Parameters
  //   ANode  - The node to look up.
  //
  // Returns
  //   The index of the node in the list or -1 if the node could not be found.
Function TXMLNodeList.IndexOf(ANode: TXMLNode): Integer;
Begin
  Result := fNodes.IndexOf(ANode);
End;

Function TXMLNodeList.GetXMLNodeString(Index: String): TXMLNode;
Var
  Idx : Integer;
Begin
  Result := Nil;
  Idx := IndexOf(Index);
  If ( Idx > -1 ) Then
    Result := fNodes[Idx] As TXMLNode;
End;

  // Use IndexOf() to retrieve a node by name in the node list.
  //
  // Parameters
  //   ANodeName   - The name of the node to retrieve.
  //   ASubIndex   - This parameter is voluntary, the default value is 0.
  //                 Use this parameter to define which node you want to
  //                 retrieve when more than one node are exisiting with the
  //                 same node name. The default value will cause the function
  //                 to retrieve the index of the first node with the passed
  //                 name.
  //
  // Returns
  //   The index of the node in the node list or -1.
Function TXMLNodeList.IndexOf(Const ANodeName: String; Const ASubIndex : Integer): Integer;
Var
  i,
  lSubIdxCtr : Integer;
  lNodeName  : String;
Begin
  Result := -1;

  lNodeName := Trim(ANodeName);

  If ( ASubIndex < 0 ) Then
  Begin
    lSubIdxCtr := -1;
    If GetSubIndexFromName(lNodeName, lSubIdxCtr) Then
      lNodeName := StripSubIndexFromName(lNodeName);
  End
  Else
    lSubIdxCtr := ASubIndex;

  For i := 0 To fNodes.Count - 1 Do
{$IFDEF NAMESCASEINSENSITIVE}
    If AnsiSameText(GetXMLNode(i).Name, lNodeName) Then
{$ELSE}
    If ( GetXMLNode(i).Name = lNodeName ) Then
{$ENDIF}
    Begin
      If ( lSubIdxCtr <= 0 ) Then
      Begin
        Result := i;
        Break;
      End;

      Dec(lSubIdxCtr);
    End;
End;

Function TXMLNodeList.GetXMLNode(Index: Integer): TXMLNode;
Begin
  Result := fNodes[Index] As TXMLNode;
End;

  // This protected function searches for the node named ANodeName in the comment
  // list. If more than one node exist in the list with the same name, the other
  // nodes can be accessed via the ASubIndex parameter.
  //
  // Parameters
  //   ANodeName    - The name of the node to search for.
  //   ASubIndex    - The number of the node to return if more than one node
  //                  exists under the same name. This parameter is voluntarily,
  //                  a sub index of 0 will be used as default (return the first
  //                  node of this name).
  //
  // Returns
  //   If the node can not be found, the function returns nil, otherwise
  //   the found TXMLNode instance.
Function TXMLNodeList.GetXMLNodeStringSubIndex(ANodeName: String; Const ASubIndex : Integer = 0): TXMLNode;
Var
  Idx : Integer;
Begin
  Result := Nil;
  Idx := IndexOf(ANodeName, ASubIndex);
  If ( Idx > -1 ) Then
    Result := fNodes[Idx] As TXMLNode;
End;

  // Deletes all nodes contained in the list. If the node list is the maintainer
  // of the nodes, the instances will be freed automatically.
Procedure TXMLNodeList.Clear;
Begin
  fNodes.Clear;
    // TObjectList cares about freeing if selected...
End;

  // Use SubNodeCount() to retrieve the amount of nodes in the list with
  // equal names. This is useful when iterating nodes while using the
  // SubIndex parameter of functions like IndexOf().
  //
  // Parameters
  //   ANodeName   - The name of the node(s)
  //
  // Returns
  //   The count of nodes with the given name or 0 otherwise. 
Function TXMLNodeList.SubNodeCount(const ANodeName: String): Integer;
Var
  i : Integer;
Begin
  Result := 0;

  For i := 0 To fNodes.Count - 1 Do
{$IFDEF NAMESCASEINSENSITIVE}
    If ( AnsiSameText(Trim(ANodeName), Trim(GetXMLNode(i).Name)) ) Then
{$ELSE}
    If ( Trim(ANodeName) = Trim(GetXMLNode(i).Name) ) Then
{$ENDIF}
      Inc(Result);
End;

  // Retrieves a node with the specified name and sub node index. This function
  // is useful to retrieve a specific node from the nodes with the same name.
  //
  // Parameters
  //   ANodeName   - The name of the node to retrieve
  //   ASubIndex   - The number of the node to return if more than one node
  //                 exists under the same name. This parameter is voluntarily,
  //                 a sub index of 0 will be used as default (return the first
  //                 node of this name).
  //
  // Returns
  //   The node instance or nil otherwise.
Function TXMLNodeList.GetNode(const ANodeName: String; Const ASubIndex: Integer): TXMLNode;
Var
  lIndex : Integer;
Begin
  lIndex := IndexOf(ANodeName, ASubIndex);

  If ( lIndex > -1 ) Then
    Result := GetXMLNode(lIndex)
  Else
    Result := Nil;
End;

  // Use GetFilteredList to retrieve a list of nodes that names matches a
  // specified mask.
  //
  // Parameters
  //   AFilter    - Defines the name filter mask. The resulting list only
  //                contains the nodes matching the mask. The mask itself is
  //                based on place holders of the filesystems\: \* and \?. Use
  //                the question mark to as a placeholder for any kind of
  //                character at this point and the star as a place holder for
  //                any kind of combination of characters in any count. Both
  //                place holders can be combined in the mask with other
  //                defined characters.
  //   AMode      - This parameter constrols the comparsion of the filter
  //                with the nodes names.
  //                Without defining this parameter it will be incase
  //                sensitive by default.
  //
  // Example
  //   Retrieve all nodes from the list:
  //    <CODE>
  //     Procedure GetAll;
  //     Var
  //       MyNodeList : TXMLNodeList;
  //     Begin
  //       MyNodeList := NodesList.GetFilteredList('*');
  //     End;
  //    </CODE>
  //
  //   Retrieve all nodes beginning with a small "a" from the list:
  //    <CODE>
  //     Procedure GetAllBeginningWithA;
  //     Var
  //       MyNodeList : TXMLNodeList;
  //     Begin
  //       MyNodeList := NodesList.GetFilteredList('a*', fmCaseSensitive);
  //     End;
  //    </CODE>
  //
  // Remarks
  //   The function always returns a instanced list, even if no entry matches
  //   the mask.
  //
  //   The caller has to care about freeing the returned list.
Function TXMLNodeList.GetFilteredList(const AFilter: String; Const AMode : TFilterMode = fmCaseInsensitive): TXMLNodeList;
Var
  i : Integer;
Begin
  Result := TXMLNodeList.Create(False);

  If ( AMode = fmCaseInsensitive ) Then
  Begin
    For i := 0 To Pred(fNodes.Count) Do
      If ( MatchesMask(GetXMLNode(i).Name, AFilter) ) Then
        Result.Add(GetXMLNode(i));
  End
  Else
  Begin
    For i := 0 To Pred(fNodes.Count) Do
      If ( MatchesMask(AnsiLowerCase(GetXMLNode(i).Name), AnsiLowerCase(AFilter)) ) Then
        Result.Add(GetXMLNode(i));
  End;
End;

  // Use GetNodeNames() to retrieve a list of all node names currently in the
  // node list.
  //
  // Remarks
  //   The function always returns a instanced list, even if the list contains
  //   no entry.
  //
  //   The caller has to care about freeing the returned list.
Function TXMLNodeList.GetNodeNames: TStrings;
Var
  i : Integer;
Begin
  Result := TStringList.Create;

  For i := 0 To fNodes.Count - 1 Do
    Result.Add(GetXMLNode(i).Name);
End;

  // The Exists() method could be used to test the existence of a node by his
  // name in the node list.
  //
  // Parameters
  //   ANodeName  - The name of the node to find
  //
  // Returns
  //   Returns True if the node could be found or false otherwise.
Function TXMLNodeList.Exists(const ANodeName: String): Boolean;
Begin
  Result := ( IndexOf(ANodeName) <> -1 );
End;

{ TXMLNode }

Constructor TXMLNode.Create(AParent : TXMLNode; AStringAdapter : TStringAdapter);
Begin
  Inherited Create;

  fParent     := AParent;
  fName       := '';
  fValue      := TXMLValue.Create;

{$IFNDEF ALLOWATTRIBUTEONROOT}
  If Assigned(AParent) Then
    fAttributes := TXMLAttributeList.Create(Self)
  Else
    fAttributes := Nil;
{$ELSE}
  fAttributes := TXMLAttributeList.Create(Self);
{$ENDIF}

  fStringAdapter := AStringAdapter;
  If ( Not Assigned(fStringAdapter) ) And ( Assigned(AParent) ) Then
    fStringAdapter := AParent.fStringAdapter;

  If Not Assigned(fStringAdapter) Then
    fStringAdapter := TStringAdapter.Create('UTF-8');

  If ( Assigned(fParent) ) Then
  Begin
    fLevel := fParent.Level + 1;
    fPath := fParent.Path;
    If ( fPath <> '' ) Then
      fPath := fPath + '.' + fParent.Name
    Else
      fPath := fParent.Name;

    fParent.Nodes.Add(Self);
  End
  Else
  Begin
    fPath  := '';
    fLevel := 0;
  End;

  fNodes   := TXMLNodeList.Create;
  Comments := TXMLCommentList.Create;
End;

Constructor TXMLNode.Create(AParent: TXMLNode);
Begin
  Inherited Create;

  fParent     := AParent;
  fName       := '';
  fValue      := TXMLValue.Create;

{$IFNDEF ALLOWATTRIBUTEONROOT}
  If Assigned(AParent) Then
    fAttributes := TXMLAttributeList.Create(Self)
  Else
    fAttributes := Nil;
{$ELSE}
  fAttributes := TXMLAttributeList.Create(Self);
{$ENDIF}

  If Assigned(AParent) Then
    fStringAdapter := AParent.fStringAdapter
  Else
  Begin
    If TStringAdapter.IsCodingAvailable('UTF-8') Then
      fStringAdapter := TStringAdapter.Create('UTF-8')
    Else
      fStringAdapter := TStringAdapter.Create('');
  End;

  If ( Assigned(fParent) ) Then
  Begin
    fLevel := fParent.Level + 1;
    fPath := fParent.Path;
    If ( fPath <> '' ) Then
      fPath := fPath + '.' + fParent.Name
    Else
      fPath := fParent.Name;

    fParent.Nodes.Add(Self);
  End
  Else
  Begin
    fPath  := '';
    fLevel := 0;
  End;

  fWriteHeader := True;
  fNodes   := TXMLNodeList.Create;
  Comments := TXMLCommentList.Create;
End;

Constructor TXMLNode.Create(ANewParent, AOriginalNode: TXMLNode);
Begin
  Inherited Create;

  fParent        := ANewParent;
  fName          := AOriginalNode.Name;
  fStringAdapter := AOriginalNode.fStringAdapter;
  fValue         := TXMLValue.Create;
  fValue.Assign(AOriginalNode.fValue);

  If
{$IFNDEF ALLOWATTRIBUTEONROOT}
     ( Assigned(fParent) ) And
{$ENDIF}
     ( Assigned(AOriginalNode.Attributes) ) Then
    fAttributes := TXMLAttributeList.Create(Self, AOriginalNode.Attributes.GetAttributesAsString(fStringAdapter), fStringAdapter);
    
  If Assigned(fParent) Then
  Begin
    fLevel      := fParent.Level + 1;
    fPath       := fParent.Path;
    If ( fPath <> '' ) Then
      fPath := fPath + '.' + fParent.Name
    Else
      fPath := fParent.Name;

    //fParent.Nodes.Add(Self); --- quatsch, a) existiert Nodes noch nicht - wir sind hier
    //                                         gerade beim bauen des selbigen
    //                                      b) machen wir das doch selber in der Schleife...
  End
  Else
  Begin
    fPath  := '';
    fLevel := 0;
  End;

  fNodes   := TXMLNodeList.Create(Self, AOriginalNode.Nodes);
  Comments := TXMLCommentList.Create(AOriginalNode.Comments);

  UpdatePath;
End;

Destructor TXMLNode.Destroy;
Begin
  fValue.Free;
  fNodes.Free;
  Comments.Free;
  fAttributes.Free;

  If Not Assigned(fParent) Then
    fStringAdapter.Free;
  //fStringAdapter := Nil;

  Inherited;
End;

Function  TXMLNode.Read(AStream: TStream): Boolean;
Var
  Tag   : String;
  Data  : String;
  Node  : TXMLNode;
  idx,
  fPos  : Integer;
  TmpName : String;
begin
  Result := False;

  Repeat
    Tag := Trim(GetTag(AStream));
    If ( Length(Tag) = 0 ) Then
      Exit;

    If ( ( Copy(Tag, 2, 1) = '!' ) Or ( Copy(Tag, 2, 1) = '?' ) ) Then
    Begin
      If AnsiSameText(Copy(Tag, 1, 6), '<?xml ') Then
      Begin
        idx := Pos('VERSION=', AnsiUpperCase(Tag));
        If ( idx > 0 ) Then
        Begin
          Data := Copy(Tag, idx+9, MaxInt);
          System.Delete(Data, Pos('"', Data), MaxInt);
        End;

        idx := Pos('ENCODING=', AnsiUpperCase(Tag));
        If ( idx > 0 ) Then
        Begin
          Data := Copy(Tag, idx+9, MaxInt);
          Data := ExtractQuotedString(Data);

          If Assigned(fStringAdapter) Then
          Begin
            fStringAdapter.EncodingID := Data;

            UpdateChildsEncoding(Data);
          End;
        End;
      End;
    End;
  Until ( ( Copy(Tag, 2, 1) <> '!' ) And ( Copy(Tag, 2, 1) <> '?' ) );

  TmpName := Trim(Copy(Tag, 2, Length(Tag)-2));

  If ( Copy(TmpName, Length(TmpName), 1) = '/' ) Then
    SetNameAndSplitAttributes(Copy(TmpName, 1, Length(TmpName)-1))
  Else
  Begin
    SetNameAndSplitAttributes(TmpName);

    Data  := Trim(GetData(AStream));

    If ( Length(Data) > 0 ) Then
    Begin
      If ( Assigned(fValue) ) Then
        fValue.AsString := fStringAdapter.StringDecode(Data)
      Else
        fValue := TXMLValue.Create(fStringAdapter.StringDecode(Data));
    End;

    Repeat
      fPos := AStream.Position;
      Tag  := Trim(GetTag(AStream, False));
{$IFDEF NAMESCASEINSENSITIVE}
      If ( Not AnsiSameText(Tag, '</' + fName + '>') ) Then
{$ELSE}
      If ( Tag <> '</' + fName + '>' ) Then
{$ENDIF}
      Begin
        AStream.Position := fPos;
        Node := TXMLNode.Create(Self, fStringAdapter);
        If Not Node.Read(AStream) Then
          Break;
      End
      Else
        Break;
    Until ( AStream.Position = AStream.Size );
  End;
  Result := True;
End;

Procedure TXMLNode.Write(AStream: TStream);
Var
  Buffer : String;
  i      : Integer;
Begin
  If ( ( fParent = Nil ) And ( fLevel = 0 ) And ( fWriteHeader ) ) Then // root node??
  Begin
    Buffer := '<?xml version="1.0"';
    If ( fStringAdapter.EncodingID <> '' ) Then
      Buffer := Buffer + ' encoding="' + fStringAdapter.EncodingID + '"';
    Buffer := Buffer + ' ?>' + #13#10;
    AStream.Write(Buffer[1], Length(Buffer));
  End;

  If ( Comments.Count > 0 ) Then
  Begin
    For i := 0 To Comments.Count - 1 Do
    Begin
      Buffer := StringOfChar(' ', Indent * fLevel) +
                '<!-- ' + Copy(Comments[i].Text, 1, Length(Comments[i].Text)-2) + ' -->' + #13#10;
      AStream.Write(Buffer[1], Length(Buffer));
    End;
  End;

  If ( Nodes.Count > 0 ) Then
  Begin
    If ( ( Assigned(fAttributes) ) And ( fAttributes.Count > 0 ) ) Then // no attributes on root nodes!
      Buffer := StringOfChar(' ', Indent * fLevel) +
                '<' + fName + ' ' + fAttributes.GetAttributesAsString(fStringAdapter) + '>' + #13#10
    Else
      Buffer := StringOfChar(' ', Indent * fLevel) +
                '<' + fName + '>' + #13#10;
    AStream.Write(Buffer[1], Length(Buffer));
    For i := 0 To Nodes.Count - 1 Do
      Nodes[i].Write(AStream);
    Buffer := StringOfChar(' ', Indent * fLevel) +
              '</' + fName + '>' + #13#10;
    AStream.Write(Buffer[1], Length(Buffer));
  End
  Else
  Begin
    If ( Not fValue.IsNull ) Then
    Begin
      If ( ( Assigned(fAttributes) ) And ( fAttributes.Count > 0 ) ) Then
      Begin
        Buffer := StringOfChar(' ', Indent * fLevel) +
                '<' + fName + ' ' + fAttributes.GetAttributesAsString(fStringAdapter) + '>';

        If ( fValue.UseCDATA ) Then
          Buffer := Buffer + '<![CDATA[' + fValue.AsString + ']]>'
        Else
          Buffer := Buffer + fStringAdapter.StringEncode(fValue.AsString);

        Buffer := Buffer + '</' + fName + '>' + #13#10;
      End
      Else
      Begin
        If ( fValue.UseCDATA ) Then
          Buffer := StringOfChar(' ', Indent * fLevel) +
                    '<' + fName + '>' +
                    '<![CDATA[' + fValue.AsString + ']]>' +
                    '</' + fName + '>' + #13#10
        Else
          Buffer := StringOfChar(' ', Indent * fLevel) +
                    '<' + fName + '>' +
                    fStringAdapter.StringEncode(fValue.AsString) +
                    '</' + fName + '>' + #13#10;
      End;
    End
    Else
    Begin
      If ( ( Assigned(fAttributes) ) And ( fAttributes.Count > 0 ) ) Then
        Buffer := StringOfChar(' ', Indent * fLevel) +
                  '<' + fName + ' ' + fAttributes.GetAttributesAsString(fStringAdapter) + ' />' + #13#10
      Else
        Buffer := StringOfChar(' ', Indent * fLevel) +
                  '<' + fName + ' />' + #13#10;
    End;

    AStream.Write(Buffer[1], Length(Buffer));
  End;
End;

  // AAllowSpecialChars: allow CR, LF and HT.
Function TXMLNode.GetChar(AStream : TStream; Const AAllowSpecialChars: Boolean = False; Const AStopOnSpace : Boolean = False): Char;
Const
  coSpecialChars = [ 10, 13, 9 ]; // LF, CR, HT
Var
  Ch : Char;
Begin
  Result := #0;
  While ( AStream.Position < AStream.Size ) Do
  Begin
    AStream.Read(Ch, 1);

    If ( Ch = #10 ) Then
      Inc(gLineNumber);

    If ( Ch = ' ' ) And ( AStopOnSpace ) Then
    Begin
      Result := Ch;
      Break;
    End
    Else If ( Ord(Ch) > $20 ) Or ( ( AAllowSpecialChars ) And ( Ord(Ch) In coSpecialChars ) ) Then
    Begin
      Result := Ch;
      Break;
    End;
  End;
End;

Function TXMLNode.ExtractQuotedString(const AString: String): String;
Const
  coQuoteChars : Array[0..1] Of Char = ( '"', '''' );
Var
  i : Integer;
Begin
  Result := AString;  // default

  For i := Low(coQuoteChars) To High(coQuoteChars) Do
  Begin
    If ( Pos(coQuoteChars[i], AString) > 0 ) Then
    Begin
      Result := Copy(AString, Pos(coQuoteChars[i], AString)+1, MaxInt);

      If ( Pos(coQuoteChars[i], Result) > 0 ) Then
      Begin
        Delete(Result, Pos(coQuoteChars[i], Result), MaxInt);
        Break;
      End
      Else
      Begin
        Result := AString;
        Continue; // the quote char has to be twice in the string
      End;
    End;
  End;
End;

Function TXMLNode.GetTag(AStream: TStream; Const AddComment : Boolean): String;
Label
  ReDoIt;
Var
  Ch    : Char;
  Comm  : TStrings;
  CommStr,
  s     : String;
  fpos,
  i     : Integer;
Begin
  Result := '';

ReDoIt:        // I know - this is really the badiest thing I could do, but
               // I'm so frustrated with outcommented data - so please - let
               // me think another time about removing this goto... ok?
  Repeat
    Ch := GetChar(AStream)
  Until ( ( Ch = #0 ) Or ( Ch = '<' ) );

  If ( Ch <> #0 ) Then
  Begin
    SetLength(s, 3);
    fpos := AStream.Position;
    AStream.Read(s[1], 3);
    If ( s = '!--' ) Then // Kommentar
    Begin
      i := 1;
      CommStr := '';
      Repeat
        AStream.Read(Ch, 1);
        s[i] := Ch;
        CommStr := CommStr + s[i];

        Case i Of
          1 : Begin
                If ( s[1] = '-' ) Then
                  Inc(i);
              End;
          2 : Begin
                If ( s[2] = '-' ) Then
                  Inc(i)
                Else
                  i := 1;
              End;
          3 : Begin
                If ( s[3] = '>' ) Then
                  Inc(i)
                Else
                  i := 1;
              End;
        End;
      Until ( i = 4 ) Or ( AStream.Position = Pred(AStream.Size) );

      If ( i = 4 ) And ( AddComment ) Then
      Begin
        Comm := TStringList.Create;
        Comm.Text := Trim(Copy(CommStr, 1, Length(CommStr)-3));
        Comments.Add(Comm);
      End;

      Goto ReDoIt;
    End
    Else
      AStream.Position := fpos;

    Repeat
      Result := Result + Ch;
      Ch := GetChar(AStream, False, True);
    Until ( ( Ch = #0 ) Or ( Ch = '>' ) );
    If ( Ch <> #0 ) Then
      Result := Result + Ch;
  End;
End;

Function TXMLNode.GetData(AStream: TStream): String;
Const
  CData_Begin = '<![CDATA[';
  CData_End   = ']]>';
Var
  Ch          : Char;
  CData_Found : Boolean;
  i, OldPos   : Integer;
  lCompare    : String;
Begin
  Result := '';
  CData_Found := False;

  Ch := GetChar(AStream, True, True);

  If ( Ch = '<' ) Then
  Begin
    OldPos := AStream.Position;

    For i := 1 To 8 Do
      lCompare := lCompare + UpCase(GetChar(AStream, True));

    CData_Found := ( CData_Begin = '<' + lCompare );

    If ( Not CData_Found ) Then
      AStream.Position := OldPos;
  End;

  If ( CData_Found ) Then
  Begin
    Ch := GetChar(AStream, True);

    While ( ( Ch <> #0 ) And
            ( CData_Found ) ) Do
    Begin
      Result := Result + Ch;
      Ch := GetChar(AStream, True, True);

      CData_Found := Not ( Copy(Result, Length(Result)-Length(CData_End)+1, Length(CData_End)) = CData_End );
    End;

    Delete(Result, Length(Result)-2, MaxInt); 
    AStream.Position := AStream.Position - 1;
  End
  Else
  Begin
    While ( ( Ch <> #0 ) And
            ( Ch <> '<' ) ) Do
    Begin
      Result := Result + Ch;
      Ch := GetChar(AStream, True, True);
    End;
  End;

  If ( Ch = '<' ) Then
    AStream.Position := AStream.Position - 1;
End;

Procedure TXMLNode.SetName(const Value: String);
Begin
  If ( fName <> Value ) Then
  Begin
    If CheckName(Value) Then
      fName := Value
    Else
      Raise EXMLNodeNameInvalid.CreateFmt('node name "%s" is invalid! (Line: %d)', [Value, gLineNumber]);

    UpdatePath;
  End;
End;

Function TXMLNode.CheckName(Const ANewName: String): Boolean;
Const
  coValidFirstChars = [':', '_', 'a'..'z', 'A'..'Z'];
  coValidChars      = [':', '_', 'a'..'z', 'A'..'Z', '0'..'9', '-']; // .
Var
  i : Integer;
Begin
{$IFDEF DOVALIDITYCHECKS}
  Result := False;

  If ( Length(Trim(ANewName)) = 0 ) Then
    Exit;

    // 1. no "xml" in any form in the beginning
  If ( AnsiSameText(Copy(ANewName, 1, 3), 'xml') ) Then
    Exit;

    // 2. first character only ":", "_" or any letter
  If ( Not ( ANewName[1] In coValidFirstChars ) ) Then
    Exit;

    // 3. all other characters of the name only coValidChars
  For i := 2 To Length(ANewName) Do
    If ( Not ( ANewName[i] In coValidChars ) ) Then
      Exit;

    // 4. at least one character or underline
  For i := 1 To Length(ANewName) Do
    If ( ANewName[i] In ['_', 'a'..'z', 'A'..'Z'] ) Then
    Begin
      Result := True;
      Break;
    End;
{$ELSE}
  Result := True;
{$ENDIF}
End;

Procedure TXMLNode.UpdatePath;
Var
  i : Integer;
Begin
  If Assigned(fParent) Then
  Begin
    fPath := fParent.Path;
    If ( fPath <> '' ) Then
      fPath := fPath + '.' + fParent.Name
    Else
      fPath := fParent.Name;
  End
  Else
    fPath := '';

  For i := 0 To Nodes.Count - 1 Do
    Nodes[i].UpdatePath;
End;

Procedure TXMLNode.SetNameAndSplitAttributes(const ANameAttr: String);
Var
  Data : String;
Begin
  If ( Pos(' ', ANameAttr) = 0 ) Then
  Begin  // no attributes
    fName := ANameAttr;
  End
  Else
  Begin
    Data := ANameAttr;

    fName := Trim(Copy(Data, 1, Pos(' ', Data)-1));
    Delete(Data, 1, Pos(' ', Data));

    If Assigned(fAttributes) Then
      fAttributes.Free;
    fAttributes := TXMLAttributeList.Create(Self, Data, fStringAdapter);
  End;
End;

Procedure TXMLNode.ClearChildren;
Begin
  Nodes.Clear;
End;

Function TXMLNode.GetHasChildren: Boolean;
Begin
  Result := ( Nodes.Count <> 0 );
End;

Function TXMLNode.GetNode(Const ANodeName: String; Const AMode: TXMLNodeHandling; Const ASubIndex: Integer): TXMLNode;
Var
  lNodeName,
  lPath      : String;
  lSubIndex,
  lPosIdx    : Integer;
Begin
  lPosIdx := Pos('.', ANodeName);
  lPath   := '';

  If ( lPosIdx > 0 ) Then  // contains path
  Begin
    lNodeName := Copy(ANodeName, 1, lPosIdx-1);
    lPath     := Copy(ANodeName, lPosIdx+1, MaxInt);
    lSubIndex := -1;

    If GetSubIndexFromName(lNodeName, lSubIndex) Then
      lNodeName := StripSubIndexFromName(lNodeName);
  End
  Else
  Begin
    lNodeName := ANodeName;
    lSubIndex := ASubIndex;
  End;

    // Den SubIndex nicht auf den Pfad anwenden - nur auf den letzten Knoten!
  Result := Nodes.GetNode(lNodeName, lSubIndex);

  If ( Not Assigned(Result) ) Then
  Begin
    Case AMode Of
      nhCreateNode :
        Begin
          // strip sub index
          Result := TXMLNode.Create(Self);
          Result.Name := StripSubIndexFromName(lNodeName);
        End;

      nhRaiseException :
        Begin
          Raise EXMLNodeNotFound.Create(format('node "%s" not found!', [ANodeName]));
        End;

      nhReturnNil : ;
    End;
  End;

  If Assigned(Result) And ( Length(lPath) > 0 ) Then
    Result := Result.GetNode(lPath, AMode, ASubIndex);
End;

Procedure TXMLNode.SetAttributeList(AttributeList: TXMLAttributeList; Const AMode : TAttrSetMode);
Var
  i : Integer;
Begin
  Case AMode Of
    asmAddAttributes :
      Begin
        For i := 0 To AttributeList.Count - 1 Do
          fAttributes.Add(AttributeList.Attributes[i]);
      End;

    asmOverwriteAttributes :
      Begin
        If Assigned(fAttributes) Then
          fAttributes.Free;

        fAttributes := AttributeList;
        fAttributes.SetNode(Self);
      End;
  End;
End;

Function TXMLNode.GetChilds(Const AFilter: String; Const AFilterMode: TFilterMode): TXMLNodeList;
Begin
  Result := Nodes.GetFilteredList(AFilter, AFilterMode);
End;

Procedure TXMLNode.UpdateChildsEncoding(const AEncodingID: String);
Var
  i : Integer;
Begin
  For i := 0 To fNodes.Count - 1 Do
    fNodes[i].EncodingID := AEncodingID;
End;

Function TXMLNode.GetEncodingID: String;
Begin
  If Assigned(fStringAdapter) Then
    Result := fStringAdapter.EncodingID
  Else
    Result := 'UTF-8';
End;

Procedure TXMLNode.SetEncodingID(const Value: String);
Begin
  If Assigned(fStringAdapter) Then
  Begin
    fStringAdapter.EncodingID := Value;
  End
  Else
    fStringAdapter := TStringAdapter.Create(Value);

  UpdateChildsEncoding(Value);
End;

Function TXMLNode.GetAttributes: TXMLAttributeList;
Begin
{$IFNDEF ALLOWATTRIBUTEONROOT}
  If ( Not Assigned(fParent) ) And ( fLevel = 0 ) Then
    Raise EXMLInvalidOperation.Create('Root node can not have attributes')
  Else
{$ENDIF}
    Result := fAttributes;
End;

Function TXMLNode.AddChilds(AChilds: TXMLNodeList): Boolean;
Var
  i       : Integer;
  NewNode : TXMLNode;
Begin
  Result := True;

  For i := 0 To AChilds.Count - 1 Do
  Begin
    NewNode := TXMLNode.Create(Self, AChilds.Nodes[i]);
    fNodes.Add( NewNode );
  End;
End;

Procedure TXMLNode.Set_WriteHeader(const Value: Boolean);
Var
  i : Integer;
Begin
  If ( Value <> fWriteHeader ) Then
  Begin
    fWriteHeader := Value;

    If Assigned(fNodes) Then
      For i := 0 To fNodes.Count - 1 Do
        fNodes[i].WriteHeader := fWriteHeader;
  End;
End;

{ TXMLCommentList }

  // The default constructor to create the class
  //
  // Remarks
  //   This constructor should be used on normal operations, but not by the
  //   user. The comment list classes are created and maintained by the
  //   node they belong to.
Constructor TXMLCommentList.Create;
Begin
  Inherited Create;

  fComments := TObjectList.Create(True);
End;

  // The copy constructor creates a new comment list with all elements
  // of the passed comment list copied to new instance.
  //
  // Parameters
  //   AComments - The original comment list to copy from. The original
  //               comment list is not changed after copy in any kind.
  //
  // Remarks
  //   This constructor should be used to copy a commentlist, but not by the
  //   user. The comment list classes are created and maintained by the
  //   node they belong to.
Constructor TXMLCommentList.Create(AComments: TXMLCommentList);
Var
  i : Integer;
Begin
  Inherited Create;

  fComments := TObjectList.Create(True);
  For i := 0 To AComments.Count - 1 Do
    fComments.Add(AComments.Comments[i]);
End;

  // Use this function to add a new comment to the owning node. The comment
  // is passed as a TStrings object.
  //
  // Parameters
  //   Comment - The new comment to add.
  //
  // Returns
  //   The index of the added item in the comment list.
  //
  // Remarks
  //   Do never free the passed instance of Comment, because it is now
  //   maintained by the TXMLCommentList class. It would be freed on
  //   destruction of the class.
Function TXMLCommentList.Add(Comment: TStrings): Integer;
Begin
  Result := fComments.Add(Comment);
End;

  // Clears the comment list and deletes all items.
  //
  // Remarks
  //   After Clear all instances holded in the class are freed and no
  //   longer valid.
Procedure TXMLCommentList.Clear;
Begin
  fComments.Clear;
    // auch hier kmmert sich die TObjectList um die Freigabe der Objekte
End;

  // The function deletes the comment at the index of AIndex.
  // is passed as a TStrings object.
  //
  // Parameters
  //   AIndex - The index of the comment in the list to delete.
  //
  // Remarks
  //   After deletion, the instance to the TStrings object at the passed
  //   position will be freed automatically. If the application holds
  //   the deleted TStrings instances in any variable, this instance will
  //   invalid. Any access to this instances will result in a
  //   EAccessViolation.
Procedure TXMLCommentList.Delete(const AIndex: Integer);
Begin
  fComments.Delete(AIndex);
End;

  // The destructor for freeing the comment list.
  //
  // Remarks
  //   Never free the comment list - the list will be freed by the node
  //   the instance belongs to.
Destructor TXMLCommentList.Destroy;
Begin
  fComments.Free;

  Inherited Destroy;
End;

Function TXMLCommentList.GetComment(Index: Integer): TStrings;
Begin
  Result := fComments[Index] As TStrings;
End;

Function TXMLCommentList.GetCommentOnText(Index: String): TStrings;
Var
  Idx : Integer;
Begin
  Result := Nil;

  Idx := IndexOf(Index);
  If ( Idx > -1 ) Then
    Result := fComments[Idx] As TStrings;
End;

Function TXMLCommentList.GetCount: Integer;
Begin
  Result := fComments.Count;
End;

  // IndexOf() retrieves the index of a special comment in the comment
  // list.
  //
  // Parameters
  //   ACommentText - The comment to find.
  //
  // Returns
  //   The index of the comment in the comment list or -1 if the comment
  //   could not be found in the list.
Function TXMLCommentList.IndexOf(Const ACommentText: String; Const ACaseMode: TFilterMode): Integer;
Var
  i : Integer;
Begin
  Result := -1;

  If ( ACaseMode = fmCaseInsensitive ) Then
  Begin
    For i := 0 To Pred(fComments.Count) Do
      If ( AnsiSameText(TStrings(fComments[i]).Text, ACommentText) ) Then
      Begin
        Result := i;
        Break;
      End;
  End
  Else
  Begin
    For i := 0 To Pred(fComments.Count) Do
      If ( TStrings(fComments[i]).Text = ACommentText ) Then
      Begin
        Result := i;
        Break;
      End;
  End;
End;

  // IndexOf() retrieves the index of a special comment in the comment
  // list.
  //
  // Parameters
  //   Comment - The comment to find.
  //
  // Returns
  //   The index of the comment in the comment list or -1 if the comment
  //   could not be found in the list.
Function TXMLCommentList.IndexOf(Comment: TStrings): Integer;
Begin
  Result := fComments.IndexOf(Comment);
End;

Procedure TXMLCommentList.SetComment(Index: Integer; Const Value: TStrings);
Begin
  fComments[Index] := Value;
    // TList sendet ein Deleted Notify, was TObjectList durch das OwnsObject
    // dazu bewegt netterweise das alte Object zu lschen
End;

{ TXMLAttribute }

  // Constructor of the attribute. Never call this constructor directly to
  // create an attribute. Please use the add method of the TXMLAttributeList
  // instead.
  //
  // Parameters
  //   AttrList  - The attribute list the attribute belongs to
  //   Data      - This is the unsplitted string in the format of
  //               "name=value". The constructor will parse and split
  //               both values from the parameter.
  //
  // Exceptions
  //   If the given attribute name already exist in the attribute list, the
  //   constructor will raise a EXMLNodeAttrNameExists exception.
Constructor TXMLAttribute.Create(AttrList: TXMLAttributeList; Const Data: String);
Begin
  Inherited Create;

  fAttrList := AttrList;
  fValue    := TXMLValue.Create;
  SplitData(Data);
End;

  // Default constructor of the attribute. Never call this constructor directly to
  // create an attribute. Please use the add method of the TXMLAttributeList
  // instead.
  //
  // Parameters
  //   AttrList  - The attribute list the attribute belongs to
  //   AName     - The name of the attribute
  //   AValue    - The value of the attribute
  //
  // Exceptions
  //   If the given attribute name already exist in the attribute list, the
  //   constructor will raise a EXMLNodeAttrNameExists exception .
Constructor TXMLAttribute.Create(AttrList: TXMLAttributeList; const AName, AValue: String);
Begin
  Inherited Create;

  fAttrList := AttrList;
  SetName(AName);
  fValue    := TXMLValue.Create;
  SetValue(AValue);            
End;

Destructor TXMLAttribute.Destroy;
Begin
  fValue.Free;

  Inherited Destroy;
End;

Procedure TXMLAttribute.SetName(Const Value: String);
Begin
  If ( Not CheckName(Value) ) Then
    Raise EXMLNodeAttrNameInvalid.CreateFmt('TXMLAttribute.SetName(): Attribute name ("%s") is invalid! (Line: %d)', [Value, gLineNumber]);

  If ( Not fAttrList.Exists(Value) ) Then
  Begin
    fName := Value;
  End
  Else
    Raise EXMLNodeAttrNameExists.CreateFmt('TXMLAttribute.SetName(): Attribute with the name already exist! (Line: %d)', [gLineNumber]);
End;

Function TXMLAttribute.CheckName(Const AName: String): Boolean;
Begin
{$IFDEF DOVALIDITYCHECKS}

{$ENDIF}
  Result := True;
End;

Procedure TXMLAttribute.SetValue(Const Value: String);
Var
  lValue: String;
Begin
    // replace quotation marks and apostrophs
    //  ' -> &apos;
    //  " -> &quot;

  lValue := StringReplace(Value,  '&apos;', '''', [rfReplaceAll, rfIgnoreCase]);
  lValue := StringReplace(lValue, '&quot;', '"',  [rfReplaceAll, rfIgnoreCase]);

  fValue.AsString := lValue;
End;

Function TXMLAttribute.GetValue: String;
Var
  lResult: String;
Begin
    // replace quotation marks and apostrophs
    //  ' -> &apos;
    //  " -> &quot;

  lResult := StringReplace(fValue.AsString, '''', '&apos;', [rfReplaceAll, rfIgnoreCase]);
  lResult := StringReplace(lResult,         '"',  '&quot;', [rfReplaceAll, rfIgnoreCase]);

  Result := lResult;
End;

Procedure TXMLAttribute.SplitData(Const AData: String);
Var
  iPos   : Integer;
Begin
  fName  := '';

  If ( Length(AData) > 0 ) Then
  Begin
    iPos := Pos('=', AData);
    If ( iPos > 0 ) Then
    Begin
      SetName(Copy(AData, 1, iPos-1));
      SetValue(Copy(AData, iPos+1, MaxInt));
    End
    Else
      SetName(AData);
  End;
End;

{ TXMLAttributesList }

  // This is the default constructor for a TXMLAttributeList.
  //
  // Parameters
  //   ANode - The node the managed attributes belonging to.
  //           This parameter has a default value of nil if no node is passed.
Constructor TXMLAttributeList.Create(ANode: TXMLNode);
Begin
  Inherited Create;

  fNode     := ANode;
  fAttrList := TObjectList.Create;
End;

  // This constructor creates the TXMLAttributeList instances with a list
  // of attributes. The attributes will be passed as a string.
  //
  // Parameters
  //   ANode          - The node the managed attributes belonging to.
  //   AttrString     - A string with all attributes in list. The string should
  //                    have the same structure as the attributes in the xml file.
  //                    For example: AttrName1=\"value1\" AttrName2=\"value2\"
  //                    The string will be parsed and decoded with the passed
  //                    StringAdapter and each attribute will be created with
  //                    it's belonging value.
  //   AStringAdapter - The StringAdapter to use for decoding the attribute
  //                    values.
Constructor TXMLAttributeList.Create(ANode: TXMLNode; Const AttrString: String; AStringAdapter : TStringAdapter);
Begin
  Inherited Create;
  
  fNode := ANode;
  fAttrList := TObjectList.Create;

  SetAttributesAsString(AttrString, AStringAdapter);
End;

  // This function creates a new attribute and adds it to the list. The
  // attribute will have no name and no value - this has to be set by the
  // user after a call to Add.
  //
  // Returns
  //   The created attribute.
  //
  // Remarks
  //   The user has to set the name and the value after a call to Add. The
  //   returned is already in the list and there is <b>no need</b> to add it
  //   a second time with the other Add function. 
Function TXMLAttributeList.Add: TXMLAttribute;
Begin
  Result := TXMLAttribute.Create(Self);
  fAttrList.Add(Result);
End;

  // This function adds a new attribute to the list. The passed attribute will
  // be <b>copied</b> and not directly added to the list.
  //
  // Parameters
  //   XMLAttribute - The attribute to copy.
  //
  // Returns
  //   The created and added attribute.
Function TXMLAttributeList.Add(XMLAttribute: TXMLAttribute): TXMLAttribute;
Begin
  Result := TXMLAttribute.Create(Self, XMLAttribute.Name, XMLAttribute.Value.AsString);
  fAttrList.Add(Result);
End;

  // The function deletes the given attribute from the list.
  //
  // Parameters
  //   AName   - The name of the attribute to delete.
  //
  // Remarks
  //   The attribute instance will be freed on delete.
Procedure TXMLAttributeList.Delete(const AName: String);
Var
  Idx : Integer;
Begin
  Idx := IndexOf(AName);
  If ( Idx > -1 ) Then
    Delete(Idx);
End;

  // The function deletes the given attribute from the list.
  //
  // Parameters
  //   AIndex  - The index of the attribute in the list to delete.
  //
  // Remarks
  //   The attribute instance will be freed on delete.
Procedure TXMLAttributeList.Delete(const AIndex: Integer);
Begin
  fAttrList.Delete(AIndex);
End;

Function TXMLAttributeList.GetAttributeByIndex(Index: Integer): TXMLAttribute;
Begin
  If ( ( Index < 0 ) Or ( Index >= fAttrList.Count ) ) Then
    Raise EListError.Create('TXMLAttributesList.GetAttributeByIndex(): Index out of range');

  Result := ( fAttrList[Index] As TXMLAttribute );
End;

Function TXMLAttributeList.GetAttributeByName(Index: String): TXMLAttribute;
Begin
{$IFNDEF CREATEUNKOWNXMLATTR}
  Result := Nil;
{$ENDIF}

  If ( Exists(Index) ) Then
    Result := GetAttributeByIndex(IndexOf(Index))
{$IFNDEF CREATEUNKOWNXMLATTR}
    ;
{$ELSE}
  Else
  Begin
    Result := Add;
    Result.Name := Index;
  End;
{$ENDIF}
End;

Function TXMLAttributeList.GetCount: Integer;
Begin
  Result := fAttrList.Count;
End;

  // Use the IndexOf() function to retrieve the index of a special
  // attribute in the list.
  //
  // Parameters
  //   AName - The name of the attribute to find.
  //
  // Returns
  //   The function returns the index of the attribute or -1 if it could not
  //   be found in the list.
Function TXMLAttributeList.IndexOf(const AName: String): Integer;
Var
  i : Integer;
Begin
  Result := -1;

  For i := 0 To Pred(fAttrList.Count) Do
{$IFDEF NAMESCASEINSENSITIVE}
    If ( AnsiSameText(GetAttributeByIndex(i).Name, AName) ) Then
{$ELSE}
    If ( GetAttributeByIndex(i).Name = AName ) Then
{$ENDIF}
    Begin
      Result := i;
      Break;
    End;
End;

  // The destructor of the attributes list. 
Destructor TXMLAttributeList.Destroy;
Begin
  fAttrList.Free;

  Inherited Destroy;
End;

  // Use Exists() to check if a special attribute exists in the list.
  //
  // Parameters
  //   AName  - The name of the attribute to check.
  //
  // Returns
  //   The function returns true if the attribute exists or false otherwise.
Function TXMLAttributeList.Exists(Const AName: String): Boolean;
Begin
  Result := ( IndexOf(AName) > -1 );
End;

Function TXMLAttributeList.Exists(Const AAttr: TXMLAttribute): Boolean;
Begin
  Result := ( fAttrList.IndexOf(AAttr) > -1 );  
End;

Procedure TXMLAttributeList.SetNode(const Value: TXMLNode);
Begin
  If ( Assigned(fNode) ) Then
    fNode.SetAttributeList(Nil, asmOverwriteAttributes);

  fNode := Value;
  If ( Assigned(fNode) ) Then
    fNode.SetAttributeList(Self, asmOverwriteAttributes);
End;

  // Use clear to clear the attribute list and delete all entries.
Procedure TXMLAttributeList.Clear;
Begin
  fAttrList.Clear;
End;

  // This function returns all attribute in a single line in the format of
  // name=\"value\" seperated by a space character. The value is encoded with
  // the passed StringAdapter. The function is used to save all attributes
  // in the list to a node in the xml.
  //
  // Parameters
  //   AStringAdapter - The StringAdapter to use for encoding the attribute
  //                    values.
  //
  // Returns
  //   The generated string as described in the description.
  //
  // See Also
  //   SetAttributesAsString()
Function TXMLAttributeList.GetAttributesAsString(AStringAdapter : TStringAdapter): String;
Var
  i       : Integer;
  lAttr   : TXMLAttribute;
  lResult : String;
Begin
  lResult := '';

  For i := 0 To Pred(fAttrList.Count) Do
  Begin
    lAttr := GetAttributeByIndex(i);

    lResult := lResult + Format('%s="%s" ', [lAttr.Name, AStringAdapter.StringEncode(lAttr.GetValue)]);
  End;

  Result := Trim(lResult);
End;

  // This function parses a passed string and creates attributes with its
  // values and adds them to the list. Any exisiting attibutes will be left
  // in the list. The format of the passed string has to be in the same format
  // as the attributes in the xml file. It can consist of more than one entry
  // in the format name=\"value\" seperated by a space character. The values
  // of the attributes are decoded with the passed StringAdapter.
  //
  // Parameters
  //   AttributesString  - The attribute string to parse.
  //   AStringAdapter    - The StringAdapter to use for encoding the attribute
  //                       values.
  //
  // See Also
  //   Create(), GetAttributesAsString()
Procedure TXMLAttributeList.SetAttributesAsString(Const AttributesString: WideString; AStringAdapter : TStringAdapter);
Var
  lAttrName,
  lAttrValue,
  lData      : WideString;
  lAttr      : TXMLAttribute;
Begin
  lData := AttributesString;

    // ok, here we have to split all the attributes. we should have
    // now a string in Data with the following format:
    //   attrname="content" attrname="string content" attrname="another string content"
    // so we can't search for a space character - we have to do it
    // in steps.
  While ( Length(lData) > 0 ) Do
  Begin
      // 1st: get the name of the attribute and remove it
    lAttrName := Copy(lData, 1, Pos('=', lData)-1);
    System.Delete(lData, 1, Pos('=', lData));
    lData := Trim(lData);

      // all attributes have to begin with " and end with " - it is required!
    If ( Copy(lData, 1, 1) = '"' ) Then // is it a string?
    Begin
      System.Delete(lData, 1, 1);
      lAttrValue := AStringAdapter.StringDecode(Copy(lData, 1, Pos('"', lData)-1));
      System.Delete(lData, 1, Pos('"', lData));
    End
    Else
    Begin
      lAttrValue := '';
      //parsing error: missing " !!
    End;


    lAttr := Add;
    lAttr.Name := Trim(lAttrName);
    lAttr.SetValue(Trim(lAttrValue));   // removing &quot; and &apos;
    
    lData := Trim(lData);
  End;
End;

  // Use AsStrings to get a list of all attribute names currently maintained
  // by the list.
  //
  // Returns
  //   The function returns a TStrings object with all attribute names. The
  //   object member is <b>not</b> filled with the attribute instance itself.
  //
  // Remarks
  //   The caller has to care about freeing the returned TStrings derivate.
Function TXMLAttributeList.AsStrings: TStrings;
Var
  i : Integer;
Begin
  Result := TStringList.Create;

  For i := 0 To Pred(fAttrList.Count) Do
    Result.Add(GetAttributeByIndex(i).Name);
End;

  // The AsValueList() functions returns a list of all attributes currently
  // maintained by the list, including the values. The returned list
  // can be used to access the values with the names[] and values[] property
  // of a TStrings object.
  //
  // Returns
  //   The list with all attribute names and values in the format of
  //   \"name\=value\" for each entry.
  //
  // Remarks
  //   The caller has to care about freeing the returned TStrings derivate.
  //   Any write access to set a new value or add a new attribute to the list
  //   will be not recognized by the attribute list and will not be applied.
Function TXMLAttributeList.AsValueList: TStrings;
Var
  i     : Integer;
  lAttr : TXMLAttribute;
Begin
  Result := TStringList.Create;

  For i := 0 To Pred(fAttrList.Count) Do
  Begin
    lAttr := GetAttributeByIndex(i);
    Result.Add(lAttr.Name + '=' + lAttr.GetValue);
  End;
End;

{ TXMLValue }

  // The constructor of TXMLValue with the possibility to define the value.
  //
  // Parameters
  //   AValue     - The value to represent.
  //   AUseCDATA  - Use CDATA Tag for saving the value.
Constructor TXMLValue.Create(Const AValue: String; Const AUseCDATA : Boolean);
Begin
  Inherited Create;

  fValue    := AValue;
  fUseCDATA := AUseCDATA;
End;

  // The default constructor of TXMLValue. It will be initialized with a empty
  // value.
Constructor TXMLValue.Create;
Begin
  Inherited Create;

  fValue    := '';
  fUseCDATA := False;
End;

  // The destructor to free the instance.
Destructor TXMLValue.Destroy;
Begin
    // nothing to do...

  Inherited Destroy;
End;

  // The function tries to convert the value to a integer value or returns a
  // default value on convert failure.
  //
  // Parameters
  //   ADefault - The default value to assign on failure of the conversion.
  //
  // Returns
  //   The converted value or the default value.
  //
  // See also
  //   AsInteger
Function TXMLValue.AsIntegerDefault(Const ADefault: Integer): Integer;
Begin
  Result := StrToIntDef(fValue, ADefault);
End;

Function TXMLValue.GetValueAsBoolean: Boolean;
Begin
  Result := ( ( AnsiSameText(fValue, 'true') ) Or ( StrToIntDef(fValue, 0) <> 0 ) );
End;

Function TXMLValue.GetValueAsDouble: Double;
Var
  lValue: String;
Begin
  Try
    lValue := StringReplace(fValue, '.', DecimalSeparator, [rfIgnoreCase, rfReplaceAll]);
    lValue := StringReplace(lValue, ',', DecimalSeparator, [rfIgnoreCase, rfReplaceAll]);
    Result := StrToFloat(lValue);
  Except
    Result := 0.0;
  End;
End;

Function TXMLValue.GetValueAsInteger: Integer;
Begin
  Result := StrToIntDef(fValue, 0);
End;

Function TXMLValue.GetValueAsString: String;
Begin
  Result := fValue;
End;

  // The function IsNull() returns the status of the value.
  //
  // Returns
  //   Returns True if no value is assigned or False otherwise.
Function TXMLValue.IsNull: Boolean;
Begin
  Result := ( fValue = '' );
End;

Procedure TXMLValue.SetValueAsBoolean(const Value: Boolean);
Begin
  If ( Value ) Then
    fValue := '1'
  Else
    fValue := '0';
End;

Procedure TXMLValue.SetValueAsDouble(const Value: Double);
Begin
  fValue := FloatToStr(Value);
End;

Procedure TXMLValue.SetValueAsInteger(const Value: Integer);
Begin
  fValue := IntToStr(Value);
End;

Procedure TXMLValue.SetValueAsString(const Value: String);
Begin
  fValue := Value;
End;

Function TXMLValue.GetUseCDATA: Boolean;
Begin
  Result := fUseCDATA Or CheckCDATANeed;
End;

Procedure TXMLValue.SetUseCDATA(Const Value: Boolean);
Begin
  If ( fUseCDATA <> Value ) Then
    fUseCDATA := Value;
End;

  // The function CheckCDATANeed() detects the need of using the CDATA surrounder
  // for the value on saving.
  //
  // Returns
  //   Returns True if CDATA needs to be used.
Function TXMLValue.CheckCDATANeed: Boolean;
Begin
  Result := ( Pos('<', fValue) > 0 ) Or
            ( Pos('>', fValue) > 0 );
End;

  // The function Assign() copies all data and settings of another TXMLValue
  // to the actual instance.
Procedure TXMLValue.Assign(AValue: TXMLValue);
Begin
  fValue     := AValue.fValue;
  fUseCDATA  := AValue.fUseCDATA;
End;

End.

//  Log List
//
// $Log: XMLLib.pas,v $
// Revision 1.29  2005-11-16 23:46:50+01  muetze1
// bug fix for sub-indexing in path and with CreatePathAndNode()
//
// Revision 1.28  2005-07-02 21:57:15+02  muetze1
// - removed setting Attribute : TXMLAttributeList on TXMLNode (memory leak)
// - removed some memory leaks
//
// Revision 1.27  2005-06-20 00:23:53+02  muetze1
// - changes for W3C (one space before /> )
// - added delete function for TXMLNodeList to accept an instance
//
// Revision 1.26  2005-06-06 00:13:55+02  muetze1
// support for CR/LF/HT on UTF-8 node data added
//
// Revision 1.25  2005-05-09 01:02:38+02  muetze1
// <>
//
// Revision 1.24  2005-02-25 19:47:14+01  muetze1
// added support for relative node paths
// added support for node sub index in node paths (ala root.node[2].3rdnode[3].name)
//
// Revision 1.23  2004-11-06 16:24:33+01  muetze1
// some documentation
// added CDATA support
//
// Revision 1.22  2004-10-07 23:08:12+02  muetze1
// added CDATA reading
//
// Revision 1.21  2004-09-26 18:23:15+02  muetze1
// reading of <![CDATA[ ... ]]> content is now implemented
// DONE: writing and setting these type of data...
//
// Revision 1.20  2003-10-19 19:25:27+02  muetze1
// removed grammar mistake
//
// Revision 1.19  2003-10-13 20:02:24+02  muetze1
// added comments,
// small changes
//
// Revision 1.18  2003-09-22 01:59:34+02  muetze1
// added TXMLValue
//
// Revision 1.17  2003-09-01 23:58:32+02  administrator
// TXMLNode Copy Constructor basicly tested - without attribute list nor comment list
//
// Revision 1.16  2003-09-01 19:33:31+02  administrator
// added copy constructors (untested)
// added WriteHeader Property
//
// Revision 1.15  2003-08-10 13:18:23+02  muetze1
// added UTF-8 thanks to the creator
//
// Revision 1.14  2003-08-10 00:38:33+02  muetze1
// changing encoding supported
//
// Revision 1.13  2003-08-09 23:16:48+02  muetze1
// added define for attribute access handling on non existing keys...
//
// Revision 1.12  2003-08-03 04:08:11+02  muetze1
// added a filter for getting nodes from a node list (also on nodes for getting sub-nodes)
// removed old "big" read/write procedures from the library
// make the XMLNode creation compatible to old standard (using TStringsAdapter from Parent)
// added function to get names lists as TStrings
//
// Revision 1.11  2003-07-29 00:59:36+02  muetze1
// prepared for derivations
// added attributes list + attribute class
// added strings adapter for decoding / encoding different charsets
//
// Revision 1.10  2003-06-01 00:40:06+02  muetze1
// changed rcs id to static string
//
// Revision 1.9  2003-05-06 23:25:36+02  administrator
// added SubNodeCount() and GetNode() in NodeList
// GetNodeFromPath() will now 100%ly return nil if path was not found
// IndexOf() will now handle SubItemIndex correctly (0, 1, 2, ... Count - 1)
//
// Revision 1.8  2003-05-05 21:37:27+02  administrator
// added Clear() to NodesList and CommentsList
// added HasChildren Property to Node
//
// Revision 1.7  2003-05-05 01:11:01+02  administrator
// added comment list class
// support for comments - they won't be lost from now
// Unit ID is now better readable w/o the $
// the root node can't get a value now
//
// Revision 1.6  2003-05-03 23:53:58+02  administrator
// rootnode now writeable
// added clear function
// updated comment
// moved updatepath function into private
//
// Revision 1.5  2003-05-03 18:12:07+02  administrator
// removed bug on writing in none exisiting keys
// added more read and write key functions for other data types
//
// Revision 1.4  2003-05-02 00:42:20+02  administrator
// removed bug: access violation on not existing keys
//
// Revision 1.3  2003-05-02 00:24:40+02  administrator
// added support for attributes
// added support for "one line" entries
//
// Revision 1.2  2003-04-28 01:59:29+02  administrator
// removed warnings
//
// Revision 1.1  2003-04-28 00:41:37+02  administrator
// removed old log
//
// Revision 1.0  2003-04-28 00:36:31+02  administrator
// Initial revision
//
//
//



